package com.twistpair.wave.thinclient.kexcrypto;

import junit.framework.Assert;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcKexCryptoServerSimulator extends WtcKexCryptoBase
{
    public static final String TAG             = WtcLog.TAG(WtcKexCryptoServerSimulator.class);

    public static final int    BYTES_PER_BLOCK = 16;

    private//
    final//
    short                      numKeys;
    private//
    final//
    short                      numBytesPerKey;

    public WtcKexCryptoServerSimulator(short numKeys)
    {
        setMessageCipherMode(WtcMessageCipherMode.AES256CTR);
        this.numKeys = numKeys;
        this.numBytesPerKey = 32;
    }

    protected static byte[] generateIV()
    {
        return WtcCryptoUtilPlatform.getRandomBytes(BYTES_PER_BLOCK);
    }

    protected static byte[] generateKey(int numBytesPerKey)
    {
        return WtcCryptoUtilPlatform.getRandomBytes(numBytesPerKey);
    }

    protected void generateKeys(int numKeys, int numBytesPerKey, byte[][] keysClientToServer, byte[][] keysServerToClient)
    {
        byte[] key;
        for (int i = 0; i < numKeys; i++)
        {
            key = generateKey(numBytesPerKey);
            keysClientToServer[i] = key;

            key = generateKey(numBytesPerKey);
            keysServerToClient[i] = key;
        }
    }

    public IWtcMemoryStream simulateServerResponse(IWtcMemoryStream inputStream) throws WtcKexCryptoException
    {
        WtcLog.info(TAG, "+simulateServerResponse");
        try
        {
            //WtcLog.info(TAG, "messageRequest(" + count + ")=" + bytesToHexString(messageRequest, offset, count));

            reset();

            int readStart = inputStream.getPosition();

            byte[] header = new byte[HEADER.length];
            inputStream.read(header, 0, header.length);
            WtcLog.info(TAG, "header=" + WtcString.toHexString(header));
            int cookie = inputStream.readInt32();
            WtcLog.info(TAG, "cookie=" + cookie);
            String keyGroupId = readString(inputStream);
            WtcLog.info(TAG, "keyGroupId=" + keyGroupId);
            String capabilities = readString(inputStream);
            WtcLog.info(TAG, "capabilities=" + capabilities);
            byte[] G = readByteArray(inputStream);
            WtcLog.info(TAG, "G(" + G.length + ")=" + WtcString.toHexString(G));
            byte[] P = readByteArray(inputStream);
            WtcLog.info(TAG, "P(" + P.length + ")=" + WtcString.toHexString(P));
            byte[] E = readByteArray(inputStream);
            WtcLog.info(TAG, "E(" + E.length + ")=" + WtcString.toHexString(E));
            byte[] MAC1 = readByteArray(inputStream);
            WtcLog.info(TAG, "MAC1(" + MAC1.length + ")=" + WtcString.toHexString(MAC1));

            int readStop = inputStream.getPosition();

            byte[] hmacKey = findKeyAndThrowExceptionIfNotValidInput(header, keyGroupId, //
                            inputStream, readStart, readStop, //
                            MAC1);
            WtcLog.info(TAG, "hmacKey(" + hmacKey.length + ")=" + WtcString.toHexString(hmacKey));

            WtcLog.info(TAG, "numKeys=" + numKeys);
            WtcLog.info(TAG, "numBytesPerKey=" + numBytesPerKey);

            WtcDhKeyPairPlatform dhkp = new WtcDhKeyPairPlatform(P, G);
            byte[] F = dhkp.getDHPublicKeyBytes();
            WtcLog.info(TAG, "F(" + F.length + ")=" + WtcString.toHexString(F));

            byte[] sharedSecret = dhkp.calculateAgreement(E);
            WtcLog.info(TAG, "sharedSecret(" + sharedSecret.length + ")=" + WtcString.toHexString(sharedSecret));

            masterKey = WtcCryptoUtilPlatform.SHA256(sharedSecret);
            WtcLog.info(TAG, "masterKey(" + masterKey.length + ")=" + WtcString.toHexString(masterKey));

            // TODO:(pv) Split capabilities by comma and determine the best capability.
            // For now the only supported mode is hard coded here...
            setMessageCipherMode(WtcMessageCipherMode.AES256CTR);
            String messageCipherMode = cipherModeToString(getMessageCipherMode());
            WtcLog.info(TAG, "messageCipherMode=" + messageCipherMode);

            byte[] ivClientToServer = generateIV();
            byte[] ivServerToClient = generateIV();
            WtcLog.info(TAG, "ivClientToServer(" + ivClientToServer.length + ")=" + WtcString.toHexString(ivClientToServer));
            WtcLog.info(TAG, "ivServerToClient(" + ivServerToClient.length + ")=" + WtcString.toHexString(ivServerToClient));

            Assert.assertEquals("ivClientToServer.length != ivServerToClient.length", ivClientToServer.length,
                            ivServerToClient.length);
            this.ivLocalToRemote = ivServerToClient;
            this.ivRemoteToLocal = ivClientToServer;

            byte[][] keysClientToServer = new byte[numKeys][];
            byte[][] keysServerToClient = new byte[numKeys][];

            WtcLog.info(TAG, "Keys generating...");
            generateKeys(numKeys, numBytesPerKey, keysClientToServer, keysServerToClient);
            WtcLog.info(TAG, "Keys generated!");

            WtcLog.info(TAG, "PayloadTransforms initializing (this can take awhile)...");
            initializePayloadTransforms(ivServerToClient, keysServerToClient, ivClientToServer, keysClientToServer,
                            WtcCipherSide.Server);
            WtcLog.info(TAG, "PayloadTransforms initialized!");

            byte[] keysEncrypted = encryptKeys(masterKey, keysClientToServer, keysServerToClient);
            WtcLog.info(TAG, "keysEncrypted(" + keysEncrypted.length + ")=" + WtcString.toHexString(keysEncrypted));

            IWtcMemoryStream outputStream = new WtcMemoryStream();
            outputStream.writeInt16(numBytesPerKey);
            outputStream.writeInt16(numKeys);
            writeByteArray(outputStream, F);
            writeString(outputStream, messageCipherMode);
            writeByteArray(outputStream, ivClientToServer);
            writeByteArray(outputStream, ivServerToClient);
            writeByteArray(outputStream, keysEncrypted);

            byte[] MAC2 = WtcCryptoUtilPlatform.HMACSHA256(hmacKey, outputStream.getBuffer(), 0, outputStream.getLength());
            WtcLog.info(TAG, "MAC2(" + MAC2.length + ")=" + WtcString.toHexString(MAC2));
            writeByteArray(outputStream, MAC2);

            outputStream.setPosition(0);

            return outputStream;

        }
        catch (WtcKexCryptoException e)
        {
            WtcLog.error(TAG, "EXCEPTION: processKexResponse", e);
            reset();
            throw e;
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, "EXCEPTION: simulateServerResponse", e);
            reset();
            throw new WtcKexCryptoException("simulateServerResponse", e);
        }
        finally
        {
            WtcLog.info(TAG, "-simulateServerResponse");
        }
    }
}
