package com.twistpair.wave.thinclient.kexcrypto;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Hashtable;

import junit.framework.Assert;

import com.twistpair.wave.thinclient.WtcException;
import com.twistpair.wave.thinclient.kexcrypto.WtcCryptoUtilPlatform.WtcDecryptorAes256Ecb;
import com.twistpair.wave.thinclient.kexcrypto.WtcCryptoUtilPlatform.WtcEncryptorAes256Ecb;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpMessage;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcArraysPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

public abstract class WtcKexCryptoBase
{
    private static final String TAG         = WtcLog.TAG(WtcKexCryptoBase.class);

    public static final boolean VERBOSE_LOG = false;

    public static final String  DEFAULT_KEY = "TPS recommends that you use a private MAC key";

    public static class WtcKexCryptoException extends WtcException
    {
        public WtcKexCryptoException(String source, String message)
        {
            super(source, message);
        }

        public WtcKexCryptoException(String source, Exception innerException)
        {
            super(source, innerException);
        }

        public WtcKexCryptoException(String source, String message, Exception innerException)
        {
            super(source, message, innerException);
        }
    }

    public interface IWtcTransformer
    {
        /**
         * Slower "copy" transform that should be used once used only during key negotiation.
         * @param inBytes
         * @param offset
         * @param length
         * @return the transformed output byte[]
         * @throws WtcKexCryptoException
         */
        byte[] transform(byte[] inBytes, int offset, int length) throws WtcKexCryptoException;

        /**
         * Faster "in-place" transform used only on fully aligned/padded blocks.
         * This should be used when transforming very frequent payloads after key negotiation.
         * @param inBytes
         * @param inOffset
         * @param outBytes
         * @param outOffset
         * @throws WtcKexCryptoException
         */
        void transform(byte[] inBytes, int inOffset, byte[] outBytes, int outOffset) throws WtcKexCryptoException;
    }

    public interface IWtcPayloadTransformer
    {
        int getBlockLength();

        /**
         * @param transformers
         * @param iv Expects non-null byte[] of length getBlockLength()
         * @throws WtcKexCryptoException
         */
        void initialize(IWtcTransformer[] transformers, byte[] iv) throws WtcKexCryptoException;

        /**
         * @param extendedSequenceNumber
         * @param messageBuffer
         * @param payloadOffset
         * @param payloadLength
         * @param workingBlockBuffer Expects non-null byte[] the same length as the initialization iv
         * @throws WtcKexCryptoException
         */
        void transformPayload(long extendedSequenceNumber, byte[] messageBuffer, int payloadOffset, int payloadLength, //
                        byte[] workingBlockBuffer) throws WtcKexCryptoException;
    }

    public interface IWtcDhKeyPair
    {
        public byte[] getDHPublicKeyBytes() throws WtcKexCryptoException;

        /**
         * Compute the combined shared secret (K) of a given local private and remote public key
         * 
         * @param remotePublicKey
         * @return the byte[] of the calculated shared secret
         * @throws WtcKexCryptoException
         */
        public byte[] calculateAgreement(byte[] remotePublicKey) throws WtcKexCryptoException;
    }

    /**
     * <p>Usually ECB ciphers are considered very insecure:<br>
     *  <a href="http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Electronic_codebook_.28ECB.29">http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Electronic_codebook_.28ECB.29</a></p>
     * <p>Our use is actually still very secure.<br>
     * We use ECB in two places:<br>
     * <ol>
     * <li>Raw ECB mode to cipher keys and IVs in Key Exchange messages.<br>
     *      The only ciphered part of a Key Exchange message are the client and server cipher keys and IVs.<br>
     *      The keys and IVs are crypto white noise (ie: no pattern) and are always the size of an cipher block.<br>
     *      ECB cipher is bad if the content extends in to the next cipher block (ie: bleed over).<br>
     *          ex: block1="hel", block2="lo"<br>
     *      ECB cipher of random content that is always the size of a cipher block is safe (ie: no bleed over).<br>
     *      A hacker cannot discern a pattern from something that doesn't have a pattern.<br>
     *      If a hacker were able to decrypt one block then it would not help them to decrypt any others.
     * </li>
     * <li>Counter (CTR) mode to cipher Control and Media messages.<br>
     *      CTR mode uses ECB to encrypt a white noise initialized counter that is always the size of a cipher block.<br> 
     *      <a href="http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Counter_.28CTR.29">http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Counter_.28CTR.29</a><br>
     *      ECB encryption of the counter is safe for the same reasons our other use of ECB is safe.<br>
     *      ie: it is initialized to white noise and is always the size of a cipher block.
     * </li>
     * </ol>
     */
    public static abstract class WtcTransformerAes256EcbBase implements IWtcTransformer
    {
        public static final int BLOCK_LENGTH = 16;
    }

    public static Hashtable GroupMacKeys = new Hashtable();

    public static byte[] findKey(String keyGroupId)
    {
        if (WtcString.isNullOrEmpty(keyGroupId))
        {
            return DEFAULT_KEY.getBytes();
        }
        return (byte[]) GroupMacKeys.get(keyGroupId);
    }

    public interface WtcMessageCipherMode
    {
        public static final int      AES256CTR = 0;
        public static final int      Unknown   = 1;
        public static final String[] names     =
                                               {
                                                   "AES256-CTR", "Unknown"
                                               };
    }

    protected interface WtcCipherSide
    {
        public static final int      Client = 0;
        public static final int      Server = 1;
        public static final String[] names  =
                                            {
                                                "Client", "Server"
                                            };
    }

    protected static final byte   VERSION      = 1;
    protected static final byte[] HEADER       = new byte[]
                                               {
        (byte) 'k', (byte) 'e', (byte) 'X', VERSION
                                               };
    public static final int       BLOCK_LENGTH = WtcTransformerAes256EcbBase.BLOCK_LENGTH;
    protected static final byte[] IV_ZERO      = new byte[]
                                               {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
                                               };

    private boolean               isInitialized;

    public boolean getIsInitialized()
    {
        return isInitialized;
    }

    /**
     * SHA256 of DiffieHellman BigInteger sharedSecret
     */
    protected byte[]               masterKey;

    protected byte[]               ivLocalToRemote;
    protected byte[]               ivRemoteToLocal;

    private IWtcPayloadTransformer payloadDecryptor;
    private IWtcPayloadTransformer payloadEncryptor;

    /*
    // TODO:(pv) For JUnit test verification purposes
    public boolean equals(WtcKexCryptoBase remote)
    {
        return ...;
    }
    */

    int                            messageCipherMode;

    protected int getMessageCipherMode()
    {
        return messageCipherMode;
    }

    /**
     * @param messageCipherMode
     * @throws IllegalArgumentException
     */
    protected void setMessageCipherMode(int messageCipherMode)
    {
        switch (messageCipherMode)
        {
            case WtcMessageCipherMode.AES256CTR:
                payloadDecryptor = new WtcPayloadTransformerAes256Counter();
                payloadEncryptor = new WtcPayloadTransformerAes256Counter();
                break;
            default:
                throw new IllegalArgumentException("Unsupported WtcMessageCipherMode=" + messageCipherMode);
        }
        this.messageCipherMode = messageCipherMode;
    }

    protected WtcKexCryptoBase()
    {
        reset();
        messageCipherMode = WtcMessageCipherMode.Unknown;
    }

    /**
     * Call this function to forget keys, reset state and start over.
     */
    protected void reset()
    {
        isInitialized = false;
        masterKey = null;
        ivLocalToRemote = null;
        ivRemoteToLocal = null;
        payloadDecryptor = null;
        payloadEncryptor = null;
    }

    // #region - Static utility methods

    protected static String cipherModeToString(int messageCipherMode)
    {
        return WtcMessageCipherMode.names[messageCipherMode];
    }

    protected static int parseCipherMode(String messageCipherMode)
    {
        String[] names = WtcMessageCipherMode.names;
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(messageCipherMode))
                return i;
        }
        throw new IllegalArgumentException("Could not parse a value for messageCipherMode=\"" + messageCipherMode + "\"");
    }

    protected static byte[] readByteArray(IWtcMemoryStream inputStream) //
                    throws IOException
    {
        short payloadLength = inputStream.readInt16();
        byte[] value = new byte[payloadLength];
        inputStream.read(value, 0, payloadLength);
        return value;
    }

    protected static void writeByteArray(IWtcMemoryStream outputStream, byte[] value) //
                    throws IOException
    {
        int length = (value == null) ? 0 : value.length;
        outputStream.writeInt16((short) length);
        if (length > 0)
        {
            outputStream.write(value, 0, length);
        }
    }

    protected static String readString(IWtcMemoryStream inputStream) //
                    throws IOException
    {
        byte[] value = readByteArray(inputStream);
        return (value == null || value.length == 0) ? "" : WtcString.getString(value, 0, value.length);
    }

    protected static void writeString(IWtcMemoryStream outputStream, String value) //
                    throws IOException
    {
        byte[] bytes = WtcString.isNullOrEmpty(value) ? null : value.getBytes();
        writeByteArray(outputStream, bytes);
    }

    // #endregion - Static utility methods

    protected byte[] throwExceptionIfNotValidInput(IWtcMemoryStream inputStream, int readStart, int readStop, //
                    byte[] key, byte[] macSpecified) //
                    throws WtcKexCryptoException
    {
        int inputLength = inputStream.getLength();
        if (readStop != inputLength)
        {
            throw new IllegalArgumentException((inputLength - readStop) + " unread bytes in inputStream");
        }

        // Compute and check MAC
        int messageNoHashLength = (readStop - readStart) - (2 + macSpecified.length);
        byte[] macComputed = WtcCryptoUtilPlatform.HMACSHA256(key, inputStream.getBuffer(), readStart, messageNoHashLength);
        if (VERBOSE_LOG)
        {
            WtcLog.info(TAG, "macComputed=" + WtcString.toHexString(macComputed));
        }

        if (!WtcArraysPlatform.equals(macComputed, macSpecified))
        {
            throw new IllegalArgumentException("Specified MAC does not equal computed/expected MAC");
        }

        return key;
    }

    protected byte[] findKeyAndThrowExceptionIfNotValidInput(byte[] header, String keyGroupId, //
                    IWtcMemoryStream inputStream, int readStart, int readStop, //
                    byte[] macSpecified) // 
                    throws WtcKexCryptoException
    {
        if (!WtcArraysPlatform.equals(header, HEADER))
        {
            throw new IllegalArgumentException("KEX header invalid");
        }
        byte[] key = findKey(keyGroupId);
        return throwExceptionIfNotValidInput(inputStream, readStart, readStop, key, macSpecified);
    }

    protected byte[] encryptKeys(byte[] masterKey, byte[][] keysClientToServer, byte[][] keysServerToClient)
                    throws WtcKexCryptoException
    {
        Assert.assertEquals("keysClientToServer.length != keysServerToClient.length", keysClientToServer.length,
                        keysServerToClient.length);

        ByteArrayOutputStream streamKeys = new ByteArrayOutputStream();

        byte[] key;
        int numKeys;

        numKeys = keysClientToServer.length;
        for (int i = 0; i < numKeys; i++)
        {
            key = keysClientToServer[i];
            if (VERBOSE_LOG)
            {
                WtcLog.info(TAG, "keyClientToServer #" + i + "(" + key.length + "): " + WtcString.toHexString(key));
            }
            streamKeys.write(key, 0, key.length);
        }

        numKeys = keysServerToClient.length;
        for (int i = 0; i < numKeys; i++)
        {
            key = keysServerToClient[i];
            if (VERBOSE_LOG)
            {
                WtcLog.info(TAG, "keyServerToClient #" + i + "(" + key.length + "): " + WtcString.toHexString(key));
            }
            streamKeys.write(key, 0, key.length);
        }

        //WtcLog.info(TAG, "masterKey(" + masterKey.length + ")=" + WtcString.toHexString(masterKey));
        WtcEncryptorAes256Ecb encryptor = new WtcEncryptorAes256Ecb(masterKey);

        byte[] keysDecrypted = streamKeys.toByteArray();
        //WtcLog.info(TAG, "keysDecrypted(" + keysDecrypted.length + ")=" + WtcString.toHexString(keysDecrypted));
        byte[] keysEncrypted = encryptor.transform/*Unpadded*/(keysDecrypted, 0, keysDecrypted.length);
        //WtcLog.info(TAG, "keysEncrypted(" + keysEncrypted.length + ")=" + WtcString.toHexString(keysEncrypted));

        return keysEncrypted;
    }

    protected void decryptKeys(byte[] masterKey, int numKeys, int numBytesPerKey, byte[] keysEncrypted,
                    byte[][] keysClientToServer, byte[][] keysServerToClient) throws WtcKexCryptoException, IOException
    {
        Assert.assertEquals("keysClientToServer.length != keysServerToClient.length", keysClientToServer.length,
                        keysServerToClient.length);

        //WtcLog.info(TAG, "masterKey(" + masterKey.length + ")=" + WtcString.toHexString(masterKey));
        WtcDecryptorAes256Ecb decryptor = new WtcDecryptorAes256Ecb(masterKey);

        //WtcLog.info(TAG, "keysEncrypted(" + keysEncrypted.length + ")=" + WtcString.toHexString(keysEncrypted));
        byte[] keysDecrypted = decryptor.transform/*Unpadded*/(keysEncrypted, 0, keysEncrypted.length);
        //WtcLog.info(TAG, "keysDecrypted(" + keysDecrypted.length + ")=" + WtcString.toHexString(keysDecrypted));

        byte[] key;
        int baseOffset;

        baseOffset = 0;
        for (int i = 0; i < numKeys; i++)
        {
            key = WtcArraysPlatform.copy(keysDecrypted, baseOffset + i * numBytesPerKey, numBytesPerKey);
            if (VERBOSE_LOG)
            {
                WtcLog.info(TAG, "keyClientToServer #" + i + ": " + WtcString.toHexString(key));
            }
            keysClientToServer[i] = key;
        }

        baseOffset = numKeys * numBytesPerKey;
        for (int i = 0; i < numKeys; i++)
        {
            key = WtcArraysPlatform.copy(keysDecrypted, baseOffset + i * numBytesPerKey, numBytesPerKey);
            if (VERBOSE_LOG)
            {
                WtcLog.info(TAG, "keyServerToClient #" + i + ": " + WtcString.toHexString(key));
            }
            keysServerToClient[i] = key;
        }
    }

    protected void initializePayloadTransforms(byte[] ivLocalToRemote, byte[][] keysLocalToRemote, byte[] ivRemoteToLocal,
                    byte[][] keysRemoteToLocal, int side) throws WtcKexCryptoException
    {

        Assert.assertEquals("keysLocalToRemote.length != keysRemoteToLocal.length", //
                        keysLocalToRemote.length, keysRemoteToLocal.length);

        int numKeys = keysLocalToRemote.length;
        IWtcTransformer[] decryptors = new IWtcTransformer[numKeys];
        IWtcTransformer[] encryptors = new IWtcTransformer[numKeys];
        IWtcTransformer transform = null;
        byte[] nonceDecryptor = null;
        byte[] nonceEncryptor = null;

        switch (messageCipherMode)
        {
            case WtcMessageCipherMode.AES256CTR:
                nonceDecryptor = ivRemoteToLocal;
                for (int i = 0; i < numKeys; i++)
                {
                    byte[] key = keysRemoteToLocal[i];

                    //WtcLog.info(TAG, WtcCipherSide.names[side] + " decryptor keysRemoteToLocal[" + i + "]=" + WtcString.toHexString(key));

                    // Yes, an AES256CTR *decryption* transformer is an *Encryptor* w/ an iv of IV_ZERO
                    transform = new WtcEncryptorAes256Ecb(key);
                    decryptors[i] = transform;
                }
                nonceEncryptor = ivLocalToRemote;
                for (int i = 0; i < numKeys; i++)
                {
                    byte[] key = keysLocalToRemote[i];

                    //WtcLog.info(TAG, WtcCipherSide.names[side] + " encryptor keysLocalToRemote[" + i + "]=" + WtcString.toHexString(key));

                    // Yes, an AES256CTR encryption transformer has an iv of IV_ZERO
                    transform = new WtcEncryptorAes256Ecb(key);
                    encryptors[i] = transform;
                }
                break;
            default:
                throw new IllegalArgumentException("Unsupported WtcMessageCipherMode=" + messageCipherMode);
        }

        payloadDecryptor.initialize(decryptors, nonceDecryptor);
        payloadEncryptor.initialize(encryptors, nonceEncryptor);

        isInitialized = true;
    }

    public void decryptPayload(long extendedSequenceNumber, WtcpMessage message, //
                    byte[] workingBlockBuffer) throws WtcKexCryptoException
    {
        if (!isInitialized || !message.getShouldBeCrypted())
        {
            return;
        }
        WtcpHeader header = message.getHeader();
        int payloadOffset = header.getPayloadOffset();
        int payloadLength = header.getPayloadLength();
        byte[] messageBuffer = message.stream.getBuffer();
        decryptPayload(extendedSequenceNumber, messageBuffer, payloadOffset, payloadLength, workingBlockBuffer);
    }

    protected void decryptPayload(long extendedSequenceNumber, byte[] messageBuffer, int payloadOffset, int payloadLength, //
                    byte[] workingBlockBuffer) throws WtcKexCryptoException
    {
        if (!isInitialized)
        {
            //WtcLog.info(TAG, "Not Decrypting: isInitialized=" + isInitialized);
            return;
        }
        //WtcLog.info(TAG,
        //                "+decryptPayload(" + extendedSequenceNumber + ", "
        //                                + WtcString.toHexString(messageBuffer, 0, payloadOffset + payloadLength, true) + ")");
        payloadDecryptor.transformPayload(extendedSequenceNumber, messageBuffer, payloadOffset, payloadLength, //
                        workingBlockBuffer);
        //WtcLog.info(TAG,
        //                "-decryptPayload(" + extendedSequenceNumber + ", "
        //                                + WtcString.toHexString(messageBuffer, 0, payloadOffset + payloadLength, true) + ")");
    }

    public void encryptPayload(long extendedSequenceNumber, WtcpMessage message, //
                    byte[] workingBlockBuffer) throws WtcKexCryptoException
    {
        if (!isInitialized || !message.getShouldBeCrypted())
        {
            //WtcLog.info(TAG, "Not Encrypting: isInitialized=" + isInitialized + ", message.ShouldBeCrypted=" + message.ShouldBeCrypted);
            return;
        }
        WtcpHeader header = message.getHeader();
        int payloadOffset = header.getPayloadOffset();
        int payloadLength = header.getPayloadLength();
        byte[] messageBuffer = message.stream.getBuffer();
        encryptPayload(extendedSequenceNumber, messageBuffer, payloadOffset, payloadLength, workingBlockBuffer);
    }

    protected void encryptPayload(long extendedSequenceNumber, byte[] messageBuffer, int payloadOffset, int payloadLength, //
                    byte[] workingBlockBuffer) throws WtcKexCryptoException
    {
        if (!isInitialized)
        {
            //WtcLog.info(TAG, "Not Encrypting: isInitialized=" + isInitialized);
            return;
        }
        //WtcLog.info(TAG,
        //                "+encryptPayload(" + extendedSequenceNumber + ", "
        //                                + WtcString.toHexString(messageBuffer, 0, payloadOffset + payloadLength, true) + ")");
        payloadEncryptor.transformPayload(extendedSequenceNumber, messageBuffer, payloadOffset, payloadLength, //
                        workingBlockBuffer);
        //WtcLog.info(TAG,
        //                "-encryptPayload(" + extendedSequenceNumber + ", "
        //                                + WtcString.toHexString(messageBuffer, 0, payloadOffset + payloadLength, true) + ")");
    }
}
