package com.twistpair.wave.thinclient;

import java.lang.ref.WeakReference;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;

/**
 * <p>
 * From <a href="http://stackoverflow.com/a/13493726/252308">http://stackoverflow.com/a/13493726/252308</a><br>
 * Reference: <a href="http://www.androiddesignpatterns.com/2013/01/inner-class-handler-memory-leak.html">http://www.androiddesignpatterns.com/2013/01/inner-class-handler-memory-leak.html</a><br>
 * Creates a weak reference around a Handler object.<br>
 * This avoids the compiler warning "This Handler class should be static or leaks might occur".<br>
 * </p>
 * <p>
 * A handler which keeps a weak reference to a fragment. According to
 * Android's lint, references to Handlers can be kept around for a long
 * time - longer than Fragments for example. So we should use handlers
 * that don't have strong references to the things they are handling for.
 * </p>
 * <p>
 * You can use this class to more or less forget about that requirement.
 * Unfortunately you cannot have anonymous static inner classes, so it is a
 * little more verbose.
 * </p>
 * <p>
 * Example use:
 * <pre>
 * private static class MsgHandler extends WtcWeakReferenceHandler<MyFragment>
 * {
 *     public MsgHandler(MyFragment fragment) { super(fragment); }
 * 
 *     {@literal @}Override
 *     public void handleMessage(MyFragment fragment, Message msg)
 *     {
 *         fragment.doStuff(msg.arg1);
 *     }
 * }
 * 
 * MsgHandler handler = new MsgHandler(this);
 * </pre>
 * </p>
 */
public abstract class WtcWeakReferenceHandler<T> extends Handler
{
    private final WeakReference<T> mReference;

    public WtcWeakReferenceHandler(T handler)
    {
        this(Looper.getMainLooper(), handler);
    }

    public WtcWeakReferenceHandler(Looper looper, T handler)
    {
        super(looper);
        mReference = new WeakReference<T>(handler);
    }

    @Override
    public void handleMessage(Message msg)
    {
        T handler = mReference.get();
        if (handler == null)
        {
            return;
        }
        handleMessage(handler, msg);
    }

    /**
     * Needed because you cannot have anonymous static inner classes.
     * @param reference will never be null
     * @param msg may be null
     */
    protected abstract void handleMessage(T reference, Message msg);
}
