package com.twistpair.wave.thinclient;

import java.io.IOException;

import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.twistpair.wave.thinclient.WtcLocatorException.WtcLocatorResponseInvalidException;
import com.twistpair.wave.thinclient.kexcrypto.WtcCryptoUtilPlatform;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.net.WtcNet;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.net.WtcUriPlatform;
import com.twistpair.wave.thinclient.util.WtcString;
import com.twistpair.wave.thinclient.util.WtcVersionString;

public class WtcUpdate
{
    private static final String TAG = WtcLog.TAG(WtcUpdate.class);

    public static class WtcUpdateException extends Exception
    {
        public final WtcUri    uri;
        public final Exception innerException;

        protected WtcUpdateException(WtcUri uri)
        {
            this.uri = uri;
            this.innerException = null;
        }

        public WtcUpdateException(String message)
        {
            this(null, message, null);
        }

        public WtcUpdateException(WtcUri uri, String message)
        {
            this(uri, message, null);
        }

        public WtcUpdateException(WtcUri uri, String message, Exception innerException)
        {
            super(message);
            this.uri = uri;
            this.innerException = innerException;
        }
    }

    /**
     * Locator XML response was valid, but contained a logical Error code. 
     */
    public static class WtcUpdateErrorException extends WtcUpdateException
    {
        public final String clientPackageCode;
        public final int    errorCode;

        public WtcUpdateErrorException(WtcUri uri, String clientPackageCode, int errorCode)
        {
            super(uri);
            this.clientPackageCode = clientPackageCode;
            this.errorCode = errorCode;
        }

        public String toString()
        {
            return "WtcUpdateErrorException [clientPackageCode=\"" + clientPackageCode + "\", errorCode="
                            + WtcLocatorErrorCodes.toString(errorCode) + "]";
        }
    }

    protected final WtcVersionString updateVersion;
    protected final WtcUri           updateUri;
    protected final boolean          updateRequired;
    protected final boolean          updateAvailable;

    protected WtcUpdate(Node softwareVersionInfoNode, WtcVersionString clientVersion) throws WtcUpdateException
    {
        NamedNodeMap attributes = softwareVersionInfoNode.getAttributes();
        if (attributes == null)
        {
            throw new WtcUpdateException("SoftwareVersionInfo node has no attributes");
        }

        String attrCurrVersion = attributes.getNamedItem("CurrVersion").getNodeValue();
        String attrInstallLocationURI = attributes.getNamedItem("InstallLocationURI").getNodeValue();
        String attrForceSoftwareUpdate = attributes.getNamedItem("ForceSoftwareUpdate").getNodeValue();

        updateVersion = new WtcVersionString(attrCurrVersion);
        updateUri = WtcUriPlatform.parse(attrInstallLocationURI);
        updateRequired = attrForceSoftwareUpdate.equals("Y") && !updateVersion.equals(clientVersion);
        updateAvailable = updateVersion.greaterThan(clientVersion) || updateRequired;
    }

    public WtcVersionString getVersion()
    {
        return updateVersion;
    }

    public WtcUri getUri()
    {
        return updateUri;
    }

    public boolean isRequired()
    {
        return updateRequired;
    }

    public boolean isAvailable()
    {
        return updateAvailable;
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append("{updateAvailable=").append(updateAvailable) //
        .append(",updateRequired=").append(updateRequired) //
        .append(",updateVersion=").append(updateVersion) //
        .append(",updateUri=\"").append(updateUri).append("\"}");
        return sb.toString();
    }

    /**
     * Example:
     *  http://wtcdev.twistpair.com/wave/_interface/proxy_locator.asp?action=GETSOFTWAREVER&pc=TPS-BB-5.x&ver=5.2.0.30000
     *  
     * @param proxy
     * @param clientPackageCode
     * @param clientVersion
     * @return WtcUpdate
     * @throws IOException 
     * @throws WtcUpdateException 
     */
    public static WtcUpdate checkForUpdate(WtcProxyInfo proxy, String clientPackageCode, WtcVersionString clientVersion)
                    throws IOException, WtcUpdateException
    {
        try
        {
            WtcLog.info(TAG, "+checkForUpdate(proxy=" + proxy + ", clientPackageCode=\"" + clientPackageCode
                            + "\", clientVersion=" + clientVersion + ")");

            WtcUri uriUpdate = WtcUriPlatform.parse(proxy.getSoftwareVersionCheckURL());
            WtcUri.Builder builder = uriUpdate.buildUpon();
            builder.appendQueryParameter("pc", clientPackageCode);
            builder.appendQueryParameter("ver", clientVersion.toString());
            uriUpdate = builder.build();

            // Add a unique nonce so that we always bypass any system cache
            uriUpdate = WtcLocator.verifyQuery(uriUpdate, "nonce", WtcCryptoUtilPlatform.getRandomInt64());

            Element xmlDoc;
            try
            {
                xmlDoc = WtcNet.makeRequestXml(uriUpdate);
            }
            catch (WtcLocatorResponseInvalidException e)
            {
                throw new WtcUpdateException(uriUpdate, "makeRequestXml", e);
            }

            if (xmlDoc == null)
            {
                throw new WtcUpdateException(uriUpdate, "XML response not valid");
            }

            String tagName = xmlDoc.getTagName();
            if (!tagName.equals("ServerResponse"))
            {
                throw new WtcUpdateException(uriUpdate, "XML root node name is not \"ServerResponse\"");
            }

            String error = xmlDoc.getAttribute("Error");
            if (WtcString.isNullOrEmpty(error))
            {
                throw new WtcUpdateException(uriUpdate, "ServerResponse node has no \"Error\" attribute");
            }

            //error = "3"; // simulate an error

            int errorCode = Integer.parseInt(error);
            if (errorCode != WtcLocatorErrorCodes.OK)
            {
                throw new WtcUpdateErrorException(uriUpdate, clientPackageCode, errorCode);
            }

            NodeList sviNodes = xmlDoc.getElementsByTagName("SoftwareVersionInfo");
            if (sviNodes == null)
            {
                throw new WtcUpdateException(uriUpdate, "ServerResponse node has no \"SoftwareVersionInfo\" child node(s)");
            }

            int sviCount = sviNodes.getLength();
            if (sviCount != 1)
            {
                throw new WtcUpdateException(uriUpdate,
                                "ServerResponse node \"SoftwareVersionInfo\" must only have one child node");
            }

            Node sviNode = sviNodes.item(0);

            WtcUpdate update = new WtcUpdate(sviNode, clientVersion);

            return update;
        }
        finally
        {
            WtcLog.info(TAG, "-checkForUpdate(proxy=" + proxy + ", clientPackageCode=\"" + clientPackageCode
                            + "\", clientVersion=" + clientVersion + ")");
        }
    }
}
