package com.twistpair.wave.thinclient;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.twistpair.wave.thinclient.WtcLocatorException.WtcLocatorErrorException;
import com.twistpair.wave.thinclient.WtcLocatorException.WtcLocatorResponseInvalidException;
import com.twistpair.wave.thinclient.WtcMessageFilter.IRequestTxResponseRxTimeoutListener;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackMessageReceiveHeaderInvalidException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackMessageReceiveOverflowException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackMessageRequestResponseTimeoutException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackProxyConnectException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackProxyLocateException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackProxyLocateLocatorException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackRemoteDisconnectException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSecurityAgreementException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSecurityInitializationException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSessionCloseException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackThreadProcessReceivedMessagesException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackThreadSendException;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase.WtcKexCryptoException;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoClient;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoClient.WtcKexPrimeSize;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.logging.WtcLogPlatform;
import com.twistpair.wave.thinclient.net.WtcInetSocketAddressPlatform;
import com.twistpair.wave.thinclient.net.WtcNetworkExceptionPlatform.WtcNetworkUnknownHostException;
import com.twistpair.wave.thinclient.net.WtcSocket;
import com.twistpair.wave.thinclient.net.WtcSocketExceptionPlatform;
import com.twistpair.wave.thinclient.net.WtcSocketPlatform;
import com.twistpair.wave.thinclient.net.WtcSocketTimeoutException;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpMessage;
import com.twistpair.wave.thinclient.protocol.WtcpMessagePool;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader.ExtendedNumber;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.IntegerPlatform;
import com.twistpair.wave.thinclient.util.WtcArrayBlockingQueue;
import com.twistpair.wave.thinclient.util.WtcArrayQueue;
import com.twistpair.wave.thinclient.util.WtcString;

//
// BEGIN WtcStackConnectionManager
//
public class WtcStackConnectionManager //
                extends Thread //
                implements IRequestTxResponseRxTimeoutListener //
{
    private//
    static//
    final//
    String  TAG                                  = WtcLog.TAG(//
                                                 WtcStackConnectionManager.class//
                                                 );

    /**
     * Set to false because sometimes WtcCryptoUtilPlatform.transform throws a mysterious ShortBufferException "need at least 32 bytes".
     * We only ever give the AES256 encoder 16 bytes; I have no idea why it says AES256 requires at least 32 bytes.
     * I think this is a fundamental Android/OS bug w/ multi-threaded encryption/decryption.
     * TODO:(pv) Find BouncyCastle AES crypto code and look for multi-threading problem.
     */
    private final//
    boolean //
            dedicatedMediaThreads                = false;

    /**
     * Can be changed at any time to enable *BASIC* tracing of all media messages.
     */
    public//
    boolean //
            traceMessageMediaRx                  = false;
    public//
    boolean //
            traceMessageMediaTx                  = false;

    public//
    boolean //
            traceKeyExchange                     = false;

    /**
     * Can be changed at any time to enable tracing of raw message bytes.
     * Raw bytes media bytes are not traced.
     */
    public//
    boolean //
            traceMessageRawBytes                 = false;
    public//
    boolean //
            traceMessageRawBytesAfterSessionOpen = false;

    public//
    boolean //
            traceMessageHeaders                  = false;

    public class WtcStackWorkItem
    {
        private static final int TYPE_UNKNOWN = -1;
        public static final int  TYPE_TXing   = 1;
        public static final int  TYPE_RXed    = 2;

        private int              workItemType;
        private long             lastRXedWtcpSequenceNumberExtended;
        private WtcpMessage      workItemMessage;

        private WtcStackWorkItem()
        {
            reset();
        }

        private void reset()
        {
            set(WtcStackWorkItem.TYPE_UNKNOWN, -1, null);
        }

        /**
         * @param workItemType TYPE_TXing or TYPE_RXed
         * @param lastRXedWtcpSequenceNumberExtended Should be -1 if the workItemType is TYPE_TXing
         * @param workItemMessage
         */
        private void set(int workItemType, long lastRXedWtcpSequenceNumberExtended, WtcpMessage workItemMessage)
        {
            this.workItemType = workItemType;
            this.lastRXedWtcpSequenceNumberExtended = lastRXedWtcpSequenceNumberExtended;
            this.workItemMessage = workItemMessage;
        }
    } // WtcStackWorkItem

    public class WtcStackWorkItemPool
    {
        private final String        TAG = WtcLog.TAG(WtcStackWorkItemPool.class);

        private final WtcArrayQueue pool;

        public WtcStackWorkItemPool(String name)
        {
            pool = new WtcArrayQueue(name);
        }

        public void add(WtcStackWorkItem workItem)
        {
            //WtcLog.error(TAG, "+add(" + workItem + ")");
            workItem.reset();
            synchronized (pool)
            {
                pool.add(workItem);
            }
            //WtcLog.error(TAG, "-add(" + workItem + ")");
        }

        /**
         * @param workItemType TYPE_TXing or TYPE_RXed
         * @param lastRXedWtcpSequenceNumberExtended Should be -1 if the workItemType is TYPE_TXing
         * @param workItemMessage
         */
        public WtcStackWorkItem get(int workItemType, long lastRXedWtcpSequenceNumberExtended, WtcpMessage workItemMessage)
        {
            //WtcLog.error(TAG, "+get()");
            WtcStackWorkItem workItem;
            synchronized (pool)
            {
                workItem = (WtcStackWorkItem) ((pool.isEmpty()) ? new WtcStackWorkItem() : pool.remove());
            }
            workItem.set(workItemType, lastRXedWtcpSequenceNumberExtended, workItemMessage);
            //WtcLog.error(TAG, "-get()");
            return workItem;
        }

        public void maintenance(boolean clear)
        {
            synchronized (pool)
            {
                pool.maintenance(clear);
            }
        }
    } // WtcStackWorkItemPool

    /**
    * Single thread that both TXes and processes RXed WtcpMessages.
    * ThreadConnection.send(...) makes sure that only UdpHello and Media are sent over UDP.
    * NOTE that, by design, we [currently] never *receive* data over UDP.
    * TODO:(pv) Find a reasonable way to implement UDP receive that works behind NAT/Firewall/etc?
    */
    private class ThreadWorker extends Thread
    {
        private final String                TAG;
        private final WtcStackWorkItemPool  workItemPool;
        private final WtcArrayBlockingQueue workItemQueue;
        private final OutputStream          outputStreamTcp;
        private final OutputStream          outputStreamUdp;
        private final byte[]                workingBlockBufferEncrypt;
        private final byte[]                workingBlockBufferDecrypt;

        /**
         * @param name
         * @param priority Thread.MAX_PRIORITY(10) to Thread.MIN_PRIORITY(1); Thread.NORM_PRIORITY=5
         * @param outputStreamTcp
         * @param outputStreamUdp
         */
        public ThreadWorker(String name, int priority, OutputStream outputStreamTcp, OutputStream outputStreamUdp)
        {
            super(name);

            //android.os.Process.setThreadPriority(android.os.Process.THREAD_PRIORITY_URGENT_AUDIO);
            setPriority(priority);

            this.TAG = name;

            // Uppercase the first letter
            name = name.substring(0, 1).toUpperCase() + name.substring(1);

            this.workItemPool = new WtcStackWorkItemPool("WorkItemPool" + name);
            this.workItemQueue = new WtcArrayBlockingQueue("WorkItemQueue" + name);
            this.outputStreamTcp = outputStreamTcp;
            this.outputStreamUdp = outputStreamUdp;
            this.workingBlockBufferEncrypt = new byte[WtcKexCryptoBase.BLOCK_LENGTH];
            this.workingBlockBufferDecrypt = new byte[WtcKexCryptoBase.BLOCK_LENGTH];
        }

        private void maintenance(boolean clear)
        {
            workItemPool.maintenance(clear);
            workItemQueue.maintenance();
        }

        /**
        * @param workItemType WtcStackWorkItem.TYPE_TXing or WtcStackWorkItem.TYPE_RXed
        * @param lastRXedWtcpSequenceNumberExtended Should be -1 if the workItemType is TYPE_TXing
        * @param workItemMessage
        */
        public void enqueue(int workItemType, long lastRXedWtcpSequenceNumberExtended, WtcpMessage workItemMessage)
        {
            //WtcLog.debug(TAG, "+enqueue(" + workItemType + ", " + workItemMessage + ")");
            WtcStackWorkItem workItem = workItemPool.get(workItemType, lastRXedWtcpSequenceNumberExtended, workItemMessage);
            workItemQueue.add(workItem);
            //WtcLog.debug(TAG, "-enqueue(" + workItemType + ", " + workItemMessage + ")");
        }

        public void run()
        {
            try
            {
                WtcLog.info(TAG, "+run()");

                synchronized (this)
                {
                    // Notify anyone waiting on us that we are now up and running.
                    this.notifyAll();
                }

                WtcStackWorkItem workItem;
                int workItemType;
                long lastRXedWtcpSequenceNumberExtended;
                WtcpMessage message;

                while (true)
                {
                    Thread.yield();

                    //try
                    //{
                    //WtcLog.info(TAG, "Waiting for next workItem...");
                    workItem = (WtcStackWorkItem) workItemQueue.take(); // NOTE: can throw InterruptedException
                    //WtcLog.info(TAG, "Dequeued workItem...");
                    //}
                    //catch (InterruptedException e)
                    //{
                    //    throw e;
                    //}
                    //catch (Exception e)
                    //{
                    //    throw new WtcStackThreadSendException("workItem = (WtcStackWorkItem) queue.take()", workItem, e);
                    //}

                    workItemType = workItem.workItemType;
                    message = workItem.workItemMessage;

                    switch (workItemType)
                    {
                        case WtcStackWorkItem.TYPE_TXing:
                            //WtcLog.info(TAG, "run(): WtcStackWorkItem.TYPE_TXing");
                            try
                            {
                                final int messageType = message.getMessageType();
                                final OutputStream outputStream;

                                // Send to the appropriate socket:
                                if (outputStreamUdp != null && //
                                                (messageType == WtcpMessageType.Media && stack.useUdpForMedia) //
                                                || messageType == WtcpMessageType.UdpHello)
                                {
                                    outputStream = outputStreamUdp;
                                }
                                else if (outputStreamTcp != null)
                                {
                                    outputStream = outputStreamTcp;
                                }
                                else
                                {
                                    // Expected to never happen...except *MAYBE* during shutdown
                                    throw new WtcStackThreadSendException("Cannot obtain TCP or UDP outputStream", message,
                                                    new IllegalStateException());
                                }

                                doMessageSend(outputStream, message);
                            }
                            catch (Exception e)
                            {
                                if (!(e instanceof WtcStackThreadSendException))
                                {
                                    e = new WtcStackThreadSendException("UNEXPECTED", //
                                                    message, e);
                                }
                                WtcLog.error(TAG, "run(): EXCEPTION", e);
                                throw e;
                            }
                            break;

                        case WtcStackWorkItem.TYPE_RXed:
                            //WtcLog.info(TAG, "run(): WtcStackWorkItem.TYPE_RXed");

                            lastRXedWtcpSequenceNumberExtended = workItem.lastRXedWtcpSequenceNumberExtended;

                            try
                            {
                                processMessageReceived(lastRXedWtcpSequenceNumberExtended, message);
                            }
                            catch (Exception e)
                            {
                                if (e instanceof WtcStackSessionCloseException)
                                {
                                    WtcLog.warn(TAG, "run(): WtcStackSessionCloseException; gracefully closing thread");
                                }
                                else
                                {
                                    if (!(e instanceof WtcStackThreadProcessReceivedMessagesException))
                                    {
                                        e = new WtcStackThreadProcessReceivedMessagesException("UNEXPECTED", //
                                                        message, e);
                                    }
                                    WtcLog.error(TAG, "run(): EXCEPTION", e);
                                }
                                throw e;
                            }
                            break;

                        default:
                            WtcLog.error(TAG, "UNHANDLED WtcStackWorkItem type " + workItemType);
                            // TODO:(pv) throw new Exception?
                            break;
                    }

                    // Return the message back to the pool that all messages are allocated from:
                    //WtcLog.debug(TAG, "Returning message back to message pool");
                    messagePool.add(message);
                    //workItemMessage = null;

                    workItemPool.add(workItem);
                    //workItem = null;
                }
            }
            catch (InterruptedException e)
            {
                WtcLog.info(TAG, "run(): InterruptedException; gracefully closing thread");
            }
            catch (Exception e)
            {
                disconnect(e);
            }
            finally
            {
                WtcLog.info(TAG, "-run()");
            }
        }

        private void doMessageSend(OutputStream outputStream, WtcpMessage message) //
                        throws WtcStackThreadSendException
        {
            try
            {
                prepareToSend(message);
            }
            catch (Exception e)
            {
                throw new WtcStackThreadSendException("prepareToSend", message, e);
            }

            try
            {
                sendToSocket(outputStream, message);
            }
            catch (Exception e)
            {
                String note = (kexSize != WtcKexPrimeSize.NONE) ? "[ENCRYPTED & NETWORK_ORDER]" : "[NETWORK_ORDER]";
                throw new WtcStackThreadSendException("sendToSocket " + note, message, e);
            }
        }

        private void prepareToSend(WtcpMessage message) //
                        throws WtcKexCryptoException
        {
            //WtcLog.info(TAG, "Adjust header fields (wtcp and wtcp-ctrl/media");
            long extendedSequenceNumber = adjustFields(message);

            //WtcLog.info(TAG, "Header Byte ordering; Except for encryption, message should be read-only after this point");
            message.dumpHeaderHostToNetworkOrder();

            //WtcLog.info(TAG, "Log message");
            logTxMessage(message);

            // Last chance to do anything with message before it is encrypted...
            messageFilter.start(message);

            //WtcLog.info(TAG, "Encrypt payload, update header");
            kexCryptoClient.encryptPayload(extendedSequenceNumber, message, workingBlockBufferEncrypt);

            //WtcLog.info(TAG, "Header Byte ordering");
            message.dumpHeaderHostToNetworkOrder();
        }

        private long adjustFields(WtcpMessage message)
        {
            WtcpHeader header = message.getHeader();

            //WtcLog.info(TAG, "Increment our internal counter and set the seq num");
            // TODO:(pv) Revisit this and perhaps clean up the logic a tad...
            long extendedSequenceNumber = lastTXedWtcpSequenceNumberExtended++;
            header.setSequenceNumber(header.normalizeSequenceNumber(extendedSequenceNumber));

            IWtcMemoryStream outputStream = message.stream;
            byte[] buffer = outputStream.getBuffer();
            int length = outputStream.getLength();

            byte messageType = header.getMessageType();
            connectionStatistics.incTxed(messageType, length);

            //WtcLog.info(TAG, "Set the payload length field");
            header.setPayloadLength(Math.max(0, length - header.getSize()));

            //WtcLog.info(TAG, "Update some fields for other message types");
            switch (header.getMessageType())
            {
                case WtcpMessageType.Control:

                    //connectionStatistics.incMessagesSentControl();

                    //WtcLog.info(TAG, "Deserialize (network to host order) the control header so that we can tweak it");
                    WtcpControlHeader controlHeader = (WtcpControlHeader) message.getSubHeader();

                    //WtcLog.info(TAG, "Increment last sent Control Sequence Number");
                    // TODO:(pv) Could this be moved in to WtcpMessageCache.removeOrCreate(...) or similar to transactionId?
                    controlHeader.sequenceNumber = controlHeader.normalizeSequenceNumber(++lastTXedWtcpControlSequenceNumber);
                    //WtcLog.info(TAG, "sequenceNumber=" + controlHeader.sequenceNumber);

                    // NOTE: controlHeader.transactionId is set during enqueue (so that it can be reported back to client/caller)

                    //WtcLog.info(TAG, "Dump the buffer Host-To-Network-Order before we calculate CRC");
                    controlHeader.dumpHostToNetworkOrder(outputStream);

                    //WtcLog.info(TAG, "Calculate the 16-bit CRC on the final Network-Order buffer");
                    controlHeader.crc = controlHeader.calculateCrc(buffer, length);
                    //WtcLog.info(TAG, "crc=0x" + WtcString.toHexString(controlHeader.crc, 2));

                    //WtcLog.info(TAG, "Re-serialize (host to network order) the control header");
                    // TODO:(pv) Re-serialize the header more efficiently
                    message.setSubHeader(controlHeader);
                    break;

                case WtcpMessageType.Media:

                    //connectionStatistics.incMessagesSentMedia();

                    //WtcLog.info(TAG, "Deserialize (network to host order) the media header so that we can tweak it");
                    WtcpMediaHeader mediaHeader = (WtcpMediaHeader) message.getSubHeader();

                    //WtcLog.info(TAG, "Increment last sent Media Sequence Number");
                    mediaHeader.setSequenceNumber(mediaHeader.normalizeSequenceNumber(++lastTXedWtcpMediaSequenceNumber));
                    //WtcLog.info(TAG, "sequenceNumber=" + mediaHeader.getSequenceNumber());

                    //WtcLog.info(TAG, "Dump the buffer Host-To-Network-Order before we calculate CRC");
                    mediaHeader.dumpHostToNetworkOrder(outputStream);

                    // TODO: Set optional Media header CRC value

                    //WtcLog.info(TAG, "Re-serialize (host to network order) the media header");
                    // TODO:(pv) Re-serialize the header more efficiently
                    message.setSubHeader(mediaHeader);
                    break;
            }

            return extendedSequenceNumber;
        }

        private void sendToSocket(OutputStream networkOutputStream, WtcpMessage message) //
                        throws IOException
        {
            IWtcMemoryStream outputStream = message.stream;
            byte[] buffer = outputStream.getBuffer();
            int length = outputStream.getLength();

            if (traceMessageHeaders)
            {
                WtcpHeader header = message.getHeader();
                WtcLog.debug(TAG, "TX " + length + "b: header=" + header.toString());
            }

            if (traceMessageRawBytes && WtcLog.isEnabled())
            {
                String raw = WtcString.toHexString(buffer, 0, length);
                WtcLog.debug(TAG, "TX " + length + "b (encrypted): " + raw);
            }

            networkOutputStream.write(buffer, 0, length);
            networkOutputStream.flush();
        }

        private void processMessageReceived(long lastRXedWtcpSequenceNumberExtended, WtcpMessage message)
                        //
                        throws WtcStackThreadProcessReceivedMessagesException, WtcStackSessionCloseException,
                        WtcStackSecurityAgreementException
        {
            WtcpHeader header = message.getHeader();
            IWtcMemoryStream inputStream = message.stream;

            header.loadNetworkToHostOrder(inputStream);

            try
            {
                kexCryptoClient.decryptPayload(lastRXedWtcpSequenceNumberExtended, message, workingBlockBufferDecrypt);
            }
            catch (Exception e)
            {
                throw new WtcStackThreadProcessReceivedMessagesException("decryptMessage", message, e);
            }

            // First chance to do anything with message after it is decrypted...

            // Log this before we cancel the timer so that we can see the message and debug any problems w/ the timer
            logRxMessage(message);

            boolean shouldIgnore;

            try
            {
                // Stop any timer that was waiting on a response to a control message request:
                shouldIgnore = messageFilter.cancel(message);
            }
            catch (Throwable tr)
            {
                // Purposes catches Throwable so that BlackBerry doesn't swallow the stack trace
                // TODO:(pv) Revisit and clean up this exception handling code...

                {
                    String stackTrace = WtcLogPlatform.getStackTraceString(this, tr);
                    WtcLog.error(TAG, stackTrace);
                }

                Exception e = (Exception) tr;
                throw new WtcStackThreadProcessReceivedMessagesException("messageFilter.cancel(message)", message, e);
            }

            int messageType = message.getMessageType();
            switch (messageType)
            {
                case WtcpMessageType.KeyExchange:
                    try
                    {
                        kexCryptoClient.processKexResponse(inputStream);
                        stack.processKeyExchangeMessage(inputStream);
                    }
                    catch (Exception e)
                    {
                        throw new WtcStackSecurityAgreementException(kexSize, e);
                    }
                    break;

                default:
                    try
                    {
                        stack.processReceivedMessage(message, shouldIgnore);
                    }
                    catch (WtcStackSessionCloseException e)
                    {
                        // Pass WtcStackSessionCloseException.errorCode to disconnect without wrapping it
                        throw e;
                    }
                    catch (Exception e)
                    {
                        /*
                        if (!(e instanceof WtcStackThreadProcessReceivedMessagesException))
                        {
                            e = new WtcStackThreadProcessReceivedMessagesException("processReceivedMessage", message, e);
                        }
                        throw (WtcStackThreadProcessReceivedMessagesException) e;
                        */
                        throw new WtcStackThreadProcessReceivedMessagesException("processReceivedMessage", message, e);
                    }
                    break;
            }
        }
    } // ThreadWorker

    /**
    * Used by isConnected()
    */
    private WtcSocket                    socketTcp;

    private WtcInetSocketAddressPlatform proxyAddress;

    private ThreadWorker                 threadWorkerMainProcessTx;
    private ThreadWorker                 threadWorkerMediaRx;
    private ThreadWorker                 threadWorkerMediaTx;

    /**
    * UInt32 extended sequence number sent that is reduced to 10bits when sent.
    */
    private long                         lastTXedWtcpSequenceNumberExtended;

    private long                         lastTXedWtcpControlSequenceNumber;
    private long                         lastTXedWtcpMediaSequenceNumber;

    /**
     * Set in "send(WtcpMessage message)"
     */
    private int                          lastTXingWtcpControlTransactionId;

    private//
    final//
    WtcStack                             stack;

    private//
    final//
    WtcConnectionStatistics              connectionStatistics;

    private//
    final//
    WtcpMessagePool                      messagePool;

    private//
    final//
    WtcKexCryptoClient                   kexCryptoClient;
    private final int                    kexSize;

    private//
    final//
    WtcMessageFilter                     messageFilter;

    private//
    final//
    Object                               syncObject    = new Object();

    private boolean                      exitReasonSet = false;
    private Exception                    exitReason    = null;

    /**
     * @param stack
     * @param threadPriority {@link Thread#MIN_PRIORITY} to {@link Thread#MAX_PRIORITY}; Recommend 6
     * @param kexSize One of WtcKexPrimeSize.*
     */
    public WtcStackConnectionManager(WtcStack stack, int threadPriority, int kexSize)
    {
        super("ThreadConnection");

        this.stack = stack;

        this.setPriority(threadPriority);

        this.connectionStatistics = stack.connectionStatistics;

        this.messagePool = new WtcpMessagePool();

        this.kexCryptoClient = new WtcKexCryptoClient();
        this.kexSize = kexSize;

        // TODO:(pv) Combine WtcConnectionStatistics & WtcMessageTxRequestRxResponseTimeout in to a WtcConnectionManager class similar to iOS
        //  https://www.assembla.com/code/twisted-pair/subversion/nodes/trunk/iOS/WAVE/WAVE/Classes/WTCStack/WTCConnectionManager.m
        //  That seems like a very nice pattern/design to use.
        this.messageFilter = new WtcMessageFilter(this, this.stack, connectionStatistics);

    }

    public void onRequestTxResponseRxTimeout(long timeoutMs, long elapsedMs, byte messageType, int opCode, int transactionId)
    {
        WtcStackMessageRequestResponseTimeoutException exitReason =
            new WtcStackMessageRequestResponseTimeoutException(timeoutMs, elapsedMs, //
                            messageType, opCode, transactionId);
        WtcLog.error(TAG, "RequestTxResponseRxTimeout", exitReason);
        disconnect(exitReason);
    }

    /**
    * If the first caller sets exitReason to null, then the exitReason is null (ie: no error).
    * @param exitReason
    */
    private void setExitReason(Exception exitReason)
    {
        synchronized (syncObject)
        {
            if (!exitReasonSet)
            {
                exitReasonSet = true;
                this.exitReason = exitReason;
            }
        }
    }

    private Exception getExitReason()
    {
        synchronized (syncObject)
        {
            return exitReason;
        }
    }

    public void disconnect(Exception exitReason)
    {
        try
        {
            WtcLog.debug(TAG, "+disconnect(exitReason=" + ((exitReason == null) ? "null" : exitReason.toString()) + ')');

            synchronized (syncObject)
            {
                setExitReason(exitReason);

                // IMPORTANT: Blocking socket reads need to be forcefully shut down in order to unblock their thread and properly close...
                shutdown(exitReason != null);

                // TODO:(pv) Might this cause any lock ups while disconnecting?
                WtcStack.interrupt(this, false);
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-disconnect(...)");
        }
    }

    public void maintenance()
    {
        // Clearing the buffers can cause a glitch in the speaker or mic.
        // Only clear the buffers if they are both not open. 
        boolean isSpeakerClosed = !stack.speaker.isOpen();
        boolean isMicrophoneClosed = !stack.microphone.isOpen();
        boolean clear = (isSpeakerClosed && isMicrophoneClosed);

        messagePool.maintenance(clear);

        if (threadWorkerMainProcessTx != null)
        {
            threadWorkerMainProcessTx.maintenance(clear);
        }

        if (threadWorkerMediaRx != null)
        {
            threadWorkerMediaRx.maintenance(clear);
        }

        if (threadWorkerMediaTx != null)
        {
            threadWorkerMediaTx.maintenance(clear);
        }

        stack.speaker.maintenance(clear);

        connectionStatistics.log();
    }

    private void logRxMessage(WtcpMessage message)
    {
        if (!WtcLog.isEnabled())
        {
            return;
        }

        //try
        //{
        //WtcLog.info(TAG, "+logRxMessage");
        switch (message.getMessageType())
        {
            case WtcpMessageType.Media:
                if (traceMessageMediaRx)
                {
                    WtcLog.debug(TAG, "RX " + message.stream.getLength() + "b: Media");
                }
                break;
            case WtcpMessageType.KeyExchange:
            {
                if (traceMessageRawBytes)
                {
                    int length = message.stream.getLength();
                    String raw = WtcString.toHexString(message.stream.getBuffer(), 0, length);
                    WtcLog.debug(TAG, "RX " + length + "b (decrypted): " + raw);
                }
                if (traceKeyExchange)
                {
                    WtcLog.debug(TAG, "RX " + message.stream.getLength() + "b: " + message.toString('x'));
                }
                break;
            }
            default:
                if (traceMessageRawBytes)
                {
                    int length = message.stream.getLength();
                    String raw = WtcString.toHexString(message.stream.getBuffer(), 0, length);
                    WtcLog.debug(TAG, "RX " + length + "b (decrypted): " + raw);
                }
                WtcLog.debug(TAG, "RX " + message.stream.getLength() + "b: " + message.toString());
                break;
        }
        //}
        //finally
        //{
        //WtcLog.info(TAG, "-logRxMessage");
        //}
    }

    private void logTxMessage(WtcpMessage message)
    {
        if (!WtcLog.isEnabled())
        {
            return;
        }

        // TODO:(pv) Censor the payload if this is a SetCredentials Request

        //try
        //{
        //WtcLog.info(TAG, "+logTxMessage");
        switch (message.getMessageType())
        {
            case WtcpMessageType.Media:
                if (traceMessageMediaTx)
                {
                    WtcLog.debug(TAG, "TX " + message.stream.getLength() + "b: Media");
                }
                break;
            case WtcpMessageType.KeyExchange:
            {
                if (traceKeyExchange)
                {
                    WtcLog.debug(TAG, "TX " + message.stream.getLength() + "b: " + message.toString('x'));
                }
                if (traceMessageRawBytes)
                {
                    int length = message.stream.getLength();
                    String raw = WtcString.toHexString(message.stream.getBuffer(), 0, length);
                    WtcLog.debug(TAG, "TX " + length + "b (decrypted): " + raw);
                }
                break;
            }
            default:
                WtcLog.debug(TAG, "TX " + message.stream.getLength() + "b: " + message.toString());
                if (traceMessageRawBytes)
                {
                    int length = message.stream.getLength();
                    String raw = WtcString.toHexString(message.stream.getBuffer(), 0, length);
                    WtcLog.debug(TAG, "TX " + length + "b (decrypted): " + raw);
                }
                break;
        }
        //}
        //finally
        //{
        //WtcLog.info(TAG, "-logTxMessage");
        //}
    }

    public WtcpMessage getMessage(byte messageType)
    {
        return messagePool.get(messageType);
    }

    public WtcpMessage getMessage(WtcpMediaHeader headerMedia)
    {
        return messagePool.get(headerMedia);
    }

    public WtcpMessage getMessage(int opCode)
    {
        return messagePool.get(opCode);
    }

    public WtcpMessage getMessage(int opType, int opCode)
    {
        return messagePool.get(opType, opCode);
    }

    public void sendKeyExchange(IWtcMemoryStream kexRequest)
    {
        try
        {
            WtcLog.info(TAG, "+sendKeyExchange([" + kexRequest.getLength() + " bytes])");

            WtcpMessage message = getMessage(WtcpMessageType.KeyExchange);
            message.payloadAppend(kexRequest);
            send(message);
        }
        finally
        {
            WtcLog.info(TAG, "-sendKeyExchange([" + kexRequest.getLength() + " bytes])");
        }
    }

    public Integer send(WtcpMessage message)
    {
        if (message == null)
        {
            throw new IllegalArgumentException("message cannot be null");
        }

        if (message.getIsMessageType(WtcpMessageType.Media))
        {
            //
            // Optimized but less robust path for sending media 
            //
            if (dedicatedMediaThreads)
            {
                threadWorkerMediaTx.enqueue(WtcStackWorkItem.TYPE_TXing, -1, message);
            }
            else
            {
                threadWorkerMainProcessTx.enqueue(WtcStackWorkItem.TYPE_TXing, -1, message);
            }
            return null;
        }

        //
        // More robust path for sending non-media
        //

        if (message.getShouldBeCrypted() && !kexCryptoClient.getIsInitialized() && kexSize != WtcKexPrimeSize.NONE)
        {
            throw new IllegalStateException("message requires secure connection");
        }

        synchronized (syncObject)
        {
            if (socketTcp == null)
            {
                // TODO:(pv) Just return?
                //throw new IllegalStateException("socketTcp is not connected");
                return null;
            }

            if (threadWorkerMainProcessTx == null)
            {
                // Should *NEVER* happen...except *MAYBE* during shutdown
                IllegalStateException e =
                    new IllegalStateException("threadWorkerMainProcessTx == null; Cannot send message either over TCP or UDP");
                disconnect(e);
                //throw e;
                return null;
            }

            Integer transactionId = null;

            if (message.getIsMessageType(WtcpMessageType.Control))
            {
                lastTXingWtcpControlTransactionId = WtcpControlHeader.getNextTransactionId(lastTXingWtcpControlTransactionId++);

                WtcpControlHeader controlHeader = (WtcpControlHeader) message.getSubHeader();
                controlHeader.transactionId = lastTXingWtcpControlTransactionId;
                // TODO:(pv) Re-serialize the header more efficiently
                message.setSubHeader(controlHeader);

                transactionId = IntegerPlatform.valueOf(lastTXingWtcpControlTransactionId);
            }

            threadWorkerMainProcessTx.enqueue(WtcStackWorkItem.TYPE_TXing, -1, message);

            return transactionId;
        }
    }

    protected void onMessageReceived(ExtendedNumber lastRXedWtcpSequenceNumberExtended, WtcpMessage message)
    {
        WtcpHeader header = message.getHeader();
        IWtcMemoryStream inputStream = message.stream;

        header.loadNetworkToHostOrder(inputStream);

        int expectedSequenceNumber = header.normalizeSequenceNumber(lastRXedWtcpSequenceNumberExtended.large + 1);
        int actualSequenceNumber = header.getSequenceNumber();
        if (lastRXedWtcpSequenceNumberExtended.large != 0 //
                        && actualSequenceNumber != 0 //
                        && actualSequenceNumber != expectedSequenceNumber)
        {
            WtcLog.warn(TAG,
                            "*POSSIBLY* invalid Header: lastReceivedWtcpSequenceNumberExtended=0x"
                                            + WtcString.toHexString(lastRXedWtcpSequenceNumberExtended.large, 8) //
                                            + ", expected sequenceNumber=" + WtcString.toHexString(expectedSequenceNumber, 2) //
                                            + ", actual sequenceNumber=" + WtcString.toHexString(actualSequenceNumber, 2) //
                                            + ", difference=" + (actualSequenceNumber - expectedSequenceNumber) //
                                            + ", header=" + header.toString());
        }

        if (traceMessageRawBytes && WtcLog.isEnabled())
        {
            int length = inputStream.getLength();
            String raw = WtcString.toHexString(inputStream.getBuffer(), 0, length);
            WtcLog.debug(TAG, "RX " + length + "b (encrypted): " + raw);
        }

        if (traceMessageHeaders)
        {
            int length = inputStream.getLength();
            WtcLog.debug(TAG, "RX " + length + "b: header=" + header.toString());
        }

        // Extend 10bit sequence # to 32bit extended sequence # used for decrypting
        lastRXedWtcpSequenceNumberExtended.small = header.getSequenceNumber();
        header.extendSequenceNumber(lastRXedWtcpSequenceNumberExtended);

        int messageType = message.getMessageType();
        switch (messageType)
        {
            case WtcpMessageType.Media:
                if (dedicatedMediaThreads)
                {
                    threadWorkerMediaRx.enqueue(WtcStackWorkItem.TYPE_RXed, lastRXedWtcpSequenceNumberExtended.large, message);
                }
                else
                {
                    threadWorkerMainProcessTx.enqueue(WtcStackWorkItem.TYPE_RXed, lastRXedWtcpSequenceNumberExtended.large,
                                    message);
                }
                break;

            default:
                threadWorkerMainProcessTx.enqueue(WtcStackWorkItem.TYPE_RXed, lastRXedWtcpSequenceNumberExtended.large, message);
                break;
        }
    }

    /**
    * Certain exceptions are ignorable and allow trying to connect to the next server
    * @param throwable
    * @return true if the Exception does not require the connection to be terminated
    */
    private boolean mayTryNextServer(Throwable throwable)
    {
        boolean mayTryNextServer = false;
        mayTryNextServer |= (throwable instanceof WtcNetworkUnknownHostException);
        mayTryNextServer |= (throwable instanceof WtcSocketTimeoutException);
        // TODO:(pv) Add more exceptions as appropriate
        // TODO:(pv) Move this to WtcNetworkExceptionPlatform.mayTryNextServer?
        return mayTryNextServer;
    }

    /**
    * Walks the array of given WtcUris and finds the first of either:
    * <ul>
    * <li>Socket address of a Proxy, given a uri scheme of "wtcp".</li>
    * <li>Socket address of a Proxy found by a Locator, given a uri scheme with "http" or "https".</li>
    * </ul> 
    * Aborts on the first non-DNS related error.
    * @param uriServers an array of any combination of "wtcp", "http", or "https" scheme uris (ex: "wtcp://192.168.0.1:4502").
    * @return WtcProxyInfo[] of the Proxy addresses to try to connect to
    * @throws InterruptedException
    * @throws WtcStackProxyLocateException if there is any problem connecting to the Locator; DNS errors are ignored *until the last Locator is attempted*.
    */
    private WtcProxyInfo[] findFirstProxyInfos(WtcUri[] uriServers) //
                    throws InterruptedException, WtcStackProxyLocateException
    {
        if (uriServers == null || uriServers.length == 0)
        {
            throw new IllegalArgumentException("uriServers cannot be null or empty");
        }

        WtcProxyInfo[] proxyInfos = null;

        WtcUri lastServer = null;
        Throwable lastThrowable = null;

        for (int i = 0; i < uriServers.length; i++)
        {
            WtcStack.checkForInterruptedException();

            lastServer = uriServers[i];

            try
            {
                proxyInfos = findProxyInfos(lastServer);
                break;
            }
            catch (Throwable throwable)
            {
                lastThrowable = throwable;

                if (!mayTryNextServer(lastThrowable))
                {
                    break;
                }
            }
        }

        if (proxyInfos == null)
        {
            throw new WtcStackProxyLocateLocatorException(lastServer, lastThrowable);
        }

        return proxyInfos;
    }

    private WtcProxyInfo[] findProxyInfos(WtcUri uriServer) //
                    throws InterruptedException, IOException, WtcLocatorResponseInvalidException, WtcLocatorErrorException
    {
        if (uriServer == null)
        {
            throw new IllegalArgumentException("uriServer cannot be null");
        }

        WtcProxyInfo[] proxyInfos = null;

        String scheme = uriServer.getScheme();
        if (WtcStack.URI_SCHEMA_WTCP.equalsIgnoreCase(scheme))
        {
            int port = uriServer.getPort();
            if (port == -1)
            {
                port = WtcStack.PORT_WTCP_DEFAULT;
            }
            proxyInfos = new WtcProxyInfo[]
            {
                new WtcProxyInfo(new WtcInetSocketAddressPlatform(uriServer.getHost(), port))
            };
            return proxyInfos;
        }

        if (!WtcUri.URI_SCHEME_HTTP.equalsIgnoreCase(scheme) //
                        && !WtcUri.URI_SCHEME_HTTPS.equalsIgnoreCase(scheme))
        {
            // Should never happen; the constructor to WtcStack should have prevented this.
            throw new IllegalArgumentException("remoteAddress scheme must be either wtcp, http, or https");
        }

        if (stack.version != null)
        {
            WtcUri.Builder builder = uriServer.buildUpon();
            builder.appendQueryParameter("version", stack.version.toString());
            uriServer = builder.build();
        }

        WtcStackListener listener = stack._listener;
        if (listener != null)
        {
            listener.onProxyLocating(stack, uriServer);
        }

        connectionStatistics.incLocatorAttempts();

        WtcLocatorResponse locatorResponse = WtcLocator.locateProxies(uriServer);
        WtcStack.checkForInterruptedException();

        if (locatorResponse.isError())
        {
            throw new WtcLocatorErrorException(locatorResponse.errorCode);
        }

        connectionStatistics.incLocatorSuccess();

        listener = stack._listener;
        if (listener != null)
        {
            listener.onProxyLocated(stack, locatorResponse.proxyInfos);
        }

        return locatorResponse.proxyInfos;
    }

    /**
    * Find the first proxyAddress that we can successfully connect to.
    * @param uriServers
    * @return InputStream of the connected TCP connection to the Proxy; never null (will throw Exception)
    * @throws InterruptedException
    * @throws IOException
    * @throws WtcStackProxyLocateException
    * @throws WtcStackProxyConnectException
    */
    private InputStream connectFirstProxyAddress(WtcUri[] uriServers) //
                    throws InterruptedException, IOException, WtcStackProxyLocateException, WtcStackProxyConnectException
    {
        WtcProxyInfo[] proxyInfos = findFirstProxyInfos(uriServers);

        // TODO:(pv) Clean up this logic a tad; handle it as a loop no-op exit condition
        if (proxyInfos == null || proxyInfos.length == 0)
        {
            throw new IllegalArgumentException("proxyInfos cannot be null or empty");
        }

        WtcInetSocketAddressPlatform proxyAddress = null;
        Throwable lastThrowable = null;

        for (int i = 0; i < proxyInfos.length; i++)
        {
            WtcProxyInfo proxyInfo = proxyInfos[i];

            WtcInetSocketAddressPlatform[] proxyAddresses = proxyInfo.getInetSocketAddresses();
            for (int j = 0; j < proxyAddresses.length; j++)
            {
                WtcStack.checkForInterruptedException();

                proxyAddress = proxyAddresses[j];

                WtcStackListener listener = stack._listener;
                if (listener != null)
                {
                    listener.onProxyConnecting(stack, proxyAddress);
                }

                connectionStatistics.incProxyConnectAttempts();

                WtcSocket proxySocket = new WtcSocketPlatform();

                // TODO:(pv) timeout, and other connection/socket options
                //conn.setReadTimeout(timeoutConnect);

                try
                {
                    WtcLog.info(TAG, "+proxySocket.connect(" + proxyAddress + ", " + stack.connectTimeoutMs + ", "
                                    + stack.localAddress + ")");

                    /*
                    if (true)
                    {
                        throw new SocketTimeoutException("manually induced for testing");
                    }
                    */

                    proxySocket.connect(proxyAddress, stack.connectTimeoutMs, stack.localAddress);
                    WtcLog.info(TAG, "-proxySocket.connect(" + proxyAddress + ", " + stack.connectTimeoutMs + ", "
                                    + stack.localAddress + ")");
                    WtcStack.checkForInterruptedException();

                    proxyAddress = proxySocket.getRemoteAddress();

                    WtcLog.info(TAG, "Connected: " + proxyAddress);
                }
                catch (Throwable throwable)
                {
                    WtcLog.warn(TAG, "-proxySocket.connect(" + proxyAddress + ", " + stack.connectTimeoutMs + ", "
                                    + stack.localAddress + ")", throwable);

                    lastThrowable = throwable;

                    if (!mayTryNextServer(lastThrowable))
                    {
                        WtcLog.error(TAG, "Connect Error: " + proxyAddress, lastThrowable);
                        throw new WtcStackProxyConnectException(proxyAddress, lastThrowable);
                    }

                    WtcLog.warn(TAG, "Connect Error: " + proxyAddress, lastThrowable);

                    continue;
                }

                connectionStatistics.incProxyConnectSuccess();

                // NOTE: We are now connected, but we are intentionally delaying notification of
                // onProxyConnected until *AFTER* all support classes are fully initialized in the "startup" method.
                InputStream inputStreamTcp = startup(proxySocket);

                // All support classes are up and running; *NOW* we notify onProxyConnected.
                listener = stack._listener;
                if (listener != null)
                {
                    listener.onProxyConnected(stack, proxyInfo, proxyAddress);
                }

                return inputStreamTcp;
            }
        }

        throw new WtcStackProxyConnectException(proxyAddress, lastThrowable);
    }

    protected InputStream startup(WtcSocket proxySocketTcp) //
                    throws IOException, InterruptedException
    {
        try
        {
            WtcLog.info(TAG, "+startup(proxySocketTcp=" + proxySocketTcp + ')');

            synchronized (syncObject)
            {
                this.proxyAddress = proxySocketTcp.getRemoteAddress();

                this.socketTcp = proxySocketTcp;

                InputStream inputStreamTcp = proxySocketTcp.getInputStream();
                OutputStream outputStreamTcp = proxySocketTcp.getOutputStream();

                // TODO:(pv) Create UDP output stream; unsupported for now
                OutputStream outputStreamUdp = null;

                // Route the microphone capture buffer to the network
                stack.microphone.setBufferListener(stack);

                /**
                 * Recommend:
                 *  threadPriorityRx = 6; // Above normal: may contain audio
                 *  threadPriorityControlRxTx = 5; // Normal; does not contain audio
                 *  threadPriorityMediaRx = 7; // Above above normal: contains audio
                 *  threadPriorityMediaTx = 7; // Above above normal: contains audio
                 */
                int threadPriorityRx = getPriority();
                int threadPriorityMainProcessTx = threadPriorityRx - 1;
                int threadPriorityMediaRx = threadPriorityRx + 1;
                int threadPriorityMediaTx = threadPriorityRx + 1;

                WtcLog.info(TAG, "Starting threadWorkerMainProcessTx");
                threadWorkerMainProcessTx =
                    new ThreadWorker("threadWorkerMainProcessTx", threadPriorityMainProcessTx, outputStreamTcp, outputStreamUdp);
                synchronized (threadWorkerMainProcessTx)
                {
                    threadWorkerMainProcessTx.start();
                    threadWorkerMainProcessTx.wait();
                }
                WtcLog.info(TAG, "Started threadWorkerMainProcessTx");

                if (dedicatedMediaThreads)
                {
                    WtcLog.info(TAG, "Starting threadWorkerMediaRx");
                    threadWorkerMediaRx =
                        new ThreadWorker("ThreadWorkerMediaRx", threadPriorityMediaRx, outputStreamTcp, outputStreamUdp);
                    synchronized (threadWorkerMediaRx)
                    {
                        threadWorkerMediaRx.start();
                        threadWorkerMediaRx.wait();
                    }
                    WtcLog.info(TAG, "Started threadWorkerMediaRx");

                    WtcLog.info(TAG, "Starting threadWorkerMediaTx");
                    threadWorkerMediaTx =
                        new ThreadWorker("ThreadWorkerMediaTx", threadPriorityMediaTx, outputStreamTcp, outputStreamUdp);
                    synchronized (threadWorkerMediaTx)
                    {
                        threadWorkerMediaTx.start();
                        threadWorkerMediaTx.wait();
                    }
                    WtcLog.info(TAG, "Started threadWorkerMediaTx");
                }

                return inputStreamTcp;
            }
        }
        finally
        {
            WtcLog.info(TAG, "-startup(proxySocketTcp=" + proxySocketTcp + ')');
        }
    }

    /**
    * Force closes first all socket operations and then threads.
    * @param error true if the shutdown is the result of an error; passed to microphone.close to optionally play an error sound.
    */
    protected void shutdown(boolean error)
    {
        try
        {
            WtcLog.info(TAG, "+shutdown(error=" + error + ')');

            synchronized (syncObject)
            {
                // TODO:(pv) Clean up the log by not running any of the below code if we are already shut down 
                try
                {
                    WtcLog.info(TAG, "+shutdown microphone");
                    stack.microphone.setBufferListener(null);
                    // TODO:(pv) Need to run callback?
                    stack.microphone.close(error, null);
                }
                finally
                {
                    WtcLog.info(TAG, "-shutdown microphone");
                }

                try
                {
                    WtcLog.info(TAG, "+shutdown speaker");
                    stack.speaker.close(error);
                }
                finally
                {
                    WtcLog.info(TAG, "-shutdown speaker");
                }

                try
                {
                    WtcLog.info(TAG, "+shutdown messageFilter");

                    // Cancel here before closing socketTcp or ThreadSend instances to prevent firing any [more] timeouts:
                    messageFilter.close();
                }
                finally
                {
                    WtcLog.info(TAG, "-shutdown timeoutTxRequestRxResponse");
                }

                try
                {
                    WtcLog.info(TAG, "+shutdown socketTcp");

                    if (socketTcp != null)
                    {
                        socketTcp.shutdown();

                        try
                        {
                            WtcLog.info(TAG, "+socketTcp.close()");
                            socketTcp.close();
                            WtcLog.info(TAG, "-socketTcp.close()");
                        }
                        catch (IOException e)
                        {
                            WtcLog.error(TAG, "socketTcp.close(); // ignore", e);
                        }

                        socketTcp = null;
                    }
                    else
                    {
                        WtcLog.info(TAG, "socketTcp == null; // ignore");
                    }
                }
                finally
                {
                    WtcLog.info(TAG, "-shutdown socketTcp");
                }

                try
                {
                    WtcLog.info(TAG, "+shutdown threads");

                    WtcStack.interrupt(threadWorkerMainProcessTx, false);
                    threadWorkerMainProcessTx = null;

                    WtcStack.interrupt(threadWorkerMediaRx, false);
                    threadWorkerMediaRx = null;

                    WtcStack.interrupt(threadWorkerMediaTx, false);
                    threadWorkerMediaTx = null;
                }
                finally
                {
                    WtcLog.info(TAG, "-shutdown threads");
                }
            }
        }
        finally
        {
            WtcLog.info(TAG, "-shutdown(error=" + error + ')');
        }
    }

    public void run()
    {
        try
        {
            WtcLog.info(TAG, "+run()");

            // TODO:(pv) synchronize(this)? Some of these fields can change in other threads?
            // Rename all non-final fields to *2 and see what conflicts occur...especially inside the "while(true)" loop below

            // Save a copy here in case another thread changes while we are KEXing below...
            final int kexSize = this.kexSize;

            try
            {
                // Calculating KEX can take quite a few milliseconds, so start this async before trying to connect.
                // TODO:(pv) Allow request for dynamic KexCrypto renegotiation from WtcClient
                // TODO:(pv) Handle dynamic KexCrypto renegotiation from WTC Server
                // TODO:(pv) Handle dynamic change of kexSize (including NONE?)
                kexCryptoClient.createKexRequestAsync(kexSize);
            }
            catch (Exception e)
            {
                WtcLog.error(TAG, "createKexRequest", e);
                throw new WtcStackSecurityInitializationException(kexSize, e);
            }

            InputStream inputStreamTcp = connectFirstProxyAddress(stack.remoteAddresses);
            WtcStack.checkForInterruptedException();

            if (kexSize == WtcKexPrimeSize.NONE)
            {
                WtcStackListener listener = stack._listener;
                if (listener != null)
                {
                    listener.onProxySecured(stack, kexSize);
                }
            }
            else
            {
                // NOTE: listener.onProxySecured will be fired after processing KEX *Response*
                WtcStackListener listener = stack._listener;
                if (listener != null)
                {
                    listener.onProxySecuring(stack, kexSize);
                }

                IWtcMemoryStream kexRequest;

                try
                {
                    // If the createKexRequest hasn't already completed, wait here for it to complete
                    kexRequest = kexCryptoClient.waitForKexRequest();
                }
                catch (Exception e)
                {
                    WtcLog.error(TAG, "waitForKexRequest", e);
                    throw new WtcStackSecurityInitializationException(kexSize, e);
                }

                sendKeyExchange(kexRequest);
                WtcStack.checkForInterruptedException();
            }

            WtcpMessage message = null;
            IWtcMemoryStream inputStream = null;
            byte[] buffer;
            int position;
            int remaining = 0;
            int received;

            /**
             * First used to verify the sequence # is contiguous, then used to decrypt the message. 
             */
            final ExtendedNumber lastRXedWtcpSequenceNumberExtended = new ExtendedNumber();

            // Loop forever, until:
            // 1) The remote side disconnects
            // 2) interrupt(new WtcStackLocalCloseException())
            // 3) Health PING times out
            // 4) A local thread encounters an error
            // ...
            while (true)
            {
                Thread.yield();

                if (remaining == 0)
                {
                    //WtcLog.debug(TAG, "RXing new message header");
                    message = messagePool.get();
                    inputStream = message.stream;
                    remaining = inputStream.getLength();
                }
                else
                {
                    //WtcLog.debug(TAG, "RXing remaining message bytes (" + remaining + ")");
                    inputStream = message.stream;
                    inputStream.setLength(inputStream.getPosition() + remaining);
                }

                buffer = inputStream.getBuffer();
                position = inputStream.getPosition();
                //WtcLog.info(TAG, "+inputStreamTcp.read([...], " + position + ", " + remaining + ")");
                received = inputStreamTcp.read(buffer, position, remaining);
                //WtcLog.info(TAG, "-inputStreamTcp.read([...], " + position + ", " + remaining + "); received=" + received);
                if (received == -1)
                {
                    WtcLog.warn(TAG, "received " + received + " bytes; remote disconnect (end of stream)");
                    throw new WtcStackRemoteDisconnectException();
                }

                //WtcLog.info(TAG, "buffer=" + WtcString.toHexString(buffer, inputStream.getPosition(), received));

                remaining = getBytesRemaining(received, message);
                if (remaining == 0)
                {
                    //WtcLog.debug(TAG, "RXed complete message; enqueue for processing.");
                    onMessageReceived(lastRXedWtcpSequenceNumberExtended, message);
                }
            }
        }
        catch (Exception e)
        {
            setExitReason(e);
        }
        finally
        {
            final Exception exitReason = getExitReason();

            final boolean isError = (exitReason != null);

            if (isError)
            {
                if (exitReason instanceof WtcStackSessionCloseException)
                {
                    WtcLog.warn(TAG, "run(): EXCEPTION " + exitReason);
                }
                else if (exitReason instanceof WtcSocketExceptionPlatform)
                {
                    WtcLog.warn(TAG, "run(): EXCEPTION", exitReason);
                }
                else
                {
                    WtcLog.error(TAG, "run(): EXCEPTION", exitReason);
                }
            }

            if (proxyAddress != null)
            {
                connectionStatistics.incProxyDisconnects();
            }

            WtcStackListener listener = stack._listener;
            if (listener != null)
            {
                listener.onDisconnected(stack, proxyAddress, exitReason, connectionStatistics);
            }

            shutdown(isError);

            WtcLog.info(TAG, "-run()");
        }
    }

    private int getBytesRemaining(int received, WtcpMessage message) //
                    throws WtcStackMessageReceiveOverflowException, WtcStackMessageReceiveHeaderInvalidException
    {
        if (received < 0)
        {
            throw new IllegalArgumentException("received must be >= 0");
        }

        if (received > WtcpHeader.MAX_PAYLOAD_LENGTH)
        {
            throw new WtcStackMessageReceiveOverflowException(WtcpHeader.MAX_PAYLOAD_LENGTH, received);
        }

        IWtcMemoryStream inputStream = message.stream;
        if (received > inputStream.getLength())
        {
            throw new WtcStackMessageReceiveOverflowException(inputStream.getLength(), received);
        }

        inputStream.setPosition(inputStream.getPosition() + received);

        int bufferPosition = inputStream.getPosition();
        if (bufferPosition < WtcpHeader.SIZE)
        {
            WtcLog.warn(TAG, "Message Underflow: expected=" + WtcpHeader.SIZE + ", actual=" + received);
            //throw new WtcStackMessageUnderflowException(received, WtcpHeader.SIZE);
            return WtcpHeader.SIZE - bufferPosition;
        }

        WtcpHeader header = message.getHeader();
        if (bufferPosition == WtcpHeader.SIZE)
        {
            header.loadNetworkToHostOrder(inputStream);
            //WtcLog.info(TAG, "header=" + header.toString());
        }

        // Now that we've loaded the header, verify the header values after *every* read; helps to detect potential attacks/corruption
        if (header.getVersion() != WtcpHeader.CURRENT_VERSION)
        {
            // Currently we only support v1 messages.
            throw new WtcStackMessageReceiveHeaderInvalidException(header);
        }

        int messageType = header.getMessageType();
        if ((messageType != WtcpMessageType.Media) && (messageType != WtcpMessageType.Control)
                        && (messageType != WtcpMessageType.KeyExchange))
        {
            // Currently we only support receiving Media, Control, and KEX messages.
            throw new WtcStackMessageReceiveHeaderInvalidException(header);
        }

        // Calculate the remaining bytes to be read
        int remaining = (WtcpHeader.SIZE + header.getPayloadLength() - bufferPosition);
        //WtcLog.info(TAG, "bytes remaining=" + remaining);

        return remaining;
    }
}
//
// END WtcStackConnectionManager
//
