package com.twistpair.wave.thinclient;

import android.app.Activity;
import android.app.ActivityManager;
import android.app.ActivityManager.RunningServiceInfo;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.util.WtcUtilsPlatform;

public class WtcServiceConnection implements ServiceConnection
{
    private static final String TAG = WtcLog.TAG(WtcServiceConnection.class);

    private final Context       mContext;
    private final Class<?>      mClassService;
    private final Messenger     mMessengerActivity;

    private Messenger           mMessengerService;
    private boolean             mIsBound;
    private boolean             mIsStarted;

    private Message             mMessageOnServiceConnected;

    /**
     * @param context Context of the activity
     * @param classService Class of the Service to connect to
     * @param handler Optional Messenger to RX messages back from the Service
     */
    public WtcServiceConnection(Context context, Class<?> classService, WtcWeakReferenceHandler<?> handler)
    {
        mContext = context;
        mClassService = classService;
        mMessengerActivity = new Messenger(handler);
    }

    public Context getContext()
    {
        return mContext;
    }

    public Class<?> getClassService()
    {
        return mClassService;
    }

    public Messenger getMessengerActivity()
    {
        return mMessengerActivity;
    }

    public Messenger getMessengerService()
    {
        return mMessengerService;
    }

    public synchronized boolean isBound()
    {
        return mIsBound;
    }

    public boolean isRunning()
    {
        return isServiceRunning(mContext, mClassService);
    }

    /**
     * Queries the system to see if the classService (passed in to the constructor) is running
     * @return true if the service is running, otherwise false
     */
    public static boolean isServiceRunning(Context context, Class<?> classService)
    {
        String classServiceCanonicalName = classService.getCanonicalName();
        ActivityManager manager = (ActivityManager) context.getSystemService(Activity.ACTIVITY_SERVICE);
        String tempServiceCanonicalName;
        for (RunningServiceInfo rsi : manager.getRunningServices(Integer.MAX_VALUE))
        {
            tempServiceCanonicalName = rsi.service.getClassName();
            if (tempServiceCanonicalName.equals(classServiceCanonicalName))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @param extras
     * @return true if the service is being started or is already running, else false (because the service does not exist)
     */
    public boolean start(Bundle extras)
    {
        Intent intent = new Intent(mContext, mClassService);
        if (extras != null)
        {
            intent.putExtras(extras);
        }
        ComponentName componentName = mContext.startService(intent);
        mIsStarted = (componentName != null);
        return mIsStarted;
    }

    public boolean bind(Bundle extras, Message onServiceConnected)
    {
        mMessageOnServiceConnected = onServiceConnected;
        Intent intent = new Intent(mContext, mClassService);
        if (extras != null)
        {
            intent.putExtras(extras);
        }
        return mContext.bindService(intent, this, Context.BIND_AUTO_CREATE);
    }

    public synchronized void unbind(Message message)
    {
        if (mMessengerService != null && message != null)
        {
            if (mMessengerActivity != null)
            {
                message.replyTo = mMessengerActivity;
            }

            try
            {
                mMessengerService.send(message);
            }
            catch (RemoteException e)
            {
                // There is nothing special we need to do if the service is not running; ignore
                WtcLog.warn(TAG, "EXCEPTION: unbind(...)", e);
            }
        }

        try
        {
            mContext.unbindService(this);
        }
        catch (IllegalArgumentException e)
        {
            // Ignore "Service not registered: ..."
            WtcLog.warn(TAG, "EXCEPTION: unbind(...)", e);
        }
        mIsBound = false;
    }

    public void stop(Message message)
    {
        if (mIsBound)
        {
            unbind(message);
        }

        Intent intent = new Intent(mContext, mClassService);
        boolean wasStopped = mContext.stopService(intent);
        mIsStarted = false;
    }

    public synchronized void onServiceConnected(ComponentName name, IBinder service)
    {
        WtcLog.debug(TAG, "+onServiceConnected(\"" + name + "\", " + service + ")");

        WtcUtilsPlatform.toastShort(mContext, name.getShortClassName() + " Connected");

        if (name.getClassName().equals(mClassService.getCanonicalName()))
        {
            // TODO:(pv) try/catch if Service binder doesn't implement ...?
            mMessengerService = new Messenger(service);
            mIsBound = true;

            if (mMessageOnServiceConnected != null)
            {
                try
                {
                    mMessageOnServiceConnected.replyTo = mMessengerActivity;
                    mMessengerService.send(mMessageOnServiceConnected);
                }
                catch (RemoteException e)
                {
                    // In this case the service has ended before we could even do anything with it;
                    // e can count on soon being disconnected (and then reconnected if it can be
                    // restarted), so there is no need to do anything here.
                    WtcLog.warn(TAG, "Unable to register client to service.", e);
                }
            }
        }

        WtcLog.debug(TAG, "-onServiceConnected(\"" + name + "\", " + service + ")");
    }

    public synchronized void onServiceDisconnected(ComponentName name)
    {
        WtcLog.debug(TAG, "+onServiceDisconnected(\"" + name + "\")");
        WtcUtilsPlatform.toastShort(mContext, name.getShortClassName() + " Disconnected");
        mMessengerService = null;
        mIsBound = false;
        WtcLog.debug(TAG, "-onServiceDisconnected(\"" + name + "\")");
    }

    public void send(Message message)
    {
        WtcLog.debug(TAG, "+send(" + message + ")");

        if (mMessengerService != null)
        {
            try
            {
                mMessengerService.send(message);
            }
            catch (RemoteException e)
            {
                // ignore?
                WtcLog.warn(TAG, "could not send message " + message, e);
            }
        }

        WtcLog.debug(TAG, "-send(" + message + ")");
    }
}
