package com.twistpair.wave.thinclient;

import java.util.Hashtable;
import java.util.Timer;
import java.util.TimerTask;

import com.twistpair.wave.thinclient.WtcPingRequestRxTimeout.IPingRequestRxTimeoutListener;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackMessageReceiveResponseUnexpectedException;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpCode;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.WtcpMessage;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.util.IntegerPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

/**
 * <p>Will fire WtcStackListener.onMessageResponseTimeout and initiate a disconnect
 * if *ANY* opCode Requests or KEX message does not receive a Response/Error
 * in a specified amount of time.</p>
 * <p>Filters out Control Responses to all but the most recent Control Request, based on the transactionId.</p>
 * That is: Old Requests are ignored if a new Request is made before/while a Response to the old Request is received.
 * <p>Example: User repeatedly quickly presses the PTT button making new PushToTalk Requests before/while old Responses are received.</p>
 */
public class WtcMessageFilter
{
    private static final String  TAG                      = WtcLog.TAG(WtcMessageFilter.class);

    private static final boolean REQUEST_TIMEOUT_DISABLED = false;
    private static final boolean VERBOSE_LOG              = false;

    public interface IRequestTxResponseRxTimeoutListener
    {
        /**
         * @param timeoutMs
         * @param elapsedMs milliseconds since the offending TimerTask was *created*.
         * @param messageType
         * @param opCode
         * @param transactionId
         */
        void onRequestTxResponseRxTimeout(long timeoutMs, long elapsedMs, byte messageType, int opCode, int transactionId);
    }

    private class RequestTxResponseRxTimeoutTask extends TimerTask
    {
        private final String TAG = WtcLog.TAG(RequestTxResponseRxTimeoutTask.class);

        public final long    timeCreated;
        public final int     timeoutMs;
        public final byte    messageType;
        public final int     opCode;
        public final int     transactionId;

        public RequestTxResponseRxTimeoutTask(int timeoutMs, byte messageType, int opCode, int transactionId)
        {
            this.timeCreated = System.currentTimeMillis();
            this.timeoutMs = timeoutMs;
            this.messageType = messageType;
            this.opCode = opCode;
            this.transactionId = transactionId;
        }

        public String toString()
        {
            StringBuffer sb = new StringBuffer();
            sb.append("timeCreated=").append(timeCreated) //
            .append(", timeoutMs=").append(timeoutMs) //
            .append(", messageType=").append(WtcpMessageType.toString(messageType));
            if (messageType == WtcpMessageType.Control)
            {
                sb.append('{') //
                .append("opCode=").append(WtcpOpCode.toString(opCode)) //
                .append(", transactionId=0x").append(WtcString.toHexString(transactionId, 2)) //
                .append('}');
            }
            return sb.toString();
        }

        /**
         * @return milliseconds since this timeout task was created
         */
        public int getElapsedMs()
        {
            return (int) (System.currentTimeMillis() - timeCreated);
        }

        public void run()
        {
            try
            {
                WtcLog.info(TAG, "+run()");

                int elapsedMs = getElapsedMs();

                // Lock here to prevent event from firing during call to WtcMessageFilter.close() in ThreadConnection shutdown
                synchronized (WtcMessageFilter.this)
                {
                    if (timer == null)
                    {
                        return;
                    }

                    listenerRequestTxResponseRxTimeout.onRequestTxResponseRxTimeout(timeoutMs, elapsedMs, //
                                    messageType, opCode, transactionId);
                }
            }
            catch (Exception e)
            {
                WtcLog.error(TAG, "run()", e);
            }
            finally
            {
                WtcLog.info(TAG, "-run()");
            }
        }
    }

    /**
     * Hashtable&lt;Integer opCode, Integer transactionId&gt;
     */
    private final Hashtable                           latestOpCodeRequestTransactionId;

    /**
    * Hashtable&lt;Integer transactionId, RequestTxResponseRxTimeoutTask task&gt;
    */
    private final Hashtable                           tasksControl;

    private final IRequestTxResponseRxTimeoutListener listenerRequestTxResponseRxTimeout;

    private final WtcConnectionStatistics             connectionStatistics;

    /**
    * Common to both KEX and Control messages.<br>
    * Defaults to DEFAULT_MESSAGE_TIMEOUT.<br>
    * Will be updated after each Response or Error RXed.
    */
    private int                                       requestTxResponseRxTimeoutMs =
                                                                                       WtcStack.TIMEOUT_REQUEST_TX_RESPONSE_RX_MS_DEFAULT;

    /**
     * The one and only allowed pending KEX message RequestTxResponseRxTimeoutTask.
     */
    private RequestTxResponseRxTimeoutTask            taskKex;

    private Timer                                     timer;

    private WtcPingRequestRxTimeout                   timeoutPingRequestRx;

    public WtcMessageFilter(IRequestTxResponseRxTimeoutListener listenerRequestTxResponseRxTimeout, //
                    IPingRequestRxTimeoutListener listenerPingRequestRxTimeout, //
                    WtcConnectionStatistics connectionStatistics)
    {
        this.listenerRequestTxResponseRxTimeout = listenerRequestTxResponseRxTimeout;
        this.connectionStatistics = connectionStatistics;

        this.latestOpCodeRequestTransactionId = new Hashtable();
        this.tasksControl = new Hashtable();

        timer = new Timer();

        timeoutPingRequestRx = new WtcPingRequestRxTimeout(listenerPingRequestRxTimeout, connectionStatistics);
        timeoutPingRequestRx.reset(false);
    }

    public void close()
    {
        try
        {
            WtcLog.info(TAG, "+close()");

            if (timeoutPingRequestRx != null)
            {
                WtcLog.info(TAG, "timeoutPingRequestRx.close()");
                timeoutPingRequestRx.close();
                timeoutPingRequestRx = null;
            }

            // Lock here to prevent event firing during call to MessageRequestResponseTimeout.close() in ThreadConnection shutdown
            synchronized (this)
            {
                if (timer != null)
                {
                    timer.cancel();
                    timer = null;
                }

                taskKex = null;

                synchronized (tasksControl)
                {
                    tasksControl.clear();
                }

                synchronized (latestOpCodeRequestTransactionId)
                {
                    latestOpCodeRequestTransactionId.clear();
                }
            }
        }
        finally
        {
            WtcLog.info(TAG, "-close()");
        }
    }

    // TODO:(pv) allocation the RequestTxResponseRxTimeoutTasks from an instance pool
    public void start(WtcpMessage message)
    {
        // Save this member property here since another thread could change it while we are in this method.
        final int requestTxResponseRxTimeoutMs = this.requestTxResponseRxTimeoutMs;

        final RequestTxResponseRxTimeoutTask task;

        final byte messageType = message.getMessageType();
        switch (messageType)
        {
            case WtcpMessageType.KeyExchange:
                synchronized (this)
                {
                    if (taskKex != null)
                    {
                        throw new IllegalStateException("KEX message already in progress");
                    }

                    task = new RequestTxResponseRxTimeoutTask(requestTxResponseRxTimeoutMs, messageType, -1, -1);

                    taskKex = task;
                }
                break;

            case WtcpMessageType.Control:
                final WtcpControlHeader controlHeader = (WtcpControlHeader) message.getSubHeader();
                final int opType = controlHeader.verOpCode.getOpType();
                if (opType == WtcpOpType.Request)
                {
                    final Integer opCode = IntegerPlatform.valueOf(controlHeader.verOpCode.getOpCode());
                    final Integer transactionId = IntegerPlatform.valueOf(controlHeader.transactionId);

                    synchronized (latestOpCodeRequestTransactionId)
                    {
                        latestOpCodeRequestTransactionId.put(opCode, transactionId);
                    }

                    task = new RequestTxResponseRxTimeoutTask(requestTxResponseRxTimeoutMs, messageType, //
                                    opCode.intValue(), transactionId.intValue());

                    synchronized (tasksControl)
                    {
                        tasksControl.put(transactionId, task);
                    }
                }
                else
                {
                    task = null;
                }
                break;

            default:
                task = null;
                break;
        }

        if (task != null)
        {
            synchronized (this)
            {
                if (timer != null && !REQUEST_TIMEOUT_DISABLED)
                {
                    if (VERBOSE_LOG)
                    {
                        WtcLog.error(TAG, "$TIMEOUT: Waiting " + requestTxResponseRxTimeoutMs
                                        + "ms to RX Response/Error to Request TX task=" + task);
                    }
                    timer.schedule(task, requestTxResponseRxTimeoutMs);
                }
            }
        }
    }

    public boolean cancel(WtcpMessage message) throws WtcStackMessageReceiveResponseUnexpectedException
    {
        boolean shouldIgnore = false;

        final RequestTxResponseRxTimeoutTask task;

        final byte messageType = message.getMessageType();
        switch (messageType)
        {
            case WtcpMessageType.KeyExchange:
            {
                if (VERBOSE_LOG)
                {
                    WtcLog.error(TAG, "$TIMING: RXed " + WtcpMessageType.toString(messageType));
                }

                synchronized (this)
                {
                    task = taskKex;
                    taskKex = null;
                }

                break;
            }

            case WtcpMessageType.Control:
            {
                final WtcpControlHeader controlHeader = (WtcpControlHeader) message.getSubHeader();

                final int opType = controlHeader.verOpCode.getOpType();
                final int opCode = controlHeader.verOpCode.getOpCode();

                switch (opType)
                {
                    case WtcpOpType.Error:
                    case WtcpOpType.Response:
                    {
                        // We only cancel tracking Control Requests upon receiving a *Response or Error*

                        final int transactionId = controlHeader.transactionId;

                        synchronized (tasksControl)
                        {
                            task = (RequestTxResponseRxTimeoutTask) tasksControl.remove(IntegerPlatform.valueOf(transactionId));
                        }

                        boolean handleMessage = opType == WtcpOpType.Response;

                        // Call PTT errors result in an ERROR opType, we need to handle those for Call PTT gating
                        handleMessage |=
                            (opType == WtcpOpType.Error && (opCode == WtcpOpCode.CallPushToTalkOn || opCode == WtcpOpCode.CallPushToTalkOff));

                        if (handleMessage)
                        {
                            if (VERBOSE_LOG)
                            {
                                WtcLog.error(TAG,
                                                "RXed *ANY* Response; Resetting RX Ping Request timeout to the last calculated Ping interval");
                            }
                            if (timeoutPingRequestRx != null)
                            {
                                timeoutPingRequestRx.reset(false);
                            }

                            // TODO:(pv) Only filter out SOME control messages (ex: PTT, Mute, Activate, etc)
                            // We still want to allow multiples of other pending messages, such as:
                            // 05-18 18:33:22.053: I/WtcStack(27249): T27267 +sendEndpointPropertiesGet(3, ["LOC"])
                            // ...
                            // 05-18 18:33:22.280: I/WtcStack(27249): T27267 +sendEndpointPropertiesGet(5, ["LOC"])
                            // ...

                            Integer latestTransactionId;
                            synchronized (latestOpCodeRequestTransactionId)
                            {
                                Integer opCodeValue = IntegerPlatform.valueOf(opCode);

                                // Don't remove...
                                // Channel PTT uses one op code, if you send an ON and OFF before receiving a response to the ON
                                // the OFF will come in and there will be a null transaction ID causing an exception 
                                latestTransactionId = (Integer) latestOpCodeRequestTransactionId.get(opCodeValue);

                                if (VERBOSE_LOG)
                                {
                                    WtcLog.info(TAG, "Removing opCode:" + opCode
                                                    + " from latestOpCodeRequestTransactionId; transactionId="
                                                    + latestTransactionId);
                                }

                                // If this is a response to a PTT on request we need to check if there is a PTT off request in transit
                                if (opCode == WtcpOpCode.CallPushToTalkOn)
                                {
                                    // Check if there's a PTT off request, but don't remove it because if there was one sent we need to handle the response
                                    Integer latestCallPushToTalkOffTransactionId =
                                        (Integer) latestOpCodeRequestTransactionId.get(IntegerPlatform.valueOf(WtcpOpCode.CallPushToTalkOff));

                                    // If there is a PTT off request in transit we want to ignore the PTT on response
                                    if (latestCallPushToTalkOffTransactionId != null)
                                    {
                                        WtcLog.info(TAG, "Call PTT OFF is in transit; ignoring Call PTT ON response");
                                        latestTransactionId = latestCallPushToTalkOffTransactionId;
                                    }
                                }
                                else if (opCode == WtcpOpCode.CallPushToTalkOff)
                                {
                                    latestOpCodeRequestTransactionId.remove(opCodeValue);
                                }
                            }

                            if (latestTransactionId == null)
                            {
                                WtcLog.error(TAG,
                                                "UNEXPECTED: RXed Response for non-tracked Request; opCode="
                                                                + WtcpOpCode.toString(opCode) + "; transactionId="
                                                                + transactionId);
                                throw new WtcStackMessageReceiveResponseUnexpectedException(opCode, transactionId);
                            }

                            //
                            // In general, we want to filter out any Response that is not relevant to the latest Request.
                            // That is, we want to only let through the filter the Response relevant to the latest Request.
                            // The problem is that we DON'T want to do this on some Requests such as "EndpointPropertiesGet" or "ChannelPropertiesGet".
                            // It is perfectly reasonable to make dozens of such Requests very quickly and expect to get back Responses for all of them. 
                            // So, we have to pick and choose which stale WtcpOpCodes we filter out, and which ones we let through.
                            // If a Request of the same OpCode can have conflicting payload contents then it is a candidate to add to the following filter:   
                            switch (opCode)
                            {
                                case WtcpOpCode.ChannelSetActive:
                                case WtcpOpCode.ChannelMute:
                                case WtcpOpCode.ChannelPushToTalk:
                                case WtcpOpCode.CallMake:
                                case WtcpOpCode.CallHangup:
                                case WtcpOpCode.CallPushToTalkOn:
                                case WtcpOpCode.CallPushToTalkOff:
                                    //
                                    // Filter out only the latest Response
                                    //
                                    // NOTE: There can still be a high-level-logic race condition that a higher transactionId is TXed after RXing this transactionId.
                                    // The high-level-logic will need to do its own filter.
                                    // Example:
                                    //      1. Thread1: TX PTT off Request transactionId=0x54
                                    //      2. Thread2: RX PTT off Response transactionId=0x54
                                    //      3. Thread1: TX PTT on Request transactionId=0x55
                                    //      4. Thread2: Process RX PTT off transactionId=0x54
                                    // Step #4 would need to ignore processing transactionId=0x54
                                    //
                                    // This can be eliminated if all TX and RX operations are funneled through a single Thread (which WtcStackConnectionManager does)
                                    //
                                    if (VERBOSE_LOG)
                                    {
                                        WtcLog.info(TAG, "shouldIgnore=(" + latestTransactionId.intValue() + " != "
                                                        + transactionId + ")");
                                    }
                                    shouldIgnore = (latestTransactionId.intValue() != transactionId);
                                    break;

                                default:
                                    shouldIgnore = false;
                            }

                            if (VERBOSE_LOG)
                            {
                                StringBuffer sb = new StringBuffer("RXed Response for");
                                if (shouldIgnore)
                                {
                                    sb.append(" old transactionId=0x").append(WtcString.toHexString(transactionId, 2)) //
                                    .append(", latest transactionId=0x").append(
                                                    WtcString.toHexString(latestTransactionId.intValue(), 2)) //
                                    .append("; **IGNORING**");
                                }
                                else
                                {
                                    sb.append(" transactionId=0x").append(WtcString.toHexString(transactionId, 2)) //
                                    .append("; *NOT* ignoring");
                                }
                                sb.append(" Request ").append(WtcpOpCode.toString(opCode));
                                WtcLog.error(TAG, sb.toString());
                            }
                        }

                        if (VERBOSE_LOG)
                        {
                            WtcLog.error(TAG, "$TIMING: RXed " //
                                            + WtcpMessageType.toString(messageType) //
                                            + " " + WtcpOpType.toString(opType) //
                                            + " " + WtcpOpCode.toString(opCode) //
                                            + " transactionId=0x" + WtcString.toHexString(transactionId, 2) //
                                            + "; canceling timeout");
                        }
                        break;
                    }

                    case WtcpOpType.Request:
                        //
                        // It is *possible* that the Client and Server will *both* TX a PING *Request* at almost the exact same time:
                        //  08-22 08:35:01.593 D/WtcPingRequestRxTimeout(  688): T1007 +taskPing.run()
                        //  08-22 08:35:01.597 W/WtcPingRequestRxTimeout(  688): T1007 $HEALTH: Did not RX PING Request in 125976ms
                        //  08-22 08:35:01.597 W/WtcPingRequestRxTimeout(  688): T1007 $HEALTH: TXing PING Request; expecting RX PING Response or onMessageResponseTimeout.
                        //  08-22 08:35:01.597 I/WtcStack(  688): T1007 +sendPing("Request"(1), 2)
                        //  *08-22 08:35:01.605 D/WtcStack(  688): T1008 TX 14b: {h={v=1,t="Control"(0x04),s=0x001D,i=0x000A},c={c=0x91F0,s=0x001D,o={v=1,t="Request"(0x01),c="Ping"(0x0002)},t=0x001B},p(2)=\0\2}
                        //  08-22 08:35:01.609 D/WtcStack(  688): T1007 -sendPing("Request"(1), 2)
                        //  08-22 08:35:01.609 D/WtcPingRequestRxTimeout(  688): T1007 -taskPing.run()
                        //  *08-22 08:35:01.785 D/WtcStack(  688): T1009 RX 14b: {h={v=1,t="Control"(0x04),s=0x006A,i=0x000A},c={c=0x843A,s=0x0059,o={v=1,t="Request"(0x01),c="Ping"(0x0002)},t=0x0000},p(2)=\0\3}
                        //  08-22 08:35:01.785 D/WtcStack(  688): T1009 +processPing
                        //  08-22 08:35:01.789 I/WtcStack(  688): T1009 RX Ping Request id=3; TX Ping Response id=3
                        //  08-22 08:35:01.789 I/WtcStack(  688): T1009 +sendPing("Response"(2), 3)
                        //  08-22 08:35:01.796 D/WtcStack(  688): T1008 TX 14b: {h={v=1,t="Control"(0x04),s=0x001E,i=0x000A},c={c=0xB5D0,s=0x001E,o={v=1,t="Response"(0x02),c="Ping"(0x0002)},t=0x0000},p(2)=\0\3}
                        //  08-22 08:35:01.800 D/WtcStack(  688): T1009 -sendPing("Response"(2), 3)
                        //  ...
                        //  08-22 08:35:01.816 D/WtcStack(  688): T1009 -processPing
                        //
                        // When this happens the Server does *NOT* Respond to the Client's PING Request.
                        // If we let the Client PING Request timeout (typically <10 seconds) then the Client will disconnect thinking the connection is bad.
                        // The fact that we got a PING Request from the Server is proof enough that the connection is not bad.
                        // So, we will need to reset the timer of any pending TXed PingRequest.
                        //
                        if (VERBOSE_LOG)
                        {
                            WtcLog.error(TAG,
                                            "RXed *ANY* Request; Resetting RX Ping Request timeout and calculating new Ping interval");
                        }

                        if (timeoutPingRequestRx != null)
                        {
                            timeoutPingRequestRx.reset(true);
                        }
                        task = null;
                        break;

                    default:
                        task = null;
                        break;
                }

                break;
            }

            case WtcpMessageType.Media:
            {
                //final WtcpMediaHeader mediaHeader = (WtcpMediaHeader) message.getSubHeader();

                //connectionStatistics.jitter.add2(latencyMs);

                //filterMediaMessages.add(mediaHeader);

                task = null;
                shouldIgnore = false; // TODO:(pv) ignore media message if jitter buffer says it is too old
                break;
            }

            default:
            {
                task = null;
                break;
            }
        }

        if (task != null)
        {
            if (VERBOSE_LOG)
            {
                WtcLog.warn(TAG, "$TIMING: task.cancel()");
            }
            task.cancel();

            int latencyMs = task.getElapsedMs();
            if (VERBOSE_LOG)
            {
                WtcLog.error(TAG, "$LATENCY: Response took " + latencyMs + "ms");
            }

            //
            // Special more visible logging of latency for important message types
            //
            switch (task.messageType)
            {
                case WtcpMessageType.KeyExchange:
                    WtcLog.warn(TAG, "$LATENCY: KEX = " + latencyMs + "ms");
                    break;

                case WtcpMessageType.Control:
                    switch (task.opCode)
                    {
                        case WtcpConstants.WtcpOpCode.ChannelPushToTalk:
                            WtcLog.warn(TAG, "$LATENCY: PTT = " + latencyMs + "ms");
                            break;
                    }
                    break;
            }

            requestTxResponseRxTimeoutMs = connectionStatistics.latency.add(latencyMs);
        }

        return shouldIgnore;
    }

    public class WtcpMediaMessageFilter
    {
        // TODO:(pv) Calculate jitter buffer and possibly ignore
        public void add(WtcpMediaHeader mediaHeader)
        {

        }
    }
}
