package com.twistpair.wave.thinclient;

import java.util.Enumeration;
import java.util.Hashtable;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpKeyValueList;
import com.twistpair.wave.thinclient.util.StereoVolume;
import com.twistpair.wave.thinclient.util.WtcInt16;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcClientChannel
{
    private static final String            TAG                          = WtcLog.TAG(WtcClientChannel.class);

    private static final boolean           VERBOSE_LOG_CHANNEL_ACTIVITY = false;

    // private: prefixed with '@' - read/write
    //  Known:
    //      "@V" = volume: "0.0" to "1.0"
    // system: prefixed with '%' - read-only (cannot be set)
    //  Known:
    //      "%T" = can transmit: "0" or "1"
    //      "%R" = can receive: "0" or "1"
    // custom: prefixed with "%custom." - read-only (cannot be set)
    // shared: not yet implemented on server
    private static final char              PROPERTY_KEY_PREFIX_SYSTEM   = '%';
    private static final char              PROPERTY_KEY_PREFIX_PRIVATE  = '@';

    private static final char              PROPERTY_KEY_SUFFIX_NAME     = 'N';
    private static final char              PROPERTY_KEY_SUFFIX_UNIQUE   = 'U';
    private static final char              PROPERTY_KEY_SUFFIX_SIMPLEX  = 'S';
    private static final char              PROPERTY_KEY_SUFFIX_RXABLE   = 'R';
    private static final char              PROPERTY_KEY_SUFFIX_TXABLE   = 'T';
    private static final char              PROPERTY_KEY_SUFFIX_VOLUME   = 'V';

    public static final String             PROPERTY_KEY_SYSTEM_UNIQUE   = "" + PROPERTY_KEY_PREFIX_SYSTEM
                                                                                        + PROPERTY_KEY_SUFFIX_UNIQUE;
    public static final String             PROPERTY_KEY_PRIVATE_VOLUME  = "" + PROPERTY_KEY_PREFIX_PRIVATE
                                                                                        + PROPERTY_KEY_SUFFIX_VOLUME;

    private final WtcClientChannelManager  mManager;

    private final WtcInt32                 mIdSession;

    /**
     * Any combo of WtcpChannelFlags.
     * Non-final because Channel flags can change dynamically.
     */
    private WtcInt16                       mFlags;

    /**
     * Initially set at log in, but can be updated later by a "%N" channel property update.
     */
    private String                         mName;

    protected final WtcClientChannelAction mActive                      = new WtcClientChannelAction();
    protected final WtcClientChannelAction mPtt                         = new WtcClientChannelAction();
    protected final WtcClientChannelAction mMute                        = new WtcClientChannelAction();

    private final WtcpKeyValueList         mProperties                  = new WtcpKeyValueList();

    /**
     * Unique ID: "%U"="..."
     * Initially set at log in to the channel name, but later updated by a "%U" channel property update.
     */
    private String                         mIdUnique;
    private boolean                        mGotUnique                   = false;

    /**
     * Simplex: "%S"="0|1"
     * true means that only one person can talk on the channel at a time.
     */
    private boolean                        mSimplex                     = false;

    /**
     * Receivable: "%R"="0|1"
     * false effectively means that the channel is remotely muted.
     */
    private boolean                        mRXable                      = true;

    /**
     * Transmitable: "%T"="0|1"
     * false effectively means that the channel is listen-only.
     */
    private boolean                        mTXable                      = true;

    /**
     * Volume(Left,Right): "@V"="0-100,0-100"
     */
    private final StereoVolume             mVolume                      = new StereoVolume();

    //
    // ChannelActivity
    //
    private int                            mEndpointCount;

    private final Object                   mActivitySync                = new Object();

    /**
     * Hashtable&lt;String(EndpointId), WtcpEndpointInfo&gt;
     */
    private final Hashtable                mEndpointsVisible            = new Hashtable();

    /**
     * Hashtable&lt;String(EndpointId), WtcpEndpointInfo&gt;
     */
    private final Hashtable                mEndpointsTalking            = new Hashtable();

    public WtcClientChannel(WtcClientChannelManager manager, WtcpChannelInfo channelInfo)
    {
        mManager = manager;
        mIdSession = WtcInt32.valueOf(channelInfo.id, true);
        setFlags(channelInfo.flags);
        mName = channelInfo.name;

        // Temporarily use the *name* as a semi-permanent value until we get an actual "%U" property
        mIdUnique = mName;
    }

    //@Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer() //
        .append("{idSession=").append(mIdSession) //
        .append(",flags=").append(mFlags) //
        .append(",name=").append('\"').append(mName).append('\"'); //
        if (mActive.isAnyTrue())
        {
            sb.append(",active=").append(mActive); //
        }
        if (mPtt.isAnyTrue())
        {
            sb.append(",ptt=").append(mPtt); //
        }
        if (mMute.isAnyTrue())
        {
            sb.append(",mute=").append(mMute); //
        }
        sb.append(",unique=").append(mGotUnique) //
        .append(",idUnique=").append('\"').append(mIdUnique).append('\"') //
        .append(",simplex=").append(mSimplex) //
        .append(",RXable=").append(mRXable) //
        .append(",TXable=").append(mTXable) //
        .append(",volume=").append(mVolume) //
        .append(",properties=").append(mProperties) //
        .append('}');
        return sb.toString();
    }

    //@Override
    public boolean equals(Object o)
    {
        if (o != null && o instanceof WtcClientChannel)
        {
            return getIdSession().value == ((WtcClientChannel) o).getIdSession().value;
        }
        return false;
    }

    public WtcInt32 getIdSession()
    {
        return mIdSession;
    }

    public boolean isUnique()
    {
        return mGotUnique;
    }

    public String getIdUnique()
    {
        return mIdUnique;
    }

    public int getFlags()
    {
        return mFlags.value;
    }

    private void setFlags(int flags)
    {
        mFlags = new WtcInt16(flags);
    }

    public String getName()
    {
        return mName;
    }

    public boolean isActivated()
    {
        return mActive.isOn();
    }

    public boolean isSimplex()
    {
        return mSimplex;
    }

    public boolean getRXable()
    {
        return mRXable;
    }

    public boolean getTXable()
    {
        return mTXable;
    }

    public StereoVolume getVolume()
    {
        return mVolume;
    }

    /**
     * @param left 0 to 100
     * @param right 0 to 100
     * @return transactionId or null 
     */
    public Integer setVolume(int left, int right)
    {
        return setVolume(new StereoVolume(left, right));
    }

    /**
     * @param volume null to set both left=0 and right=0
     * @return transactionId or null 
     */
    public Integer setVolume(StereoVolume volume)
    {
        if (volume == null)
        {
            volume = new StereoVolume(0, 0);
        }
        String value = volume.getLeft() + "," + volume.getRight();
        return mManager.setChannelProperty(mIdSession, PROPERTY_KEY_PRIVATE_VOLUME, value);
    }

    /**
     * *BEFORE CALLING THIS METHOD TO DEACTIVATE A CHANNEL*, if the user is talking then the the caller is responsible for turning off the microphone and ideally waiting for the current capture buffer to complete.
     * Failure to do so can result in failure to transmit the final captured audio buffer(s), cutting off their content (speech). 
     * After calling this method to activate a channel, the caller should wait for the onChannelActivated event and may then call mute/talk as desired.
     * @param on
     * @return transactionId or null 
     */
    public Integer activate(boolean on)
    {
        return mManager.activate(mIdSession, on);
    }

    public boolean isTalking()
    {
        return mPtt.isOn();
    }

    /**
     * *BEFORE CALLING THIS METHOD TO *STOP* TALKING*, the caller is responsible for turning off the microphone and ideally waiting for the current capture buffer to complete.
     * After calling this method TO TALK on a channel, the caller should wait for the onChannelTalking event and *ONLY THEN* open the microphone.
     * The caller should *NOT* open the microphone before calling this method to *START* talking, for several reasons:
     * 1) The talk request can be denied by the server, in which case the mic will just have to be closed again
     * 2) The user could be mumbling something they they don't intend to broadcast until  
     * @param on
     */
    public Integer talk(boolean on)
    {
        return mManager.talk(mIdSession, on);
    }

    public boolean isMuted()
    {
        return mMute.isOn();
    }

    public Integer mute(boolean on)
    {
        return mManager.mute(mIdSession, on);
    }

    protected void updateProperties(WtcpKeyValueList keyValues)
    {
        // Frequently called method, so use logging sparingly
        //WtcLog.info(TAG, "updateProperties(" + keyValues + ")");

        String key;
        String value;
        int offset;

        Enumeration keys = keyValues.keys();
        while (keys.hasMoreElements())
        {
            key = (String) keys.nextElement();
            value = (String) keyValues.get(key);
            //WtcLog.error(TAG, "updateProperties: key=\"" + key + "\", value=\"" + value + "\"");

            mProperties.put(key, value);

            offset = 0;
            switch (key.charAt(offset++))
            {
                case PROPERTY_KEY_PREFIX_SYSTEM:
                    switch (key.charAt(offset++))
                    {
                        case PROPERTY_KEY_SUFFIX_NAME:
                            mName = value;
                            break;
                        case PROPERTY_KEY_SUFFIX_UNIQUE:
                            mIdUnique = value;
                            mGotUnique = true;
                            break;
                        case PROPERTY_KEY_SUFFIX_SIMPLEX:
                            mSimplex = !"0".equals(value);
                            break;
                        case PROPERTY_KEY_SUFFIX_RXABLE:
                            mRXable = !"0".equals(value);
                            break;
                        case PROPERTY_KEY_SUFFIX_TXABLE:
                            mTXable = !"0".equals(value);
                            break;
                    }
                    break;

                case PROPERTY_KEY_PREFIX_PRIVATE:
                    switch (key.charAt(offset++))
                    {
                        case PROPERTY_KEY_SUFFIX_VOLUME:
                            String[] volumeLeftRight = WtcString.split(value, ",", 2);
                            int left = Integer.parseInt(volumeLeftRight[0]);
                            int right = Integer.parseInt(volumeLeftRight[1]);
                            mVolume.set(left, right);
                            break;
                    }
                    break;
            }
        }
    }

    protected void updateActivity(int channelFlags, short endpointCount, WtcpEndpointInfoList activityEndpoints)
    {
        // Frequently called method, so use logging sparingly
        //WtcLog.info(TAG, "updateActivity(channelFlags, endpointCount, activityEndpoints)");
        setEndpointCount(endpointCount);
        setActivityEndpoints(activityEndpoints);
    }

    public int getEndpointCount()
    {
        return mEndpointCount;
    }

    private void setEndpointCount(int endpointCount)
    {
        mEndpointCount = endpointCount;
    }

    /**
     * @return a copy of Map&lt;String(EndpointId),WtcpEndpointInfo&gt;<b><i>.values()</i></b>
     */
    public WtcpEndpointInfoList getEndpointsVisible()
    {
        synchronized (mActivitySync)
        {
            return new WtcpEndpointInfoList(mEndpointsVisible.elements());
        }
    }

    /**
     * @return a copy of Map&lt;String(EndpointId),WtcpEndpointInfo&gt;<b><i>.values()</i></b>
     */
    public WtcpEndpointInfoList getEndpointsTalking()
    {
        synchronized (mActivitySync)
        {
            return new WtcpEndpointInfoList(mEndpointsTalking.elements());
        }
    }

    public boolean getHasTalkers()
    {
        synchronized (mActivitySync)
        {
            return mEndpointsTalking.size() > 0;
        }
    }

    private void setActivityEndpoints(WtcpEndpointInfoList activityEndpoints)
    {
        if (VERBOSE_LOG_CHANNEL_ACTIVITY)
        {
            // TODO:(pv) Remove the below WtcLog.warn(...) lines after we resolve some reported channel activity bugs
            WtcLog.warn(TAG, "setActivityEndpoints(activityEndpoints=" + activityEndpoints + ")");

            WtcLog.warn(TAG, "setActivityEndpoints: channel mIdSession=" + mIdSession + ", mName=\"" + mName + "\"");

            //WtcLog.warn(TAG, "setActivityEndpoints: BEFORE mEndpointsVisible=" + mEndpointsVisible);
            WtcLog.warn(TAG, "setActivityEndpoints: BEFORE mEndpointsTalking=" + mEndpointsTalking);
        }

        WtcpEndpointInfo endpointInfo;
        String endpointId;
        for (int i = 0; i < activityEndpoints.size(); i++)
        {
            endpointInfo = (WtcpEndpointInfo) activityEndpoints.elementAt(i);
            endpointId = endpointInfo.getId();

            synchronized (mActivitySync)
            {
                if (endpointInfo.isVisible())
                {
                    //WtcLog.warn(TAG, "setActivityEndpoints: endpointId=" + endpointId + " visible; adding to mEndpointsVisible");
                    mEndpointsVisible.put(endpointId, endpointInfo);

                    //
                    // Only visible endpoints can also be added to mEndpointsTalking
                    //
                    if (endpointInfo.isSpeaking())
                    {
                        if (VERBOSE_LOG_CHANNEL_ACTIVITY)
                        {
                            WtcLog.warn(TAG, "setActivityEndpoints: endpointId=" + endpointId
                                            + " talking; adding to mEndpointsTalking");
                        }
                        mEndpointsTalking.put(endpointId, endpointInfo);
                    }
                    else
                    {
                        if (VERBOSE_LOG_CHANNEL_ACTIVITY)
                        {
                            WtcLog.warn(TAG, "setActivityEndpoints: endpointId=" + endpointId
                                            + " not talking; removing from mEndpointsTalking");
                        }
                        mEndpointsTalking.remove(endpointId);
                    }
                }
                else
                {
                    if (VERBOSE_LOG_CHANNEL_ACTIVITY)
                    {
                        WtcLog.warn(TAG, "setActivityEndpoints: endpointId=" + endpointId
                                        + " not visible; removing from mEndpointsVisible and mEndpointsTalking");
                    }
                    mEndpointsVisible.remove(endpointId);

                    //
                    // Non-visible endpoints must be removed from mEndpointsTalking
                    //
                    mEndpointsTalking.remove(endpointId);
                }
            }
        }

        if (VERBOSE_LOG_CHANNEL_ACTIVITY)
        {
            //WtcLog.warn(TAG, "setActivityEndpoints: AFTER mEndpointsVisible=" + mEndpointsVisible);
            WtcLog.warn(TAG, "setActivityEndpoints: AFTER mEndpointsTalking=" + mEndpointsTalking);
        }
    }
}
