package com.twilio.voice;

import static android.content.pm.PackageManager.PERMISSION_GRANTED;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.URLDecoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class Utils {
    /*
     * Return a handler on the thread looper, or the main thread looper if the calling thread
     * does not have a looper. If neither are available this handler will return null.
     */
    static Handler createHandler() {
        Handler handler = null;
        Looper looper;
        if ((looper = Looper.myLooper()) != null) {
            handler = new Handler(looper);
        } else if ((looper = Looper.getMainLooper()) != null) {
            handler = new Handler(looper);
        }
        if (handler == null) {
            throw new IllegalThreadStateException("This thread must be able to obtain a Looper");
        }

        return handler;
    }

    static boolean isAudioPermissionGranted(Context context) {
        int MICROPHONE_PERMISSION =
                context.checkCallingOrSelfPermission(Manifest.permission.RECORD_AUDIO);
        return MICROPHONE_PERMISSION == PERMISSION_GRANTED;
    }

    /**
     * Get IP address from first non-localhost interface
     *
     * @param useIPv4 true=return ipv4, false=return ipv6
     * @return address or empty string
     */
    public static String getIPAddress(boolean useIPv4) {
        try {
            List<NetworkInterface> interfaces =
                    Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                List<InetAddress> addrs = Collections.list(intf.getInetAddresses());
                for (InetAddress addr : addrs) {
                    if (!addr.isLoopbackAddress()) {
                        String sAddr = addr.getHostAddress();
                        boolean isIPv4 = sAddr.indexOf(':') < 0;

                        if (useIPv4) {
                            if (isIPv4) return sAddr;
                        } else {
                            if (!isIPv4) {
                                int delim = sAddr.indexOf('%'); // drop ip6 zone suffix
                                return delim < 0
                                        ? sAddr.toUpperCase()
                                        : sAddr.substring(0, delim).toUpperCase();
                            }
                        }
                    }
                }
            }
        } catch (Exception ex) {
        } // for now eat exceptions
        return "";
    }

    static boolean permissionGranted(Context context, String permission) {
        int permissionCheck = context.checkCallingOrSelfPermission(permission);

        return permissionCheck == PackageManager.PERMISSION_GRANTED;
    }

    @Nullable
    static String parseClientIdentity(@Nullable String toField) {
        if (toField != null) {
            return toField.replaceFirst("^client:", "");
        }
        return null;
    }

    static void parseCustomParams(
            final @NonNull String queryPairs, @NonNull final Map<String, String> customParameters) {
        final String[] pairs = queryPairs.split("&");
        for (String pair : pairs) {
            final int idx = pair.indexOf("=");
            final String key;
            try {
                key = idx > 0 ? pair.substring(0, idx) : pair;
                final String value =
                        idx > 0 && pair.length() > idx + 1
                                ? URLDecoder.decode(
                                        pair.substring(idx + 1).replaceAll("\\+", "%20"), "UTF-8")
                                : null;
                customParameters.put(key, value);
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
        }
    }

    /*
     * Converts a map of key value string pairs to two separate arrays of keys and values.
     */
    static Pair<String[], String[]> mapToArrays(Map<String, String> map) {
        Pair<String[], String[]> keyValueArrays =
                new Pair<>(new String[map.size()], new String[map.size()]);

        int index = 0;
        for (Map.Entry<String, String> mapEntry : map.entrySet()) {
            keyValueArrays.first[index] = mapEntry.getKey();
            keyValueArrays.second[index] = mapEntry.getValue();
            index++;
        }

        return keyValueArrays;
    }

    /*
     * Converts a Bundle to a map.
     */
    static Map<String, String> bundleToMap(final Bundle data) {
        Map<String, String> dataMap = new HashMap<>();
        for (String key : data.keySet()) {
            dataMap.put(key, String.valueOf(data.get(key)));
        }
        return dataMap;
    }

    /*
     * Dumps the backtrace, for debugging
     */
    static void DumpBacktrace() {
        try {
            throw new RuntimeException();
        } catch (Exception e) {
            e.printStackTrace();
        }
        ;
    }
}
