package com.twilio.voice;

import android.os.Handler;
import androidx.annotation.VisibleForTesting;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONException;
import org.json.JSONObject;

class HttpsRegistrar {
    private static final String REGISTRATION_ID_LOCATION = "Location";
    @VisibleForTesting static final String JSON_MESSAGE_KEY = "message";
    @VisibleForTesting static final String JSON_CODE_KEY = "code";

    @VisibleForTesting
    static final String DEFAULT_REGISTRATION_FAILED_MESSAGE = "Registration failed";

    @VisibleForTesting
    static final String DEFAULT_UNREGISTRATION_FAILED_MESSAGE = "Unregistration failed";

    private HttpsRegistrar() {}

    public static void register(
            final String accessToken,
            final String jsonPayloadString,
            final String hostURL,
            final RegistrarListener registrarListener) {
        HttpsRegistrar.register(accessToken, jsonPayloadString, hostURL, false, registrarListener);
    }

    public static void register(
            final String accessToken,
            final String jsonPayloadString,
            final String hostURL,
            final boolean isUnregistrationRequest,
            final RegistrarListener registrarListener) {
        String requestId = SidUtil.generateGUID(Constants.TWILIO_REQUEST_SID_PREFIX, accessToken);
        final Executor executor = Executors.newCachedThreadPool();
        final Handler handler = Utils.createHandler();
        executor.execute(
                () -> {
                    HttpsURLConnection urlConnection = null;
                    try {
                        urlConnection =
                                VoiceURLConnection.create(
                                        accessToken, hostURL, VoiceURLConnection.METHOD_TYPE_POST);

                        urlConnection.addRequestProperty(
                                Constants.TWILIO_REQUEST_HEADER, requestId);

                        OutputStreamWriter wr =
                                new OutputStreamWriter(urlConnection.getOutputStream());
                        wr.write(jsonPayloadString);
                        wr.close();
                        int result = urlConnection.getResponseCode();
                        String responseMessage = urlConnection.getResponseMessage();

                        if (result == HttpsURLConnection.HTTP_CREATED) {
                            final String registrationLocation =
                                    urlConnection.getHeaderField(REGISTRATION_ID_LOCATION);
                            if (registrationLocation != null) {
                                // Report REGISTRATION_EVENT only for registration
                                if (!isUnregistrationRequest) {
                                    publishRegistrationSuccessfulEvent(
                                            accessToken, requestId, EventType.REGISTRATION_EVENT);
                                }
                                handler.post(
                                        () -> registrarListener.onSuccess(registrationLocation));
                            } else {
                                final String errorMessage =
                                        isUnregistrationRequest
                                                ? DEFAULT_UNREGISTRATION_FAILED_MESSAGE
                                                : DEFAULT_REGISTRATION_FAILED_MESSAGE;
                                publishError(
                                        isUnregistrationRequest,
                                        accessToken,
                                        requestId,
                                        result,
                                        responseMessage);
                                handler.post(
                                        () ->
                                                registrarListener.onError(
                                                        new RegistrationException(
                                                                RegistrationException
                                                                        .EXCEPTION_REGISTRATION_ERROR,
                                                                errorMessage,
                                                                "Registration Location is null")));
                            }
                        } else {
                            JSONObject jsonObject =
                                    processJSONError(urlConnection.getErrorStream());
                            int jsonCode = jsonObject.getInt(JSON_CODE_KEY);
                            String jsonMessage = jsonObject.getString(JSON_MESSAGE_KEY);
                            final RegistrationException registrationException =
                                    handleHttpErrorCode(
                                            isUnregistrationRequest,
                                            result,
                                            jsonCode,
                                            jsonMessage,
                                            responseMessage);

                            publishError(
                                    isUnregistrationRequest,
                                    accessToken,
                                    requestId,
                                    registrationException.getErrorCode(),
                                    registrationException.getMessage()
                                            + " : "
                                            + registrationException.getExplanation());
                            handler.post(() -> registrarListener.onError(registrationException));
                        }
                    } catch (Exception e) {
                        handleException(
                                isUnregistrationRequest,
                                e,
                                urlConnection,
                                handler,
                                registrarListener);
                        String errorMessage =
                                isUnregistrationRequest
                                        ? DEFAULT_UNREGISTRATION_FAILED_MESSAGE
                                        : DEFAULT_REGISTRATION_FAILED_MESSAGE;
                        publishError(
                                isUnregistrationRequest,
                                accessToken,
                                requestId,
                                RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                errorMessage + " : " + e.getMessage());
                    } finally {
                        VoiceURLConnection.release(urlConnection);
                    }
                });
    }

    public static void unregister(
            final String accessToken,
            final String hostURL,
            final RegistrarListener registrarListener) {
        final boolean isUnregistrationRequest = true;
        String requestId = SidUtil.generateGUID(Constants.TWILIO_REQUEST_SID_PREFIX, accessToken);
        final Executor executor = Executors.newCachedThreadPool();
        final Handler handler = Utils.createHandler();
        executor.execute(
                () -> {
                    HttpsURLConnection urlConnection = null;
                    try {
                        urlConnection =
                                VoiceURLConnection.create(
                                        accessToken,
                                        hostURL,
                                        VoiceURLConnection.METHOD_TYPE_DELETE);
                        urlConnection.addRequestProperty(
                                Constants.TWILIO_REQUEST_HEADER, requestId);

                        urlConnection.connect();
                        int result = urlConnection.getResponseCode();
                        String responseMessage = urlConnection.getResponseMessage();

                        if (result == HttpsURLConnection.HTTP_OK
                                || result == HttpsURLConnection.HTTP_NO_CONTENT) {
                            publishRegistrationSuccessfulEvent(
                                    accessToken, requestId, EventType.UNREGISTRATION_EVENT);
                            handler.post(() -> registrarListener.onSuccess(null));
                        } else {
                            JSONObject jsonObject =
                                    processJSONError(urlConnection.getErrorStream());
                            int jsonCode = jsonObject.getInt(JSON_CODE_KEY);
                            String jsonMessage = jsonObject.getString(JSON_MESSAGE_KEY);

                            final RegistrationException registrationException =
                                    handleHttpErrorCode(
                                            isUnregistrationRequest,
                                            result,
                                            jsonCode,
                                            jsonMessage,
                                            responseMessage);

                            publishRegistrationErrorEvent(
                                    accessToken,
                                    requestId,
                                    EventType.UNREGISTRATION_ERROR_EVENT,
                                    registrationException.getErrorCode(),
                                    registrationException.getMessage()
                                            + " : "
                                            + registrationException.getExplanation());

                            handler.post(() -> registrarListener.onError(registrationException));
                        }
                    } catch (Exception e) {
                        handleException(
                                isUnregistrationRequest,
                                e,
                                urlConnection,
                                handler,
                                registrarListener);
                        String errorMessage =
                                isUnregistrationRequest
                                        ? DEFAULT_UNREGISTRATION_FAILED_MESSAGE
                                        : DEFAULT_REGISTRATION_FAILED_MESSAGE;
                        publishError(
                                isUnregistrationRequest,
                                accessToken,
                                requestId,
                                RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                errorMessage + " : " + e.getMessage());
                    } finally {
                        VoiceURLConnection.release(urlConnection);
                    }
                });
    }

    private static void handleException(
            Boolean isUnregistrationRequest,
            Exception e,
            HttpsURLConnection urlConnection,
            Handler handler,
            RegistrarListener registrarListener) {
        final String errorMessage =
                isUnregistrationRequest
                        ? DEFAULT_UNREGISTRATION_FAILED_MESSAGE
                        : DEFAULT_REGISTRATION_FAILED_MESSAGE;
        if (urlConnection != null) {
            /*
             * The exception with the message "No authentication challenges found" may be
             * thrown on some devices if the server response omits the "WWW-Authenticate"
             * header when returning a 401 or 407. To workaround this issue it is possible
             * to still parse the error stream from the url connection to get the error
             * response information.
             * For reference see: https://stackoverflow.com/a/13624655/759872
             */
            try {
                JSONObject jsonObject = processJSONError(urlConnection.getErrorStream());
                int code = jsonObject.getInt(JSON_CODE_KEY);
                String message = jsonObject.getString(JSON_MESSAGE_KEY);

                /*
                 * Determine RegistrationException in handleHttpAuthErrorCode to ensure that
                 * the SDK does not incorrectly surface the default RegistrationException in
                 * cases where an authorization exception occurred.
                 */
                handler.post(
                        () ->
                                registrarListener.onError(
                                        handleHttpAuthErrorCode(
                                                isUnregistrationRequest, code, message)));
            } catch (Exception e1) {
                // Throw the original exception e
                handler.post(
                        () ->
                                registrarListener.onError(
                                        new RegistrationException(
                                                RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                                errorMessage,
                                                e.getMessage())));
            }
        } else {
            handler.post(
                    () ->
                            registrarListener.onError(
                                    new RegistrationException(
                                            RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                            errorMessage,
                                            e.getMessage())));
        }
    }

    @VisibleForTesting
    static RegistrationException handleHttpErrorCode(
            boolean isUnregistrationRequest,
            int httpCode,
            int jsonCode,
            String jsonMessage,
            String httpResponse) {
        int errorCode = RegistrationException.EXCEPTION_REGISTRATION_ERROR;
        String errorMessage =
                isUnregistrationRequest
                        ? DEFAULT_UNREGISTRATION_FAILED_MESSAGE
                        : DEFAULT_REGISTRATION_FAILED_MESSAGE;
        String errorExplanation = String.valueOf(jsonCode) + " : " + jsonMessage;
        RegistrationException registrationException;

        switch (httpCode) {
            case HttpsURLConnection.HTTP_UNAUTHORIZED:
                registrationException =
                        handleHttpAuthErrorCode(isUnregistrationRequest, jsonCode, jsonMessage);
                break;
            case HttpsURLConnection.HTTP_BAD_REQUEST:
                errorCode = RegistrationException.BadRequestException.getErrorCode();
                errorMessage = RegistrationException.BadRequestException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.BadRequestException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_FORBIDDEN:
                errorCode = RegistrationException.ForbiddenException.getErrorCode();
                errorMessage = RegistrationException.ForbiddenException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.ForbiddenException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_NOT_FOUND:
                errorCode = RegistrationException.NotFoundException.getErrorCode();
                errorMessage = RegistrationException.NotFoundException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.NotFoundException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_CLIENT_TIMEOUT:
                errorCode = RegistrationException.RequestTimeoutException.getErrorCode();
                errorMessage = RegistrationException.RequestTimeoutException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.RequestTimeoutException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_CONFLICT:
                errorCode = RegistrationException.ConflictException.getErrorCode();
                errorMessage = RegistrationException.ConflictException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.ConflictException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case VoiceURLConnection.HTTP_TOO_MANY_REQUEST:
                errorCode = RegistrationException.TooManyRequestException.getErrorCode();
                errorMessage = RegistrationException.TooManyRequestException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.TooManyRequestException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_INTERNAL_ERROR:
                errorCode = RegistrationException.InternalServerErrorException.getErrorCode();
                errorMessage = RegistrationException.InternalServerErrorException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.InternalServerErrorException
                                        .getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_BAD_GATEWAY:
                errorCode = RegistrationException.BadGatewayException.getErrorCode();
                errorMessage = RegistrationException.BadGatewayException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.BadGatewayException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_UNAVAILABLE:
                errorCode = RegistrationException.ServiceUnavailableException.getErrorCode();
                errorMessage = RegistrationException.ServiceUnavailableException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.ServiceUnavailableException
                                        .getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case HttpsURLConnection.HTTP_GATEWAY_TIMEOUT:
                errorCode = RegistrationException.GatewayTimeoutException.getErrorCode();
                errorMessage = RegistrationException.GatewayTimeoutException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.GatewayTimeoutException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            case VoiceURLConnection.HTTP_UPGRADE_REQUIRED:
                errorCode = RegistrationException.UpgradeRequiredException.getErrorCode();
                errorMessage = RegistrationException.UpgradeRequiredException.getMessage();
                errorExplanation =
                        jsonMessage != null
                                ? jsonCode + " : " + jsonMessage
                                : RegistrationException.UpgradeRequiredException.getExplanation();
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            default:
                errorExplanation =
                        jsonCode == 0
                                ? String.valueOf(httpCode) + " : " + httpResponse
                                : errorExplanation;
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
        }
        return registrationException;
    }

    private static RegistrationException handleHttpAuthErrorCode(
            boolean isUnregistrationRequest, int jsonErrorCode, String jsonErrorMessage) {
        int errorCode = jsonErrorCode;
        String errorMessage = jsonErrorMessage;
        String errorExplanation = jsonErrorCode + " : " + errorMessage;
        RegistrationException registrationException;

        switch (jsonErrorCode) {
            case VoiceException.EXCEPTION_INVALID_ACCESS_TOKEN:
            case VoiceException.EXCEPTION_INVALID_ACCESS_TOKEN_HEADER:
            case VoiceException.EXCEPTION_INVALID_ISSUER_SUBJECT:
            case VoiceException.EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY:
            case VoiceException.EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET:
            case VoiceException.EXCEPTION_INVALID_ACCESS_TOKEN_GRANT:
            case VoiceException.EXCEPTION_INVALID_SIGNATURE:
            case VoiceException.EXCEPTION_AUTH_FAILURE:
            case VoiceException.EXCEPTION_INVALID_TTL:
            case VoiceException.EXCEPTION_INVALID_TOKEN:
            case VoiceException.EXCEPTION_ACCESS_TOKEN_REJECTED:
                registrationException =
                        new RegistrationException(errorCode, errorMessage, errorExplanation);
                break;
            default:
                if (!isUnregistrationRequest) {
                    registrationException =
                            new RegistrationException(
                                    RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                    DEFAULT_REGISTRATION_FAILED_MESSAGE,
                                    errorExplanation);
                } else {
                    registrationException =
                            new RegistrationException(
                                    RegistrationException.EXCEPTION_REGISTRATION_ERROR,
                                    DEFAULT_UNREGISTRATION_FAILED_MESSAGE,
                                    errorExplanation);
                }
                break;
        }
        return registrationException;
    }

    private static JSONObject processJSONError(InputStream errorStream)
            throws IOException, JSONException {
        BufferedReader bufferedReader;
        StringBuilder stringBuilder = new StringBuilder();
        if (errorStream != null) {
            bufferedReader = new BufferedReader(new InputStreamReader((errorStream)));
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                stringBuilder.append(line);
                stringBuilder.append('\n');
            }
            bufferedReader.close();
        }
        return new JSONObject(stringBuilder.toString());
    }

    private static void publishError(
            boolean isUnregistrationRequest,
            String accessToken,
            String requestId,
            int errorCode,
            String errorMessage) {
        if (!isUnregistrationRequest) {
            // Report REGISTRATION_ERROR for registration
            publishRegistrationErrorEvent(
                    accessToken,
                    requestId,
                    EventType.REGISTRATION_ERROR_EVENT,
                    errorCode,
                    errorMessage);
        } else {
            /*
             * Report REGISTRATION_UNREGISTRATION_ERROR if registration was called
             * as part of unregistration flow
             */
            publishRegistrationErrorEvent(
                    accessToken,
                    requestId,
                    EventType.UNREGISTRATION_REGISTRATION_ERROR_EVENT,
                    errorCode,
                    errorMessage);
        }
    }

    private static void publishRegistrationErrorEvent(
            String accessToken,
            String requestId,
            String eventName,
            int errorCode,
            String errorMessage) {

        EventPublisher publisher =
                new EventPublisher(Constants.getClientSdkProductName(), accessToken);
        EventPayload eventPayload =
                new EventPayload.Builder()
                        .productName(com.twilio.voice.Constants.getClientSdkProductName())
                        .requestId(requestId)
                        .errorCode(Long.valueOf(errorCode))
                        .errorMessage(errorMessage)
                        .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE)
                        .build();
        try {
            JSONObject registrationEventPayload = eventPayload.getPayload();
            Event event =
                    publisher.createEvent(
                            com.twilio.voice.Constants.SeverityLevel.ERROR,
                            EventGroupType.REGISTRATION_EVENT_GROUP,
                            eventName,
                            registrationEventPayload);
            publisher.publish(
                    com.twilio.voice.Constants.SeverityLevel.ERROR,
                    EventGroupType.REGISTRATION_EVENT_GROUP,
                    eventName,
                    event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void publishRegistrationSuccessfulEvent(
            String accessToken, String requestId, String eventName) {
        EventPublisher publisher =
                new EventPublisher(Constants.getClientSdkProductName(), accessToken);
        EventPayload eventPayload =
                new EventPayload.Builder()
                        .productName(com.twilio.voice.Constants.getClientSdkProductName())
                        .requestId(requestId)
                        .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE)
                        .build();
        try {
            JSONObject registrationEventPayload = eventPayload.getPayload();
            Event event =
                    publisher.createEvent(
                            com.twilio.voice.Constants.SeverityLevel.INFO,
                            EventGroupType.REGISTRATION_EVENT_GROUP,
                            eventName,
                            registrationEventPayload);
            publisher.publish(
                    com.twilio.voice.Constants.SeverityLevel.INFO,
                    EventGroupType.REGISTRATION_EVENT_GROUP,
                    eventName,
                    event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
