package com.twilio.voice;

import android.content.Context;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Handler;
import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.util.Map;

class CallInviteProxy implements MessageListener {
    private static final Logger logger = Logger.getLogger(CallInviteProxy.class);

    private final ThreadUtils.ThreadChecker threadChecker;
    private final Context context;
    long nativeCallInviteProxy;
    private final Handler handler;
    private final MessageListener messageListener;
    private final MediaFactory mediaFactory;
    @Nullable private Call.EventListener eventListener;
    private boolean released = false;
    private ConnectivityReceiver connectivityReceiver = null;
    private Call call;
    private CallInvite callInvite;
    private String selectedRegion = Voice.region;
    private String gateway;
    private String region;
    String codecParams;
    String selectedCodec;
    private EventPublisher publisher;

    private String tempCallSid;

    /*
     * This event listener is used at the JNI level to proxy event listener callbacks to event
     * listeners provided when accept and reject are called.
     */
    @SuppressWarnings("unused")
    private final Call.EventListener eventListenerProxy =
            new Call.EventListener() {
                @Override
                public void onEvent(Map<String, Pair<String, Class>> data) {
                    handler.post(
                            () -> {
                                CallInviteProxy.this.threadChecker.checkIsOnValidThread();
                                logger.d("CallInviteProxy::eventListenerProxy::onEvent(...)");
                                if ((data.get(EventKeys.EVENT_GROUP)
                                                .first
                                                .equals(EventGroupType.REGISTRATION_EVENT_GROUP))
                                        && (data.get(EventKeys.EVENT_NAME)
                                                .first
                                                .equals(
                                                        EventType
                                                                .UNSUPPORTED_CANCEL_MESSAGE_ERROR_EVENT))) {
                                    CallInviteProxy.this.release(
                                            data.get(EventKeys.CALL_SID_KEY).first);
                                }
                                if (eventListener != null) {
                                    eventListener.onEvent(data);
                                }
                                publishEvent(data);
                            });
                }

                @Override
                public void onMetric(Map<String, Pair<String, Class>> data) {
                    handler.post(
                            () -> {
                                CallInviteProxy.this.threadChecker.checkIsOnValidThread();
                                logger.d("onMetric");
                                if (eventListener != null) {
                                    eventListener.onMetric(data);
                                }

                                if (data.get(EventKeys.EVENT_GROUP)
                                        .first
                                        .equals(EventGroupType.CALL_QUALITY_STATS_GROUP)) {
                                    call.onSample(InsightsUtils.createRtcSample(data));
                                }
                            });
                }
            };

    @Override
    public void onCallInvite(final @NonNull CallInvite callInvite) {
        handler.post(
                () -> {
                    threadChecker.checkIsOnValidThread();
                    CallInviteProxy.this.threadChecker.checkIsOnValidThread();
                    logger.d("onCallInvite");
                    Voice.callInviteProxyMap.put(callInvite.getCallSid(), CallInviteProxy.this);
                    registerConnectivityBroadcastReceiver();
                    messageListener.onCallInvite(callInvite);
                });
    }

    @Override
    public void onCancelledCallInvite(
            @NonNull CancelledCallInvite cancelledCallInvite,
            @Nullable CallException callException) {
        handler.post(
                () -> {
                    threadChecker.checkIsOnValidThread();
                    logger.d(
                            "onCancelledCallInvite: CallException code: "
                                    + ((callException == null)
                                            ? "null"
                                            : callException.getErrorCode()));
                    boolean wasReleased = released;
                    release(cancelledCallInvite.getCallSid());

                    /*
                     * Surface the callback if the call invite proxy was not released as part of calling
                     * accept or reject.
                     *
                     */
                    if (!wasReleased) {
                        messageListener.onCancelledCallInvite(cancelledCallInvite, callException);
                    }
                });
    }

    synchronized void setEventListener(Call.EventListener eventListener) {
        this.threadChecker.checkIsOnValidThread();
        this.eventListener = eventListener;
    }

    synchronized void setCall(Call call) {
        this.threadChecker.checkIsOnValidThread();
        this.call = call;
    }

    synchronized void setTempCallSid(String tempCallSid) {
        this.threadChecker.checkIsOnValidThread();
        this.tempCallSid = tempCallSid;
    }

    synchronized EventPublisher getPublisher() {
        this.threadChecker.checkIsOnValidThread();
        return publisher;
    }

    void networkChange(Voice.NetworkChangeEvent networkChangeEvent) {
        threadChecker.checkIsOnValidThread();

        if (!released && nativeCallInviteProxy != 0) {
            nativeNetworkChange(nativeCallInviteProxy, networkChangeEvent);
        } else {
            logger.d(
                    "Ignoring networkChangeEvent: "
                            + networkChangeEvent.name()
                            + " because CallInviteProxy is either released or is not set.");
        }
    }

    synchronized void release(String callSid) {
        logger.d("CallInviteProxy::release");
        this.threadChecker.checkIsOnValidThread();

        if (!released) {
            unregisterConnectivityBroadcastReceiver();

            Voice.callInviteProxyMap.remove(callSid);

            if (nativeCallInviteProxy != 0) {
                nativeRelease(nativeCallInviteProxy);
                nativeCallInviteProxy = 0;
            }

            mediaFactory.release(this);

            released = true;
        }
    }

    String sendMessage(CallMessage message) {
        if (!released && nativeCallInviteProxy != 0) {
            return nativeSendMessage(nativeCallInviteProxy, message);
        } else {
            logger.d(
                    "Ignoring sendMessage:"
                            + message.getContent()
                            + " because CallInviteProxy is either released or is not set.");
            return "";
        }
    }

    CallInviteProxy(
            Context context,
            Handler handler,
            MessageListener messageListener,
            Call.EventListener eventListener,
            CallInvite callInvite) {
        Preconditions.checkApplicationContext(context, "must create Call with application context");
        this.context = context;
        this.handler = handler;
        this.threadChecker = new ThreadUtils.ThreadChecker(handler.getLooper().getThread());
        this.messageListener = messageListener;
        this.eventListener = eventListener;
        this.mediaFactory = MediaFactory.instance(this, context);
        this.callInvite = callInvite;
        String bridgeToken = null;

        if (Voice.callSidBridgeTokenPair != null) {
            bridgeToken =
                    Voice.callSidBridgeTokenPair.first.equals(callInvite.getCallSid())
                            ? Voice.callSidBridgeTokenPair.second
                            : null;
        }

        if (bridgeToken != null) {
            publisher =
                    new EventPublisher(context, Constants.getClientSdkProductName(), bridgeToken);
            publisher.addListener(
                    voiceException ->
                            logger.e(
                                    "Error publishing data : "
                                            + voiceException.getMessage()
                                            + ":"
                                            + voiceException.getErrorCode()));
        }
    }

    private void registerConnectivityBroadcastReceiver() {
        connectivityReceiver = new ConnectivityReceiver();
        context.registerReceiver(
                connectivityReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
    }

    private void unregisterConnectivityBroadcastReceiver() {
        if (connectivityReceiver != null) {
            context.unregisterReceiver(connectivityReceiver);
        }
    }

    private void publishEvent(Map<String, Pair<String, Class>> data) {
        if (data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.SETTINGS_GROUP)) {
            if (data.get(EventKeys.EVENT_NAME).first.equals(EventType.CODEC_EVENT)) {
                codecParams = data.get(EventKeys.CODEC_PARAMS).first;
                selectedCodec = data.get(EventKeys.SELECTED_CODEC).first;
            } else if (data.get(EventKeys.EVENT_NAME).first.equals(EventType.EDGE_EVENT)) {
                gateway = (data.get(EventKeys.EDGE_HOST_NAME)).first;
                region = (data.get(EventKeys.EDGE_HOST_REGION)).first;
            }
            InsightsUtils.processEvent(
                    data,
                    createEventPayloadBuilderForSettingsEvent(),
                    publisher,
                    Constants.Direction.INCOMING);
        } else {
            InsightsUtils.processEvent(
                    data, createEventPayloadBuilder(), publisher, Constants.Direction.INCOMING);
        }
    }

    private EventPayload.Builder createEventPayloadBuilder() {
        return new EventPayload.Builder()
                .callSid(callInvite.getCallSid())
                .tempCallSid(tempCallSid)
                .messageSid(callInvite.getMessageSid())
                .direction(Constants.Direction.INCOMING)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.getClientSdkProductName())
                .clientName(Utils.parseClientIdentity(callInvite.getTo()))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE);
    }

    EventPayload.Builder createEventPayloadBuilderForSettingsEvent() {
        return createEventPayloadBuilder().codecParams(codecParams).selectedCodec(selectedCodec);
    }

    private native void nativeNetworkChange(
            long nativeCallInviteProxy, Voice.NetworkChangeEvent networkChangeEvent);

    private native void nativeRelease(long nativeCallInviteProxy);

    private native String nativeSendMessage(long nativeCallInviteProxy, CallMessage callMessage);
}
