/*
 * Copyright (C) 2018 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.voice;

import androidx.annotation.NonNull;

/**
 * Lossy audio coding format.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Opus_(audio_format)">Opus</a>
 */
public class OpusCodec extends AudioCodec {
    @NonNull
    public static final String NAME = "opus";
    static int DEFAULT_BITRATE = 0;
    static int MIN_BITRATE = 6000;
    static int MAX_BITRATE = 510000;
    private int maxAverageBitrate = DEFAULT_BITRATE;
    private boolean useDtx = false;

    public OpusCodec() {
        super(NAME);
    }

    /**
     * A constructor to initialize the OpusCodec with non-default attributes.
     *
     * @param maxAverageBitrate `maxAverageBitrate` - The maximum average audio bitrate to use, in
     *                          bits per second (bps) based on [RFC-7587 7.1](https://tools.ietf.org/html/rfc7587#section-7.1).
     *                          By default, the setting is not used. If you specify 0, then the
     *                          setting is not used. Any positive integer is allowed, but values
     *                          outside the range 6000 to 510000 are ignored and treated as 0. The
     *                          recommended bitrate for speech is between 8000 and 40000 bps as noted
     *                          in [RFC-7587 3.1.1](https://tools.ietf.org/html/rfc7587#section-3.1.1).
     */
    public OpusCodec(int maxAverageBitrate) {
        super(NAME);
        if (maxAverageBitrate >= MIN_BITRATE && maxAverageBitrate <= MAX_BITRATE) {
            this.maxAverageBitrate = maxAverageBitrate;
        }
    }

    /**
     * A constructor to initialize the OpusCodec with non-default attributes.
     *
     * @param maxAverageBitrate `maxAverageBitrate` - The maximum average audio bitrate to use, in
     *                          bits per second (bps) based on [RFC-7587 7.1](https://tools.ietf.org/html/rfc7587#section-7.1).
     *                          By default, the setting is not used. If you specify 0, then the
     *                          setting is not used. Any positive integer is allowed, but values
     *                          outside the range 6000 to 510000 are ignored and treated as 0. The
     *                          recommended bitrate for speech is between 8000 and 40000 bps as noted
     *                          in [RFC-7587 3.1.1](https://tools.ietf.org/html/rfc7587#section-3.1.1).
     * @param useDtx Discontinuous transmission reduces the bitrate for certain input signals,
     *               like periods of silence [RFC-7587 3.1.3](https://tools.ietf.org/html/rfc7587#section-3.1.3).
     *               The default is false. This setting applies to sender and receiver transmission.
     */
    OpusCodec(int maxAverageBitrate, boolean useDtx) {
        super(NAME);
        if (maxAverageBitrate >= MIN_BITRATE && maxAverageBitrate <= MAX_BITRATE) {
            this.maxAverageBitrate = maxAverageBitrate;
        }
        this.useDtx = useDtx;
    }

    public int getMaxAverageBitrate() {
        return maxAverageBitrate;
    }

    boolean isDtxEnabled() {
        return useDtx;
    }
}
