package com.twilio.voice;

import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;

class InsightsUtils {

    static RTCStatsSample createRtcSample(Map<String, Pair<String, Class>> data) {

        RTCStatsSample rtcStatsSample = new RTCStatsSample();
        rtcStatsSample.packetsReceived = fromString(data.get(MetricEventConstants.MetricEventKeys.PACKETS_RECEIVED), Long.class);
        rtcStatsSample.totalPacketsLost = fromString(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_LOST), Long.class);
        rtcStatsSample.fractionLost = fromString(data.get(MetricEventConstants.MetricEventKeys.PACKETS_LOST_FRACTION), Float.class);
        rtcStatsSample.packetsLost = fromString(data.get(MetricEventConstants.MetricEventKeys.PACKETS_LOST), Long.class);
        rtcStatsSample.totalPacketsSent = fromString(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_SENT), Long.class);
        rtcStatsSample.totalPacketsReceived = fromString(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_RECEIVED), Long.class);
        rtcStatsSample.totalBytesReceived = fromString(data.get(MetricEventConstants.MetricEventKeys.TOTAL_BYTES_RECEIVED), Long.class);
        rtcStatsSample.totalBytesSent = fromString(data.get(MetricEventConstants.MetricEventKeys.TOTAL_BYTES_SENT), Long.class);
        rtcStatsSample.jitter = fromString(data.get(MetricEventConstants.MetricEventKeys.JITTER), Long.class);
        rtcStatsSample.rtt = fromString(data.get(MetricEventConstants.MetricEventKeys.RTT), Long.class);
        rtcStatsSample.audioInputLevel = fromString(data.get(MetricEventConstants.MetricEventKeys.AUDIO_INPUT_LEVEL), Long.class);
        rtcStatsSample.audioOutputLevel = fromString(data.get(MetricEventConstants.MetricEventKeys.AUDIO_OUTPUT_LEVEL), Long.class);
        rtcStatsSample.mos = fromString(data.get(MetricEventConstants.MetricEventKeys.MOS), Float.class);
        rtcStatsSample.codec = fromString(data.get(MetricEventConstants.MetricEventKeys.AUDIO_CODEC), String.class);
        rtcStatsSample.timestampMS = fromString(data.get(MetricEventConstants.MetricEventKeys.TIMESTAMP_MS), String.class);

        // populate sample TimeStamp
        TimeZone tz = TimeZone.getTimeZone("UTC");
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        df.setTimeZone(tz);
        Date now = new Date(Double.valueOf(rtcStatsSample.timestampMS).longValue());
        rtcStatsSample.timeStamp = df.format(now);

        return rtcStatsSample;
    }

    static JSONArray publishMetrics(RTCStatsSample currentSample,
                               int sampleCounter,
                               String callSid,
                               Constants.Direction direction,
                               JSONArray payload,
                               EventPublisher publisher) {

        if (sampleCounter % MetricEventConstants.PUBLISH_SAMPLE_INTERVAL == 0) {
            currentSample.setCallSid(callSid);
            currentSample.setDirection(direction);

            if (payload == null) {
                payload = new JSONArray();
            }
            payload.put(currentSample.toJson());
            if (payload.length() == MetricEventConstants.BATCH_PAYLOAD_COUNT) {
                if (publisher != null) {
                    try {
                        MetricEvent event = publisher.createMetricEvent(EventGroupType.CALL_QUALITY_STATS_GROUP,
                                EventType.CALL_METRIC_EVENT,
                                payload);
                        publisher.publishMetrics(event);

                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    payload = new JSONArray();
                } else {
                    // If the payload reaches the BATCH_PAYLOAD_COUNT but the
                    // publisher is null, reset the payload to
                    // eliminate buffer overflow.
                    payload = new JSONArray();
                }
            }
        }
        return payload;
    }

    // Insights Events
    static void processEvent(Map<String, Pair<String, Class>> data,
                             EventPayload.Builder builder,
                             EventPublisher publisher,
                             Constants.Direction direction) {
        if (fromString(data.get(EventKeys.EVENT_GROUP), String.class).equals(EventGroupType.ICE_CANDIDATE_GROUP)) {
            if (fromString(data.get(EventKeys.EVENT_NAME), String.class).equals(EventType.ICE_CANDIDATE_EVENT)) {
                publishIceCandidateEvent(data, builder, publisher, direction);
            } else if (fromString(data.get(EventKeys.EVENT_NAME), String.class).equals(EventType.SELECTED_ICE_CANDIDATE_PAIR_EVENT)) {
                publishSelectedIceCandidateEvent(data, builder, publisher, direction);
            }
        } else {
            if (fromString(data.get(EventKeys.EVENT_NAME), String.class).equals(EventType.ERROR_EVENT) ||
                fromString(data.get(EventKeys.EVENT_NAME), String.class).equals(EventType.LISTENING_ERROR_EVENT) ||
                fromString(data.get(EventKeys.EVENT_NAME), String.class).equals(EventType.UNSUPPORTED_CANCEL_MESSAGE_ERROR_EVENT)) {
                if (data.get(EventKeys.ERROR_CODE_KEY) != null) {
                    InsightsUtils.publishConnectionErrorEvent(data, builder, publisher);
                }
            } else {
                InsightsUtils.publishEvent(data, builder, publisher);
            }
        }
    }

    static void processWarningEvent(Map<String, Pair<String, Class>> data,
                                    EventPayload.Builder builder,
                                    EventPublisher publisher) {
        String eventName = fromString(data.get(EventKeys.EVENT_NAME), String.class);
        Constants.SeverityLevel severityLevel = Constants.SeverityLevel.valueOf((fromString(data.get(EventKeys.LEVEL_TAG), String.class)));
        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);

        EventPayload warningEventPayload = createWarningEventPayload(data, builder);

        publish(warningEventPayload,
                severityLevel,
                group,
                eventName,
                publisher);
    }

    private static void publishConnectionErrorEvent(Map<String, Pair<String, Class>> data,
                                            EventPayload.Builder builder,
                                            EventPublisher publisher) {

        String eventName = fromString(data.get(EventKeys.EVENT_NAME), String.class);
        Constants.SeverityLevel severityLevel = Constants.SeverityLevel.valueOf((fromString(data.get(EventKeys.LEVEL_TAG), String.class)));
        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);

        EventPayload eventPayload = builder
                .errorCode(fromString(data.get(EventKeys.ERROR_CODE_KEY), Long.class))
                .errorMessage(fromString(data.get(EventKeys.ERROR_MESSAGE_KEY), String.class) + " : " + fromString(data.get(EventKeys.ERROR_EXPLANATION_KEY), String.class))
                .build();

        publish(eventPayload,
                severityLevel,
                group,
                eventName,
                publisher);
    }

    private static void publishEvent(Map<String, Pair<String, Class>> data,
                             EventPayload.Builder builder,
                             EventPublisher publisher) {

        String eventName = fromString(data.get(EventKeys.EVENT_NAME), String.class);
        Constants.SeverityLevel severityLevel = Constants.SeverityLevel.valueOf((fromString(data.get(EventKeys.LEVEL_TAG), String.class)));
        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);

        EventPayload eventPayload = builder.build();

        publish(eventPayload,
                severityLevel,
                group,
                eventName,
                publisher);
    }

    private static void publishIceCandidateEvent(Map<String, Pair<String, Class>> data,
                                         EventPayload.Builder builder,
                                         EventPublisher publisher,
                                                 Constants.Direction direction) {
        String eventName = fromString(data.get(EventKeys.EVENT_NAME), String.class);
        Constants.SeverityLevel severityLevel = Constants.SeverityLevel.valueOf((fromString(data.get(EventKeys.LEVEL_TAG), String.class)));
        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);

        EventPayload eventPayload = null;

        eventPayload = builder.transportId(fromString(data.get(EventKeys.TRANSPORT_ID), String.class))
                .isRemote((fromString(data.get(EventKeys.IS_REMOTE), Boolean.class)))
                .ip(fromString(data.get(EventKeys.IP), String.class))
                .port(fromString(data.get(EventKeys.PORT), Long.class))
                .protocol(fromString(data.get(EventKeys.PROTOCOL), String.class))
                .candidateType(fromString(data.get(EventKeys.CANDIDATE_TYPE), String.class))
                .priority(fromString(data.get(EventKeys.PRIORITY), Long.class))
                .url(fromString(data.get(EventKeys.URL), String.class))
                .deleted(fromString(data.get(EventKeys.DELETED), Boolean.class))
                .networkCost(fromString(data.get(EventKeys.NETWORK_COST), Long.class))
                .networkId(fromString(data.get(EventKeys.NETWORK_ID), Long.class))
                .relatedPort(fromString(data.get(EventKeys.RELATED_PORT), Long.class))
                .level(fromString(data.get(EventKeys.LEVEL), String.class))
                .relatedAddress(fromString(data.get(EventKeys.RELATED_ADDRESS), String.class))
                .name(fromString(data.get(EventKeys.NAME), String.class))
                .tcpType(fromString(data.get(EventKeys.TCP_TYPE), String.class))
                .networkType(fromString(data.get(EventKeys.NETWORK_TYPE), String.class))
                .direction(direction)
                .build();

        publish(eventPayload,
                        severityLevel,
                        group,
                        eventName,
                        publisher);
    }

    private static void publishSelectedIceCandidateEvent(Map<String, Pair<String, Class>> data,
                                                 EventPayload.Builder builder,
                                                 EventPublisher publisher,
                                                 Constants.Direction direction) {
        String eventName = fromString(data.get(EventKeys.EVENT_NAME), String.class);
        Constants.SeverityLevel severityLevel = Constants.SeverityLevel.valueOf((fromString(data.get(EventKeys.LEVEL_TAG), String.class)));
        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);

        EventPayload eventPayload = null;

        eventPayload = builder.reason(fromString(data.get(EventKeys.REASON), String.class))
                .lastDataReceivedMs(fromString(data.get(EventKeys.LAST_DATA_RECEIVED_MS), Long.class))
                .level(fromString(data.get(EventKeys.LEVEL), String.class))
                .name(fromString(data.get(EventKeys.NAME), String.class))
                .localCandidate(fromString(data.get(EventKeys.LOCAL_CANDIDATE), org.json.JSONObject.class))
                .remoteCandidate(fromString(data.get(EventKeys.REMOTE_CANDIDATE), org.json.JSONObject.class))
                .build();

        publish(eventPayload,
                severityLevel,
                group,
                eventName,
                publisher);
    }

    private static void publish(EventPayload eventPayload,
                 Constants.SeverityLevel severityLogLevel,
                 String group, String eventName,
                 EventPublisher publisher) {
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (publisher != null) {
                Event event = publisher.createEvent(severityLogLevel,
                        group, eventName, connectionEventPayload);
                publisher.publish(severityLogLevel,
                        group, eventName, event);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    static <T> T fromString(Pair<String, Class> stringValuePair, Class<T> reifiedType) {
        Object object = null;

        if (BuildConfig.DEBUG && stringValuePair.second != reifiedType) {
            throw new AssertionError("Data type received from the core does not match expected value. Expected" + reifiedType + " Received : " + stringValuePair.second);
        }

        if (stringValuePair.second.equals(Boolean.class)) {
            object = Boolean.parseBoolean(stringValuePair.first);
        } else if (stringValuePair.second.equals(Float.class)) {
            object = Float.parseFloat(stringValuePair.first);
        } else if (stringValuePair.second.equals(Long.class)) {
            object = Long.parseLong(stringValuePair.first);
        } else if (stringValuePair.second.equals(org.json.JSONObject.class)) {
            try {
                object = new JSONObject(stringValuePair.first);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            object = stringValuePair.first;
        }

        return reifiedType.cast(object);
    }

    static EventPayload createWarningEventPayload(Map<String, Pair<String, Class>> data,
                                    EventPayload.Builder builder) {

        String group = fromString(data.get(EventKeys.EVENT_GROUP), String.class);
        String values = fromString(data.get(EventKeys.VALUES_KEY), String.class);
        Pair<String, Class> thresholdStringValue = data.get(EventKeys.THRESHOLD_KEY);

        EventPayload warningEventPayload;
        if (group.equals(EventGroupType.AUDIO_LEVEL_WARNING_RAISED)) {
            warningEventPayload = builder
                    .value(values)
                    .qualityThresholdValuePair(thresholdStringValue)
                    .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        } else {
            warningEventPayload = builder
                    .values(values)
                    .qualityThresholdValuePair(thresholdStringValue)
                    .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        }
        return warningEventPayload;
    }
}
