package com.twilio.voice;

import androidx.annotation.NonNull;
import android.util.Pair;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents options when connecting to a {@link Call}.
 */
public class ConnectOptions extends CallOptions {
    private final String accessToken;
    private final Map<String, String> params;
    private Call.EventListener eventListener;

    /**
     * ConnectOptions can only be constructed via {@link ConnectOptions.Builder}
     */
    @SuppressWarnings("unused")
    private ConnectOptions() {
        // Make non-null to avoid warning
        accessToken = "";
        params = new HashMap<>();
    }

    private ConnectOptions(Builder builder) {
        this.accessToken = builder.accessToken;
        this.audioTracks = (builder.audioTracks != null) ?
                builder.audioTracks : new ArrayList<LocalAudioTrack>();
        this.iceOptions = builder.iceOptions;
        this.preferredAudioCodecs = builder.preferredAudioCodecs;
        this.params = (builder.params == null) ? new HashMap<String, String>() : builder.params;
        this.platformInfo = new PlatformInfo();
        this.eventListener = builder.eventListener;
        this.enableDscp = builder.enableDscp;
        this.enableIceGatheringOnAnyAddressPorts = builder.enableIceGatheringOnAnyAddressPorts;
    }

    /**
     * The accessToken that provides the identity and grants of the caller.
     */
    @NonNull public String getAccessToken() {
        return accessToken;
    }

    /**
     * The parameters that are passed to the TwiML application specified by the access token.
     */
    @NonNull public Map<String, String> getParams() {
        return params;
    }

    /**
     * Returns the eventListener specified.
     */
    Call.EventListener getEventListener() {
        return eventListener;
    }

    /*
     * Invoked by JNI CallDelegate to get pointer to twilio::video::ConnectOptions::Builder
     */
    @SuppressWarnings("unused")
    private long createNativeConnectOptionsBuilder() {
        checkAudioTracksReleased(audioTracks);

        Pair<String[], String[]> paramsKeyValues =
                Utils.mapToArrays(params);

        return nativeCreate(accessToken,
                paramsKeyValues.first,
                paramsKeyValues.second,
                getLocalAudioTracksArray(),
                iceOptions,
                enableDscp,
                enableIceGatheringOnAnyAddressPorts,
                getAudioCodecsArray(),
                platformInfo);
    }

    private native long nativeCreate(String accessToken,
                                     String[] paramKeys,
                                     String[] paramValues,
                                     LocalAudioTrack[] audioTracks,
                                     IceOptions iceOptions,
                                     boolean enableDscp,
                                     boolean enableIceGatheringOnAnyAddressPorts,
                                     AudioCodec[] preferredAudioCodecs,
                                     PlatformInfo platformInfo);

    /**
     * Build new {@link ConnectOptions}.
     *
     * <p>All methods are optional.</p>
     */
    public static class Builder extends CallOptions.Builder {
        private final String accessToken;
        private Map<String, String> params;
        /**
         * Use this Builder when making a {@link Call}
         *
         * <p>
         * The maximum number of characters for the identity provided in the token is 121. The identity
         * may only contain alpha-numeric and underscore characters. Other characters, including spaces,
         * or exceeding the maximum number of characters, will result in not being able to place or receive
         * calls.
         * </p>
         *
         * @param accessToken The accessToken that provides the identity and grants of the caller in a JSON Web Token(JWT) format.
         */
        public Builder(@NonNull String accessToken) {
            Preconditions.checkNotNull(accessToken, "accessToken must not be null");
            this.accessToken = accessToken;
        }

        /**
         * Set the parameters that are passed to the TwiML application specified by the access token.
         *
         * <p>
         * Parameters specified in this map will be provided as HTTP GET query params or as part of the HTTP POST body
         * to the request url specified in your TwiML Application (https://www.twilio.com/console/voice/twiml/apps).
         * These parameters can be used to configure the behavior of your call within the context of your TwiML Application.
         *
         * NOTE: The Voice SDK URI encodes the parameters before passing them to your TwiML Application.
         * </p>
         */
        @NonNull public Builder params(@NonNull Map<String, String> params) {
            Preconditions.checkNotNull(params, "params must not be null");
            this.params = params;
            return this;
        }

        /**
         * Audio tracks that will be published upon connection.
         */
        @NonNull Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            super.audioTracks(audioTracks);
            return this;
        }

        /**
         * Custom ICE configuration used to connect to a Call.
         */
        @NonNull public Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            super.iceOptions(iceOptions);
            return this;
        }


        /**
         * Enable <a href="https://tools.ietf.org/html/draft-ietf-tsvwg-rtcweb-qos-18#section-5">Differentiated Services Field Code Point (DSCP)</a> with Expedited Forwarding (EF).
         *
         * @param enable Enable DSCP. Defaults to {@code true}.
         */
        @NonNull public Builder enableDscp(@NonNull boolean enable) {
            super.enableDscp(enable);
            return this;
        }

        /**
         * Enable gathering of ICE candidates on "any address" ports. When this flag is set, ports not
         * bound to any specific network interface will be used, in addition to normal ports bound to
         * the enumerated interfaces. This flag may need to be set to support certain network configurations
         * (e.g. some VPN implementations) where ports are not bound to specific interfaces. Setting
         * this flag means that additional candidates might need to be gathered and evaluated, which
         * could lead to slower ICE connection times for regular networks.
         *
         * @param enable Enable ICE candidates on "any address" ports. Defaults to {@code false}.
         */
        @NonNull public Builder enableIceGatheringOnAnyAddressPorts(@NonNull boolean enable) {
            super.enableIceGatheringOnAnyAddressPorts(enable);
            return this;
        }

        /**
         * Set preferred audio codecs. The list specifies which audio codecs would be preferred when
         * negotiating audio with the backend service. The preferences are applied in the order found in
         * the list starting with the most preferred audio codec to the least preferred audio codec.
         * Audio codec preferences are not guaranteed to be satisfied because the backend service
         * is not guaranteed to support all audio codecs.
         *
         * <p>The following snippet demonstrates how to prefer a single audio codec.
         *
         * <pre><code>
         *     ConnectOptions connectOptions = new ConnectOptions.Builder(token)
         *          .preferAudioCodecs(Collections.&lt;AudioCodec&gt;singletonList(new PcmuCodec()))
         *          .build();
         * </code></pre>
         *
         * <p>The following snippet demonstrates how to specify the exact order of codec
         * preferences.
         *
         * <pre><code>
         *     ConnectOptions connectOptions = new ConnectOptions.Builder(token)
         *          .preferAudioCodecs(Arrays.asList(new PcmuCodec(), new OpusCodec()))
         *          .build();
         * </code></pre>
         */
        @NonNull public Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(preferredAudioCodecs, "preferredAudioCodecs must not be null");
            checkAudioCodecs(preferredAudioCodecs);
            super.preferAudioCodecs(preferredAudioCodecs);
            return this;
        }

        Builder eventListener(Call.EventListener eventListener) {
            this.eventListener = eventListener;
            return this;
        }

        /**
         * Builds {@link ConnectOptions} object.
         */
        @NonNull public ConnectOptions build() {
            checkAudioTracksReleased(audioTracks);
            if (params != null) {
                for (Map.Entry<String, String> entry : params.entrySet()) {
                    Preconditions.checkState(entry.getKey() != null,"params entry key should not be null");
                    Preconditions.checkState(entry.getValue() != null, "params entry value should not be null");
                }
            }
            return new ConnectOptions(this);
        }
    }
}
