package com.twilio.voice;

import android.content.Context;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Handler;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import android.util.Pair;

import java.lang.reflect.Field;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;

import tvo.webrtc.voiceengine.WebRtcAudioManager;

/**
 * The Call class represents a signaling and media session between the host device and Twilio
 * infrastructure. Calls can represent an outbound call initiated from the SDK via
 * {@code Voice.connect(...)} or an incoming call accepted via a {@link CallInvite}. Devices that
 * initiate an outbound call represent the <i>caller</i> and devices that receive a
 * {@link CallInvite} represent the <i>callee</i>.
 * <p>
 * The lifecycle of a call differs for the caller and the callee. Making or accepting a call
 * requires a {@link Call.Listener} which provides {@link Call} state changes defined in
 * {@link Call.State}. The {@link Call.State} can be obtained at any time by calling
 * {@link Call#getState()}.
 * <p>
 * A subset of Android devices provide an <a href="https://www.khronos.org/opensles/">OpenSLES</a>
 * implementation. Although more efficient, the use of OpenSLES occasionally results in echo.
 * As a result, the Voice SDK disables OpenSLES by default for both incoming and outgoing
 * {@link Call}s unless explicitly enabled. To enable OpenSLES execute the following before invoking
 * {@link Voice#connect(Context, ConnectOptions, Listener)} or
 * {@link CallInvite#accept(Context, Listener)}:
 * <p>
 * {@code tvo.webrtc.voiceengine.WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(false)}
 * <p>
 * The following table provides expected call sequences for common scenarios from the caller
 * perspective. The caller scenario sequences are affected by the {@code answerOnBridge} flag provided
 * in the {@code Dial} verb of your TwiML application associated with the client. If the
 * {@code answerOnBridge} flag is {@code false}, which is the default, the
 * {@link Listener#onConnected(Call)} callback will be emitted immediately after
 * {@link Listener#onRinging(Call)}. If the {@code answerOnBridge} flag is {@code true} this will
 * cause the call to emit the {@link Listener#onConnected(Call)} callback only until the call is
 * answered by the callee.
 * See the <a href="https://www.twilio.com/docs/voice/twiml/dial#answeronbridge">Answer on Bridge documentation</a> for more details on how to use
 * it with the {@code Dial} TwiML verb.
 *
 * <table border="1">
 * <caption>Caller Call Sequences</caption>
 * <tr>
 *  <td>Caller Scenario</td>
 *  <td>{@link Call.Listener} callbacks</td>
 *  <td>{@link Call.State} transitions</td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the caller disconnects before reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated, reaches the {@link State#RINGING} state, and the caller disconnects before being {@link State#CONNECTED}.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onRinging(Call)} </li>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#RINGING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the {@link Call} fails to connect before reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the {@link Call} fails to connect after reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onRinging(Call)} </li>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#RINGING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated, becomes {@link State#CONNECTED}, and ends for one of the following reasons:
 *      <ul>
 *          <li>The caller disconnects the {@link Call}</li>
 *          <li>The callee disconnects the {@link Call}</li>
 *          <li>An on-device error occurs. (eg. network loss)</li>
 *          <li>An error between the caller and callee occurs. (eg. media connection lost or Twilio infrastructure failure)</li>
 *      </ul>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link Listener#onRinging(Call)}</li>
 *          <li>{@link Listener#onConnected(Call)}</li>
 *          <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *      </ol>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link State#CONNECTING}</li>
 *          <li>{@link State#RINGING}</li>
 *          <li>{@link State#CONNECTED}</li>
 *          <li>{@link State#DISCONNECTED}</li>
 *      </ol>
 *  </td>
 * </tr>
 * </table>
 * <p>
 * The following table provides expected call sequences for common scenarios from the callee
 * perspective:
 *
 * <table border="1">
 * <caption>Callee Call Sequences</caption>
 * <tr>
 * <td>Callee Scenario</td><td>{@link Call.Listener} callbacks</td><td>{@link Call.State} transitions</td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted and the callee disconnects the {@link Call} before being {@link State#CONNECTED}.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted and the {@link Call} fails to reach the {@link State#CONNECTED} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted, becomes {@link State#CONNECTED}, and ends for one of the following reasons:
 *      <ul>
 *          <li>The caller disconnects the {@link Call}</li>
 *          <li>The callee disconnects the {@link Call}</li>
 *          <li>An on-device error occurs. (eg. network loss)</li>
 *          <li>An error between the caller and callee occurs. (eg. media connection lost or Twilio infrastructure failure)</li>
 *      </ul>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link Listener#onConnected(Call)}</li>
 *          <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *      </ol>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link State#CONNECTING}</li>
 *          <li>{@link State#CONNECTED}</li>
 *          <li>{@link State#DISCONNECTED}</li>
 *      </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is rejected.</td>
 *  <td>Reject will not result in any callbacks.</td>
 *  <td>The call invite was not accepted, therefore the {@link Call.State} is not available.</td>
 * </tr>
 * </table>
 *

 * <p>
 * It is important to call the {@link #disconnect()} method to terminate the call. Not calling
 * {@link #disconnect()} results in leaked native resources and may lead to an out-of-memory crash.
 * If the call is disconnected by the caller, callee, or Twilio, the SDK will automatically free
 * native resources. However, {@link #disconnect()} is an idempotent operation so it is best to
 * call this method when you are certain your application no longer needs the object.
 * <p>
 * It is strongly recommended that Call instances are created and accessed from a single application
 * thread. Accessing an instance from multiple threads may cause synchronization problems.
 * Listeners are called on the thread that created the Call instance, unless the thread that created
 * the Call instance does not have a Looper. In that case, the listener will be called on the application's
 * main thread.
 *
 * <p>
 * An ongoing call will automatically switch to a more preferred network type if one becomes available.
 * The following are the network types listed in preferred order: ETHERNET, LOOPBACK, WIFI, VPN, and
 * CELLULAR. For example, if a WIFI network becomes available whilst in a call that is using CELLULAR
 * data, the call will automatically switch to using the WIFI network.
 *
 * <p>
 * Media and signaling reconnections are handled independently. As a result, it is possible that
 * a single network change event produces consecutive `onReconnecting()` and `onReconnected()` callbacks.
 * For example, a network change may result in the following callback sequence.
 * <ul>
 * 	<li>onReconnecting() with `CallException` with `EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED` error code</li>
 * 	<li>onReconnected()</li>
 * 	<li>onReconnecting() with `CallException` with `EXCEPTION_MEDIA_CONNECTION_FAILED` error code</li>
 * 	<li>onReconnected()</li>
 * 	</ul>
 *
 *  <p>Note SDK will always raise a `onReconnecting()` callback followed by a `onReconnected()` callback when reconnection is successful.</p>
 *
 * 	During the reconnecting state operations such as {@link #hold(boolean)}, {@link #mute(boolean)}, or {@link #sendDigits(String)} will result in
 * 	undefined behavior. When building applications, the recommended practice is to show a UI update
 * 	when the Call enters reconnecting regardless of the cause, and then clear the UI update once the
 * 	call has reconnected or disconnected.
 *
 * <p>Insights :</p>
 *
 * <p>The following connection events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights connection events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>muted</td><td>The audio input of the {@link Call} is muted by calling {@link Call#mute(boolean)} with {@code true}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>unmuted</td><td>The audio input of the {@link Call} is unmuted by calling {@link Call#mute(boolean)} with {@code false}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>accepted-by-remote</td><td>The server returned an answer to the offer over the signaling channel</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>ringing</td><td>The {@link Call} state transitions to {@link Call.State#RINGING}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>connected</td><td>The {@link Call} state transitions to {@link Call.State#CONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnect-called</td><td>The {@link Call#disconnect()} method was called</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnected-by-local</td><td>The {@link Call} disconnected as a result of calling {@link Call#disconnect()}. {@link Call.State} transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnected-by-remote</td><td>The {@link Call} was disconnected by the server. {@link Call.State} transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>hold</td><td>The {@link Call} is on hold by calling {@link Call#hold(boolean)}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>unhold</td><td>The {@link Call} is unhold by calling {@link Call#hold(boolean)}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>error</td><td>Error description. The Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>reconnecting</td><td>The {@link Call} is attempting to recover from a signaling or media connection interruption</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>reconnected</td><td>The {@link Call} has recovered from a signaling or media connection interruption</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following ICE candidate event is reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights ICE candidate event</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>ice-candidate</td><td>ice-candidate</td><td>ICE candidate events are raised when OnIceCandidate is called on the PeerConnection</td><td>4.1.0</td>
 * </tr>
 * <tr>
 * <td>ice-candidate</td><td>selected-ice-candidate-pair</td><td>The active local ICE candidate and remote ICE candidate</td><td>5.5.0</td>
 * </tr>
 * </table>
 *
 * <p>The following ICE connection state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights ICE connection events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>new</td><td>The PeerConnection ice connection state changed to "new"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>checking</td><td>The PeerConnection ice connection state changed to "checking"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>connected</td><td>The PeerConnection ice connection state changed to "connected"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>completed</td><td>The PeerConnection ice connection state changed to "completed"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>closed</td><td>The PeerConnection ice connection state changed to “closed”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>disconnected</td><td>The PeerConnection ice connection state changed to “disconnected”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>failed</td><td>The PeerConnection ice connection state changed to “failed”</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following ICE gathering state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights ICE gathering events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>new</td><td>The PeerConnection ice gathering state changed to "new"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>gathering</td><td>The PeerConnection ice gathering state changed to "checking"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>complete</td><td>The PeerConnection ice gathering state changed to "connected"</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following signaling state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights peer connection signaling state events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>stable</td><td>The PeerConnection signaling state changed to "stable"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-local-offer</td><td>The PeerConnection signaling state changed to "have-local-offer"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-remote-offers</td><td>The PeerConnection signaling state changed to "have-remote-offers"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-local-pranswer</td><td>The PeerConnection signaling state changed to "have-local-pranswer"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-remote-pranswer</td><td>The PeerConnection signaling state changed to “have-remote-pranswer”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>closed</td><td>The PeerConnection signaling state changed to “closed”</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following peer connection state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights peer connection state events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>new</td><td>The PeerConnection state is "new"</td><td>5.4.0</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>connecting</td><td>The PeerConnection state changed to "connecting"</td><td>5.4.0</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>connected</td><td>The PeerConnection state changed to "connected"</td><td>5.4.0</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>disconnected</td><td>Raised when peer connection state is disconnected</td><td>5.4.0</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>failed</td><td>Raised when peer connection state is failed</td><td>5.4.0</td>
 * </tr>
 * <tr>
 * <td>pc-connection-state</td><td>closed</td><td>Raised when peer connection state is closed</td><td>5.4.0</td>
 * </tr>
 * </table>
 *
 * <p>The following network quality warning and network quality warning cleared events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights network quality events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-jitter</td><td>Three out of last five jitter samples exceed 30 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-jitter</td><td>high-jitter warning cleared if last five jitter samples are less than 30 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>low-mos</td><td>Three out of last five mos samples are lower than 3</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>low-mos</td><td>low-mos cleared if last five mos samples are higher than 3</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-packet-loss</td><td>Three out of last five packet loss samples show loss greater than 1%</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-packet-loss</td><td>high-packet-loss cleared if last five packet loss samples are lower than 1%</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-rtt</td><td>Three out of last five RTT samples show greater than 400 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-rtt</td><td>high-rtt warning cleared if last five RTT samples are lower than 400 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following audio level warning and audio level warning cleared events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights audio level events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-raised</td><td>constant-audio-input-level</td><td>Last ten audio input samples have the same audio level</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-cleared</td><td>constant-audio-input-level</td><td>constant-audio-input-level warning cleared if the current audio input level sample differs from the previous audio input level sample</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-cleared</td><td>constant-audio-output-level</td><td>constant-audio-output-level warning cleared if the current audio output level sample differs from the previous audio output level sample</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following feedback events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1">
 * <caption>Insights feedback events</caption>
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>feedback</td><td>received</td><td>{@link Call#postFeedback(Score, Issue)} is called with a score other than {@link Call.Score#NOT_REPORTED} or an issue other than {@link Call.Issue#NOT_REPORTED} i</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>feedback</td><td>received-none</td><td>{@link Call#postFeedback(Score, Issue)} is called with {@link Call.Score#NOT_REPORTED} and {@link Call.Issue#NOT_REPORTED}</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 */
public class Call extends InternalCall {
    /*
     * The following fields are statically initialized so the SDK only pays a one time cost of
     * fetching the private members of WebRtcAudioManager. These fields are used to apply the
     * OpenSLES configuration policy.
     */
    private static final Field blacklistDeviceForOpenSLESUsageField;
    private static final Field blacklistDeviceForOpenSLESUsageIsOverriddenField;
    static {
        try {
            blacklistDeviceForOpenSLESUsageField =
                    WebRtcAudioManager.class.getDeclaredField("blacklistDeviceForOpenSLESUsage");
            blacklistDeviceForOpenSLESUsageIsOverriddenField =
                    WebRtcAudioManager.class.getDeclaredField("blacklistDeviceForOpenSLESUsageIsOverridden");
            blacklistDeviceForOpenSLESUsageField.setAccessible(true);
            blacklistDeviceForOpenSLESUsageIsOverriddenField.setAccessible(true);
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e.getMessage());
        }
    }
    private final ThreadUtils.ThreadChecker threadChecker;
    private Listener listener;
    private EventListener eventListener;
    private static final Logger logger = Logger.getLogger(Call.class);
    private List<LocalAudioTrack> localAudioTracks = Collections.emptyList();
    private MediaFactory mediaFactory;
    private long nativeCallDelegate;
    private Queue<Pair<Handler, StatsListener>> statsListenersQueue;
    private ConnectivityReceiver connectivityReceiver = null;
    private Set<CallQualityWarning> currentCallQualityWarning;

    /**
     * An enum representing call quality score.
     */
    public enum Score {
        /**
         * No score reported.
         */
        NOT_REPORTED(0),
        /**
         * Terrible call quality, call dropped, or caused great difficulty in communicating.
         */
        ONE(1),
        /**
         * Bad call quality, like choppy audio, periodic one-way-audio.
         */
        TWO(2),
        /**
         * Average call quality, manageable with some noise/minor packet loss.
         */
        THREE(3),
        /**
         * Good call quality, minor issues.
         */
        FOUR(4),
        /**
         * Great call quality. No issues.
         */
        FIVE(5);

        private final int score;

        private Score(int score) {
            this.score = score;
        }

        public int getValue() {
            return this.score;
        }
    }

    /**
     * An enum representing issue type associated with a call.
     */
    public enum Issue {
        /**
         * No issue reported.
         */
        NOT_REPORTED("not-reported"),
        /**
         * Call initially connected but was dropped.
         */
        DROPPED_CALL("dropped-call"),
        /**
         * Participants can hear each other but with significant delay.
         */
        AUDIO_LATENCY("audio-latency"),
        /**
         * One participant couldn’t hear the other.
         */
        ONE_WAY_AUDIO("one-way-audio"),
        /**
         * Periodically, participants couldn’t hear each other. Some words were lost.
         */
        CHOPPY_AUDIO("choppy-audio"),
        /**
         * There was disturbance, background noise, low clarity.
         */
        NOISY_CALL("noisy-call"),
        /**
         * There was echo during call.
         */
        ECHO("echo");

        private final String issueName;

        private Issue(String issueName) {
            this.issueName = issueName;
        }

        public String toString() {
            return this.issueName;
        }
    }

    public enum CallQualityWarning {
        /* High Round Trip Time warning */
        WARN_HIGH_RTT("high-rtt"),
        /* High Jitter warning */
        WARN_HIGH_JITTER("high-jitter"),
        /* High Packets lost fraction warning */
        WARN_HIGH_PACKET_LOSS("high-packet-loss"),
        /* Low Mean Opinion Score warning */
        WARN_LOW_MOS("low-mos"),
        /* Constant audio in level warning */
        WARN_CONSTANT_AUDIO_IN_LEVEL("constant-audio-input-level");

        private final String warningName;

        CallQualityWarning(String warningName) {
            this.warningName = warningName;
        }

        public String toString() {
            return this.warningName;
        }

        public static CallQualityWarning fromString(String warningName) {
            if (warningName.equals(WARN_HIGH_RTT.warningName)) {
                return WARN_HIGH_RTT;
            } else if (warningName.equals(WARN_HIGH_JITTER.warningName)) {
                return WARN_HIGH_JITTER;
            }  else if (warningName.equals(WARN_HIGH_PACKET_LOSS.warningName)) {
                return WARN_HIGH_PACKET_LOSS;
            }  else if (warningName.equals(WARN_LOW_MOS.warningName)) {
                return WARN_LOW_MOS;
            }  else if (warningName.equals(WARN_CONSTANT_AUDIO_IN_LEVEL.warningName)) {
                return WARN_CONSTANT_AUDIO_IN_LEVEL;
            } else {
                throw new RuntimeException("Unsupported warning name string -> " + warningName);
            }
        }
    }

    /*
     * The contract for Call JNI callbacks is as follows:
     *
     * 1. All event callbacks are done on the same thread the developer used to connect to a call.
     * 2. Create and release all native memory on the same thread. In the case of a Call, the
     * CallDelegate is created and released on the developer thread and the native call and call
     * observer are created and released on notifier thread.
     * 3. All Call fields must be mutated on the developer's thread.
     *
     * Not abiding by this contract, may result in difficult to debug JNI crashes,
     * incorrect return values in the synchronous API methods, or missed callbacks.
     */
    private final Call.Listener callListenerProxy = new Call.Listener() {
        @Override
        public void onRinging(@NonNull final Call call) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onRinging()");

                    // Update call state
                    Call.this.state = State.RINGING;
                    call.sid = nativeGetSid(nativeCallDelegate);
                    Call.this.listener.onRinging(call);
                }
            });
        }

        @Override
        public void onConnected(@NonNull final Call call) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onConnected()");

                    // Update call state
                    Call.this.state = State.CONNECTED;
                    call.sid = nativeGetSid(nativeCallDelegate);
                    Call.this.listener.onConnected(call);
                }
            });
        }

        @Override
        public void onReconnecting(@NonNull Call call, @NonNull CallException callException) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onReconnecting()");

                    // Update call state
                    Call.this.state = State.RECONNECTING;
                    Call.this.listener.onReconnecting(call, callException);
                }
            });
        }

        @Override
        public void onReconnected(@NonNull Call call) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onReconnected()");

                    // Update call state
                    Call.this.state = State.CONNECTED;
                    Call.this.listener.onReconnected(call);
                }
            });
        }

        @Override
        public void onConnectFailure(@NonNull final Call call, @NonNull final CallException callException) {

            handler.post(new Runnable() {
                @Override
                public void run() {
                    // Release native call
                    releaseCall();

                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onConnectFailure()");

                    unregisterConnectivityBroadcastReceiver(context);
                    Voice.calls.remove(Call.this);
                    Voice.rejects.remove(Call.this);

                    // Update call state
                    Call.this.state = State.DISCONNECTED;

                    // Release native call delegate
                    release();

                    // Notify developer
                    Call.this.listener.onConnectFailure(call, callException);
                }
            });
        }

        @Override
        public void onDisconnected(@NonNull final Call call, final CallException callException) {

            handler.post(new Runnable() {
                @Override
                public void run() {
                    // Release native call
                    releaseCall();

                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onDisconnected()");

                    unregisterConnectivityBroadcastReceiver(context);
                    Voice.calls.remove(Call.this);
                    Voice.rejects.remove(Call.this);

                    // Update call state
                    Call.this.state = State.DISCONNECTED;

                    // Release native call delegate
                    release();

                    // Notify developer
                    Call.this.listener.onDisconnected(call, callException);
                }
            });
        }

        @Override
        public void onCallQualityWarningsChanged(Call call, Set<CallQualityWarning> currentWarnings, Set<CallQualityWarning> previousWarnings) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onCallQualityWarningsChanged()");
                    Call.this.currentCallQualityWarning = currentWarnings;
                    Call.this.listener.onCallQualityWarningsChanged(call, currentWarnings, previousWarnings);
                }
            });
        }
    };

    Call.EventListener eventListenerProxy = new Call.EventListener() {

        @Override
        public void onEvent(Map<String, Pair<String, Class>> data) {

            handler.post((new Runnable() {
                @Override
                public void run() {
                    if (Call.this.eventListener != null) {
                        Call.this.eventListener.onEvent(data);
                    }

                    if (data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.AUDIO_LEVEL_WARNING_RAISED) ||
                            data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.NETWORK_QUALITY_WARNING_RAISED)) {
                        InsightsUtils.processWarningEvent(data, createEventPayloadBuilder(), publisher);
                    } else {
                        if (data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.CONNECTION_EVENT_GROUP)
                                && (data.get(EventKeys.EVENT_NAME).first.equals(EventType.RINGING_EVENT))) {
                            sid = data.get(EventKeys.CALL_SID_KEY).first;
                        }
                        if (data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.SETTINGS_GROUP)) {
                            if (data.get(EventKeys.EVENT_NAME).first.equals(EventType.CODEC_EVENT)) {
                                codecParams = data.get(EventKeys.CODEC_PARAMS).first;
                                selectedCodec = data.get(EventKeys.SELECTED_CODEC).first;
                            } else if (data.get(EventKeys.EVENT_NAME).first.equals(EventType.EDGE_EVENT)) {
                                gateway = (data.get(EventKeys.EDGE_HOST_NAME)).first;
                                region = (data.get(EventKeys.EDGE_HOST_REGION)).first;
                            }
                            InsightsUtils.processEvent(data, createEventPayloadBuilderForSettingsEvent(), publisher, direction);
                        } else {
                            InsightsUtils.processEvent(data, createEventPayloadBuilder(), publisher, direction);
                        }
                    }
                }
            }));
        }

        @Override
        public void onMetric(Map<String, Pair<String, Class>> data) {
            if (Call.this.eventListener != null) {
                Call.this.eventListener.onMetric(data);
            }

            if (data.get(EventKeys.EVENT_GROUP).first.equals(EventGroupType.CALL_QUALITY_STATS_GROUP)) {
                onSample(InsightsUtils.createRtcSample(data));
            }
        }
    };

    /**
     * An enum describing the possible states of a Call.
     */
    public enum State {
        /**
         * The {@link Call} was created or was accepted and is in the process of connecting.
         */
        CONNECTING,
        /**
         * The {@link Call} is ringing.
         */
        RINGING,
        /**
         * The {@link Call} is connected.
         */
        CONNECTED,
        /**
         * The {@link Call} is reconnecting.
         */
        RECONNECTING,
        /**
         * The {@link Call} was disconnected, either due to a disconnect or an error.
         */
        DISCONNECTED
    }

    /**
     * Call.Listener interface defines a set of callbacks for events related to
     * call.
     *
     * @see Call Reference the call overview for a reference of callback sequences for making and
     * receiving calls.
     */
    public interface Listener {
        /**
         * The call failed to connect.
         * <p>
         * Calls that fail to connect will result in {@link Call.Listener#onConnectFailure(Call, CallException)}
         * and always return a {@link CallException} providing more information about what failure occurred.
         * </p>
         *
         * @param call          An object model representing a call that failed to connect.
         * @param callException CallException that describes why the connect failed.
         */
        void onConnectFailure(@NonNull Call call, @NonNull CallException callException);

        /**
         * Emitted once before the {@link Call.Listener#onConnected(Call)} callback. If
         * {@code answerOnBridge} is true, this represents the callee being alerted of a call.
         *
         * The {@link Call#getSid()} is now available.
         *
         * @param call  An object model representing a call.
         */
        void onRinging(@NonNull Call call);

        /**
         * The call has connected.
         *
         * @param call An object model representing a call.
         */
        void onConnected(@NonNull Call call);

        /**
         * The call starts reconnecting.
         *
         * Reconnect is triggered when a network change is detected and Call is already in {@link Call.State#CONNECTED} state.
         * If the call is in {@link Call.State#CONNECTING} or in {@link Call.State#RINGING} when network
         * change happened the SDK will continue attempting to connect, but a reconnect event will not be raised.
         *
         * @param call           An object model representing a call.
         * @param callException  CallException that describes the reconnect reason. This would have one of the two
         * possible values with error codes 53001 "Signaling connection disconnected" and 53405 "Media connection failed".
         */
        void onReconnecting(@NonNull Call call, @NonNull CallException callException);

        /**
         * The call is reconnected.
         *
         * @param call An object model representing a call.
         */
         void onReconnected(@NonNull Call call);

        /**
         * The call was disconnected.
         * <p>
         * A call can be disconnected for the following reasons:
         * <ul>
         * <li>A user calls `disconnect()` on the `Call` object.</li>
         * <li>The other party disconnects or terminates the call.</li>
         * <li>An error occurs on the client or the server that terminates the call.</li>
         * </ul>
         * <p>
         * If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.
         * </p>
         *
         * @param call          An object model representing a call.
         * @param callException CallException that caused the call to disconnect.
         */
        void onDisconnected(@NonNull Call call, @Nullable CallException callException);

        /**
         * Emitted when network quality warnings have changed for the {@link Call}.
         *
         * <p>
         * The trigger conditions for the {@link CallQualityWarning}s are defined as below:
         *
         * <ol>
         * <li> {@link CallQualityWarning#WARN_HIGH_RTT} - Round Trip Time (RTT) &gt; 400 ms for 3 out of last 5 samples.</li>
         * <li> {@link CallQualityWarning#WARN_HIGH_JITTER}  - Jitter &gt; 30 ms for 3 out of last 5 samples.</li>
         * <li> {@link CallQualityWarning#WARN_HIGH_PACKET_LOSS}  - Packet loss &gt; 1% in 3 out of last 5 samples.</li>
         * <li> {@link CallQualityWarning#WARN_LOW_MOS}  - Mean Opinion Score (MOS) &lt; 3.5 for 3 out of last 5 samples.</li>
         * <li> {@link CallQualityWarning#WARN_CONSTANT_AUDIO_IN_LEVEL} ` - Audio input level is unchanged for 10 seconds and call is not in muted state.</li>
         * </ol>
         * The two sets help to determine what warnings have changed since the last callback was received.
         *
         *
         * @param call              An object model representing a call.
         * @param currentWarnings   A {@link Set} that contains the current {@link CallQualityWarning}s.
         * @param previousWarnings  A {@link Set} that contains the previous {@link CallQualityWarning}s.
         */
        default void onCallQualityWarningsChanged(@NonNull Call call,
                                                  @NonNull Set<CallQualityWarning> currentWarnings,
                                                  @NonNull Set<CallQualityWarning> previousWarnings) {}
    }

    interface EventListener {
        void onEvent(Map<String, Pair<String, Class>> data);
        void onMetric(Map<String, Pair<String, Class>> data);
    }

    private final StatsListener statsListenerProxy = new StatsListener() {
        @Override
        public void onStats(final List<StatsReport> statsReports) {
            final Pair<Handler, StatsListener> statsPair = Call.this.statsListenersQueue.poll();
            if (statsPair != null) {
                statsPair.first.post(new Runnable() {
                    @Override
                    public void run() {
                        statsPair.second.onStats(statsReports);
                    }
                });
            }
        }
    };

    Call(final Context context, final CallInvite callInvite, final Listener listener) {
        Preconditions.checkApplicationContext(context, "must create Call with application context");
        this.context = context;
        this.listener = listener;
        this.from = callInvite.getFrom();
        this.to = callInvite.getTo();
        this.sid = callInvite.getCallSid();
        this.bridgeToken = callInvite.getBridgeToken();
        this.disconnectCalled = false;
        this.direction = Constants.Direction.INCOMING;
        this.handler = Utils.createHandler();
        this.threadChecker = new ThreadUtils.ThreadChecker(handler.getLooper().getThread());
        this.state = State.CONNECTING;
        this.publisher = new EventPublisher(context, Constants.CLIENT_SDK_PRODUCT_NAME,
                bridgeToken);
        this.publisher.addListener(this);
        this.statsListenersQueue = new ConcurrentLinkedQueue<>();
        configureOpenSLES();
    }

    Call(final Context context, final String accessToken, final Listener listener) {
        Preconditions.checkApplicationContext(context, "must create Call with application context");
        this.context = context;
        this.listener = listener;
        this.state = State.CONNECTING;
        this.direction = Constants.Direction.OUTGOING;
        this.handler = Utils.createHandler();
        this.threadChecker = new ThreadUtils.ThreadChecker(handler.getLooper().getThread());
        this.publisher = new EventPublisher(context, Constants.CLIENT_SDK_PRODUCT_NAME, accessToken);
        this.publisher.addListener(this);
        this.statsListenersQueue = new ConcurrentLinkedQueue<>();
        configureOpenSLES();
    }

    /**
     * Returns the caller information when available. The from field is {@code null} for an outgoing call
     * and may be {@code null} if it was not provided in the {@link CallInvite} for an incoming call.
     */
    @Nullable public String getFrom() {
        return from;
    }

    /**
     * Returns the callee information when available. Returns null for an outgoing call.
     */
    @Nullable public String getTo() {
        return to;
    }

    /**
     * Returns the call sid. The call sid is null until the call is in {@link State#RINGING} state.
     */
    @Nullable public String getSid() {
        return sid;
    }

    /**
     * Returns the current state of the call.
     *
     * <p>Call is in {@link State#CONNECTING} state when it is made or accepted.</p>
     * <p>Call is in {@link State#RINGING} state when it is ringing.</p>
     * <p>Call transitions to {@link State#CONNECTED} state when connected to Twilio.</p>
     * <p>Call transitions to {@link State#DISCONNECTED} state when disconnected.</p>
     *
     * @return Provides the state of this call.
     */
    @Override
    @NonNull public State getState() {
        return state;
    }

    /**
     * Retrieve stats for all media tracks and notify {@link StatsListener} via calling thread.
     * The call needs to be in {@link State#CONNECTED} state for reports to be delivered.
     *
     * @param statsListener listener that receives stats reports for all media tracks.
     */
    public synchronized void getStats(@NonNull StatsListener statsListener) {
        threadChecker.checkIsOnValidThread();

        Preconditions.checkNotNull(statsListener, "statsListener must not be null");
        if (state == State.DISCONNECTED) {
            return;
        }
        statsListenersQueue.offer(new Pair<>(Utils.createHandler(), statsListener));
        nativeGetStats(nativeCallDelegate);
    }

    /**
     * Get current set of Call quality warnings.
     *
     * @return Provides the set of current {@link CallQualityWarning}.
     */
    public Set<CallQualityWarning> getCallQualityWarnings() {
        return this.currentCallQualityWarning;
    }

    /**
     * Posts the feedback collected for this call to Twilio. If `0` `Score` and `not-reported`
     * `Issue` are passed, Twilio will report feedback was not available for this call.
     *
     * @param score - the call quality score.
     * @param issue - the issue type associated with the call.
     */
    public void postFeedback(@NonNull Score score, @NonNull Issue issue) {
        Preconditions.checkNotNull(score, "score must not be null");
        Preconditions.checkNotNull(issue, "issue must not be null");
        publishFeedbackEvent(score, issue);
    }

    void connect(final ConnectOptions connectOptions) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.calls.add(this);

        // Check if audio or video tracks have been released
        ConnectOptions.checkAudioTracksReleased(connectOptions.getAudioTracks());

        localAudioTracks = connectOptions.getAudioTracks();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            mediaFactory = MediaFactory.instance(this, context);
            if (connectOptions.getEventListener() != null) {
                this.eventListener = connectOptions.getEventListener();
            }
            nativeCallDelegate = nativeConnect(connectOptions,
                    callListenerProxy,
                    statsListenerProxy,
                    eventListenerProxy,
                    mediaFactory.getNativeMediaFactoryHandle(),
                    handler);
        }
    }

    void accept(final AcceptOptions acceptOptions, final long nativeCallInviteProxy) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.calls.add(this);

        // Check if audio or video tracks have been released
        AcceptOptions.checkAudioTracksReleased(acceptOptions.getAudioTracks());

        localAudioTracks = acceptOptions.getAudioTracks();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            mediaFactory = MediaFactory.instance(this, context);
            nativeCallDelegate = nativeAccept(acceptOptions,
                    callListenerProxy,
                    statsListenerProxy,
                    eventListenerProxy,
                    handler,
                    nativeCallInviteProxy);
        }
    }

    /*
     * Synchronize accesses to call listener during initialization and make
     * sure that onConnect() callback won't get called before connect() exits and Call
     * creation is fully completed.
     */
    void reject(final AcceptOptions acceptOptions, final long nativeCallInviteProxy) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.rejects.add(this);

        localAudioTracks = acceptOptions.getAudioTracks();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            mediaFactory = MediaFactory.instance(this, context);
            nativeCallDelegate = nativeReject(acceptOptions,
                    callListenerProxy,
                    eventListenerProxy,
                    handler,
                    nativeCallInviteProxy);
        }
    }

    /**
     * Mutes or unmutes the audio input.
     */
    public synchronized void mute(final boolean mute) {
        threadChecker.checkIsOnValidThread();
        if (isValidState()) {
            isMuted = mute;
            nativeMute(nativeCallDelegate, mute);
        }
    }

    /**
     * Sends a string of DTMF digits.
     *
     * @param digits A string of digits to be sent. Valid values are "0" - "9", "*", "#", and "w". Each "w" will cause a 500 ms pause between digits sent.
     */
    public synchronized void sendDigits(@NonNull String digits) {
        threadChecker.checkIsOnValidThread();
        Preconditions.checkNotNull(digits, "digits must not be null");
        if (!digits.matches("^[0-9\\*\\#w]+$")) {
            throw new IllegalArgumentException("digits string must not be null and should only contains 0-9, *, #, or w characters");
        }
        if (isValidState()) {
            nativeSendDigits(nativeCallDelegate, digits);
        }
    }

    /**
     * Holds or un-holds the audio.
     */
    public synchronized void hold(final boolean hold) {
        threadChecker.checkIsOnValidThread();
        if (isValidState()) {
            isOnHold = hold;
            nativeHold(nativeCallDelegate, hold);
        }
    }

    /**
     * Reports whether the audio input is muted.
     */
    public boolean isMuted() {
        return isMuted;
    }

    /**
     * Reports whether the call is on hold.
     */
    public boolean isOnHold() {
        return isOnHold;
    }

    /**
     * Disconnects the Call.
     * <p>
     * Invoking disconnect will result in a subsequent {@link Listener#onDisconnected(Call, CallException)}
     * being raised unless any of the following conditions are true:
     * <p>
     * <ul>
     *   <li>Disconnect was already called on the call object. In this case, only one {@link Listener#onDisconnected(Call, CallException)} will be raised</li>
     *   <li>The call is already {@link State#DISCONNECTED}</li>
     *   <li>The call has been cancelled. In this case, the application has already received a {@link MessageListener#onCancelledCallInvite(CancelledCallInvite, CallException)} with a {@link CancelledCallInvite} with the same call SID of the call object</li>
     * </ul>
     */
    @Override
    public synchronized void disconnect() {
        threadChecker.checkIsOnValidThread();

        if (!disconnectCalled && isValidState() && nativeCallDelegate != 0) {
            disconnectCalled = true;
            logger.d("Calling disconnect " + state);
            nativeDisconnect(nativeCallDelegate);
        }
    }

    void networkChange(Voice.NetworkChangeEvent networkChangeEvent) {
        threadChecker.checkIsOnValidThread();
        if (isValidState() && isPermittedNetworkChangeEvent(networkChangeEvent)) {
            nativeNetworkChange(nativeCallDelegate, networkChangeEvent);
        } else {
            logger.d("Ignoring networkChangeEvent: " + networkChangeEvent.name() +
                    " in Call.State: " + state);
        }
    }

    /*
     * Release the native CallDelegate from developer thread once the native Call memory
     * has been released.
     */
    synchronized void release() {
        this.threadChecker.checkIsOnValidThread();

        for (LocalAudioTrack localAudioTrack : localAudioTracks) {
            localAudioTrack.release();
        }

        if (nativeCallDelegate != 0) {
            nativeRelease(nativeCallDelegate);
            nativeCallDelegate = 0;
            if (mediaFactory != null) {
                mediaFactory.release(this);
            }
        }
    }

    /*
     * Disable OpenSLES unless the developer has explicitly requested to enabled it via
     * WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(false).
     */
    @VisibleForTesting
    static void configureOpenSLES() {
        if (!openSLESEnabled()) {
            WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(true);
        }
    }

    @VisibleForTesting
    static boolean openSLESEnabled() {
        try {
            return !(boolean) blacklistDeviceForOpenSLESUsageField.get(null) &&
                    (boolean) blacklistDeviceForOpenSLESUsageIsOverriddenField.get(null);
        } catch (Exception e) {
            throw new RuntimeException("Failed to determine if OpenSLES is enabled.");
        }
    }

    private void registerConnectivityBroadcastReceiver(Context context) {
        connectivityReceiver = new ConnectivityReceiver();
        context.registerReceiver(connectivityReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
    }

    /*
     * A customer observed (CLIENT-5899) CONNECTION_CHANGED events handled before reaching
     * State.CONNECTED resulted in the Call being disconnected. The SDK has intentionally been
     * updated to ignore CONNECTION_CHANGED events while a Call is in CONNECTING or RINGING state to
     * prevent these premature Call disconnects in cases where the Call can connect despite the
     * network change occurring. This behavior is different from the iOS SDK and can result in the
     * following outcomes:
     *
     * 1. Ignoring the CONNECTION_CHANGED event results in the Call successfully being connected
     * 2. Ignoring the CONNECTION_CHANGED event results in the Call being disconnected. With this
     * behavior, the SDK could take more time to disconnect the Call than in previous releases.
     *
     * This change is considered a temporary workaround until the completion of the Mobile
     * Connection Robustness epic CLIENT-5756.
     */
    private boolean isPermittedNetworkChangeEvent(Voice.NetworkChangeEvent networkChangeEvent) {
        return !(networkChangeEvent == Voice.NetworkChangeEvent.CONNECTION_CHANGED &&
                (state == Call.State.CONNECTING || state == Call.State.RINGING));
    }

    private void unregisterConnectivityBroadcastReceiver(Context context) {
        context.unregisterReceiver(connectivityReceiver);
        connectivityReceiver = null;
    }

    private synchronized void releaseCall() {
        if (nativeCallDelegate != 0) {
            nativeReleaseCall(nativeCallDelegate);
        }
    }

    private native long nativeConnect(ConnectOptions ConnectOptions,
                                      Listener listenerProxy,
                                      StatsListener statsListenerProxy,
                                      EventListener eventListenerProxy,
                                      long nativeMediaFactoryHandle,
                                      Handler handler);

    private native long nativeAccept(AcceptOptions acceptOptions,
                                     Listener listenerProxy,
                                     StatsListener statsListenerProxy,
                                     EventListener eventListenerProxy,
                                     Handler handler,
                                     long nativeCallInviteProxy);

    private native long nativeReject(AcceptOptions acceptOptions,
                                     Listener listenerProxy,
                                     EventListener eventListenerProxy,
                                     Handler handler,
                                     long nativeCallInviteProxy);

    private native String nativeGetSid(long nativeCallDelegate);

    private native void nativeGetStats(long nativeCallDelegate);

    private native void nativeMute(long nativeCallDelegate, boolean mute);

    private native void nativeSendDigits(long nativeCallDelegate, String digits);

    private native void nativeHold(long nativeCallDelegate, boolean hold);

    private native void nativeDisconnect(long nativeCallDelegate);

    private native void nativeNetworkChange(long nativeCallDelegate,
                                            Voice.NetworkChangeEvent networkChangeEvent);

    private native void nativeReleaseCall(long nativeCallDelegate);

    private native void nativeRelease(long nativeCallDelegate);

}
