package com.twilio.voice;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents options when accepting a {@link CallInvite}.
 */
public class AcceptOptions extends CallOptions {
    private final Map<String, String> callInviteMessage;

    /**
     * AcceptOptions can only be constructed via {@link AcceptOptions.Builder}
     */
    @SuppressWarnings("unused")
    private AcceptOptions() {
        callInviteMessage = null;
    }

    private AcceptOptions(Builder builder) {
        this.audioTracks = (builder.audioTracks != null) ?
                builder.audioTracks : new ArrayList<LocalAudioTrack>();
        this.iceOptions = builder.iceOptions;
        this.enableDscp = builder.enableDscp;
        this.enableIceGatheringOnAnyAddressPorts = builder.enableIceGatheringOnAnyAddressPorts;
        this.preferredAudioCodecs = builder.preferredAudioCodecs;
        this.callInviteMessage = (builder.callInvite == null) ?
                new HashMap<String, String>() : builder.callInvite.callInviteMessage;
        this.platformInfo = new PlatformInfo();
    }

    /*
     * Invoked by JNI CallDelegate to get pointer to twilio::voice::AcceptOptions::Builder
     */
    @SuppressWarnings("unused")
    private long createNativeAcceptOptionsBuilder() {
        checkAudioTracksReleased(audioTracks);

        return nativeCreate(getLocalAudioTracksArray(),
                iceOptions,
                enableDscp,
                enableIceGatheringOnAnyAddressPorts,
                getAudioCodecsArray(),
                platformInfo);
    }

    private native long nativeCreate(LocalAudioTrack[] audioTracks,
                                     IceOptions iceOptions,
                                     boolean enableDscp,
                                     boolean enableIceGetheringOnAnyAddressPorts,
                                     AudioCodec[] preferredAudioCodecs,
                                     PlatformInfo platformInfo);

    /**
     * Build new {@link AcceptOptions}.
     * <p>All methods are optional.</p>
     */
    public static class Builder extends CallOptions.Builder {
        private final CallInvite callInvite;

        /**
         * Use this Builder when accepting a {@link CallInvite}
         */
        public Builder() {
            this.callInvite = null;
        }

        /**
         * This builder is used internally when accepting or rejecting a {@link CallInvite}
         */
        Builder(@NonNull CallInvite callInvite, boolean reject) {
            Preconditions.checkNotNull(callInvite);
            this.callInvite = callInvite;
        }

        /**
         * Audio tracks that will be published upon connection.
         */
        @NonNull Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            super.audioTracks(audioTracks);
            return this;
        }

        /**
         * Custom ICE configuration used to connect to a Call.
         */
        @NonNull public Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            super.iceOptions(iceOptions);
            return this;
        }

        /**
         * Enable <a href="https://tools.ietf.org/html/draft-ietf-tsvwg-rtcweb-qos-18#section-5">Differentiated Services Field Code Point (DSCP)</a> with Expedited Forwarding (EF).
         *
         * @param enable Enable DSCP. Defaults to {@code true}.
         */
        @NonNull public Builder enableDscp(@NonNull boolean enable) {
            super.enableDscp(enable);
            return this;
        }

        /**
         * Enable gathering of ICE candidates on "any address" ports. When this flag is set, ports not
         * bound to any specific network interface will be used, in addition to normal ports bound to
         * the enumerated interfaces. This flag may need to be set to support certain network configurations
         * (e.g. some VPN implementations) where ports are not bound to specific interfaces. Setting
         * this flag means that additional candidates might need to be gathered and evaluated, which
         * could lead to slower ICE connection times for regular networks.
         *
         * @param enable Enable ICE candidates on "any address" ports. Defaults to {@code false}.
         */
        @NonNull public Builder enableIceGatheringOnAnyAddressPorts(@NonNull boolean enable) {
            super.enableIceGatheringOnAnyAddressPorts(enable);
            return this;
        }

        /**
         * Set preferred audio codecs. The list specifies which audio codecs would be preferred when
         * negotiating audio with the backend service. The preferences are applied in the order found in
         * the list starting with the most preferred audio codec to the least preferred audio codec.
         * Audio codec preferences are not guaranteed to be satisfied because the backend service
         * is not guaranteed to support all audio codecs.
         *
         * <p>The following snippet demonstrates how to prefer a single audio codec.
         *
         * <pre><code>
         *     AcceptOptions acceptOptions = new AcceptOptions.Builder(token)
         *          .preferAudioCodecs(Collections.&lt;AudioCodec&gt;singletonList(new PcmuCodec()))
         *          .build();
         * </code></pre>
         *
         * <p>The following snippet demonstrates how to specify the exact order of codec
         * preferences.
         *
         * <pre><code>
         *     AcceptOptions acceptOptions = new AcceptOptions.Builder(token)
         *          .preferAudioCodecs(Arrays.asList(new PcmuCodec(), new OpusCodec()))
         *          .build();
         * </code></pre>
         */
        @NonNull public Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(preferredAudioCodecs,"preferredAudioCodecs must not be null");
            checkAudioCodecs(preferredAudioCodecs);
            super.preferAudioCodecs(preferredAudioCodecs);
            return this;
        }

        /**
         * Builds {@link AcceptOptions} object.
         */
        @NonNull public AcceptOptions build() {
            checkAudioTracksReleased(audioTracks);
            return new AcceptOptions(this);
        }
    }
}
