package com.twilio.video;

import android.os.Handler;
import androidx.annotation.NonNull;
import java.nio.ByteBuffer;

class AudioDeviceProxy extends AudioDeviceContext implements AudioDevice {
    private static final Logger logger = Logger.getLogger(AudioDeviceProxy.class);

    private static long nativeAudioDeviceProxyHandle;
    Handler handler;
    private AudioDevice audioDevice;
    private boolean released = false;
    private boolean capturingStopped = false;
    private boolean renderingStopped = false;

    // Created in the native layer
    private AudioDeviceProxy(long nativeAudioDeviceProxy, AudioDevice audioDevice) {
        this.nativeAudioDeviceProxyHandle = nativeAudioDeviceProxy;
        this.audioDevice = audioDevice;
    }

    public void writeCaptureData(ByteBuffer buffer) {
        if (!capturingStopped && !released) {
            nativeWriteData(nativeAudioDeviceProxyHandle, buffer, buffer.capacity());
        } else {
            logger.d(
                    "Ignoring writeCaptureData because either AudioDeviceProxy is released or "
                            + "capturing stopped");
        }
    }

    public void readRenderData(ByteBuffer buffer) {
        if (!renderingStopped && !released) {
            nativeReadData(nativeAudioDeviceProxyHandle, buffer, buffer.capacity());
        } else {
            logger.d(
                    "Ignoring readRenderData because either AudioDeviceProxy is released or "
                            + "rendering stopped");
        }
    }

    void release() {
        logger.d("release");
        if (!released) {
            if (nativeAudioDeviceProxyHandle != 0) {
                nativeRelease(nativeAudioDeviceProxyHandle);
                nativeAudioDeviceProxyHandle = 0;
            }
            released = true;
        }
    }

    @Override
    public AudioFormat getCapturerFormat() {
        return null;
    }

    @Override
    public boolean onInitCapturer() {
        logger.d("onInitCapturer");
        checkIsOnValidThread();
        capturingStopped = false;

        if (!released) {
            try {
                audioDevice.onInitCapturer();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return true;
    }

    @Override
    public boolean onStartCapturing(@NonNull AudioDeviceContext audioDeviceContext) {
        logger.d("onStartCapturing");
        checkIsOnValidThread();
        if (!released) {
            try {
                audioDevice.onStartCapturing(this);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public boolean onStopCapturing() {
        logger.d("onStopCapturing");
        checkIsOnValidThread();
        capturingStopped = true;
        if (!released) {
            try {
                audioDevice.onStopCapturing();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public AudioFormat getRendererFormat() {
        return null;
    }

    @Override
    public boolean onInitRenderer() {
        logger.d("onInitRenderer");
        checkIsOnValidThread();
        renderingStopped = false;
        if (!released) {
            try {
                audioDevice.onInitRenderer();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public boolean onStartRendering(@NonNull AudioDeviceContext audioDeviceContext) {
        logger.d("onStartRendering");
        checkIsOnValidThread();
        if (!released) {
            try {
                audioDevice.onStartRendering(this);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public boolean onStopRendering() {
        logger.d("onStopRendering");
        checkIsOnValidThread();
        renderingStopped = true;
        if (!released) {
            try {
                audioDevice.onStopRendering();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    void executeWorkerBlock(final Runnable runnable) {
        if (isValid()) {
            nativeExecuteWorkerBlock(nativeAudioDeviceProxyHandle, runnable);
        } else {
            logger.w("Calling executeWorkerBlock not a valid operation");
        }
    }

    void formatChanged() {
        logger.d("formatChanged");
        if (isValid()) {
            nativeFormatChanged(
                    nativeAudioDeviceProxyHandle,
                    audioDevice.getCapturerFormat(),
                    audioDevice.getRendererFormat());
        } else {
            logger.w("Calling formatChanged not a valid operation");
        }
    }

    private void checkIsOnValidThread() {
        if (handler == null) {
            handler = Util.createCallbackHandler();
        }
        ThreadChecker.checkIsValidThread(handler);
    }

    private boolean isValid() {
        return (!renderingStopped && !released) || (!capturingStopped && !released);
    }

    private static native void nativeRelease(long nativeAudioDeviceProxyHandle);

    private static native void nativeWriteData(
            long nativeAudioDeviceProxyHandle, ByteBuffer buffer, int size);

    private static native void nativeReadData(
            long nativeAudioDeviceProxyHandle, ByteBuffer buffer, int size);

    private static native void nativeExecuteWorkerBlock(
            long nativeAudioDeviceProxyHandle, Runnable runnable);

    private static native void nativeFormatChanged(
            long nativeAudioDeviceProxyHandle,
            AudioFormat capturerFormat,
            AudioFormat renderFormat);
}
