package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * A local audio track publication represents a {@link LocalAudioTrack} that has been shared to a
 * {@link Room}.
 */
public class LocalAudioTrackPublication implements AudioTrackPublication {
    private final String sid;
    private final LocalAudioTrack localAudioTrack;
    private TrackPriority priority;
    private long nativeLocalAudioTrackPublicationContext;

    LocalAudioTrackPublication(
            @NonNull String sid,
            @NonNull LocalAudioTrack localAudioTrack,
            @NonNull TrackPriority priority,
            long nativeLocalAudioTrackPublicationContext) {
        Preconditions.checkNotNull(sid, "SID must not be null");
        Preconditions.checkNotNull(localAudioTrack, "Local audio track must not be null");
        Preconditions.checkArgument(!sid.isEmpty(), "SID must not be empty");
        Preconditions.checkNotNull(priority, "priority should not be null");
        this.sid = sid;
        this.localAudioTrack = localAudioTrack;
        this.priority = priority;
        this.nativeLocalAudioTrackPublicationContext = nativeLocalAudioTrackPublicationContext;
    }

    /**
     * Returns the local audio track's server identifier. This value uniquely identifies the local
     * audio track within the scope of a {@link Room}.
     */
    @NonNull
    @Override
    public String getTrackSid() {
        return sid;
    }

    /**
     * Returns the name of the local audio track. An empty string is returned if not name was
     * specified.
     */
    @NonNull
    @Override
    public String getTrackName() {
        return localAudioTrack.getName();
    }

    /** Check if local audio track is enabled. */
    @Override
    public boolean isTrackEnabled() {
        return localAudioTrack.isEnabled();
    }

    /**
     * Unlike {@link AudioTrackPublication} or {@link RemoteAudioTrackPublication} this {@link
     * AudioTrack} is guaranteed to be non null.
     *
     * @return Returns the base audio track object of the published local audio track.
     */
    @NonNull
    @Override
    public AudioTrack getAudioTrack() {
        return localAudioTrack;
    }

    /** Returns the published local audio track. */
    @NonNull
    public LocalAudioTrack getLocalAudioTrack() {
        return localAudioTrack;
    }

    /**
     * Returns the {@link TrackPriority} of the published local audio track.
     *
     * @deprecated Retrieving track priority of local audio tracks is deprecated since {@link
     *     TrackPriority} is a deprecated feature.
     */
    @NonNull
    @Deprecated
    public TrackPriority getPriority() {
        return priority;
    }

    /**
     * Update the publisher's {@link TrackPriority} for this {@link LocalAudioTrack} after it has
     * been published. This method is a no-op if the {@link LocalAudioTrack} has been unpublished,
     * or the {@link LocalParticipant} has disconnected from the {@link Room}.
     *
     * @param priority The {@link TrackPriority} to be set.
     * @deprecated Setting track priority of local audio tracks is deprecated since {@link
     *     TrackPriority} is a deprecated feature.
     */
    @Deprecated
    public void setPriority(TrackPriority priority) {
        this.priority = priority;
        if (!isReleased()) {
            nativeSetPriority(nativeLocalAudioTrackPublicationContext, priority);
        }
    }

    /*
     * Releases native memory owned by local audio track publication.
     */
    synchronized void release() {
        if (!isReleased()) {
            nativeRelease(nativeLocalAudioTrackPublicationContext);
            nativeLocalAudioTrackPublicationContext = 0;
        }
    }

    synchronized boolean isReleased() {
        return nativeLocalAudioTrackPublicationContext == 0;
    }

    private native void nativeSetPriority(
            long nativeLocalAudioTrackPublicationContext, TrackPriority priority);

    private native void nativeRelease(long nativeLocalAudioTrackPublicationContext);
}
