package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/** Represents options when creating {@link LocalDataTrack}s. */
public class DataTrackOptions {
    /** Default value for max packet life time. */
    public static final int DEFAULT_MAX_PACKET_LIFE_TIME = -1;

    /** Default value for max retransmits */
    public static final int DEFAULT_MAX_RETRANSMITS = -1;

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DataTrackOptions that = (DataTrackOptions) o;
        if ((name == null && that.name != null) || name != null && that.name == null) return false;
        if ((name != null && that.name != null)) return name.equals(that.name);
        if (!(ordered == that.ordered)) return false;
        if (!(maxPacketLifeTime == that.maxPacketLifeTime)) return false;
        return maxRetransmits == that.maxRetransmits;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = 31 * result + (ordered ? 1 : 0);
        result = 31 * result + (maxPacketLifeTime != 0 ? maxPacketLifeTime : 0);
        result = 31 * result + (maxRetransmits != 0 ? maxRetransmits : 0);
        result = 31 * result + (name != null ? name.hashCode() : 0);

        return result;
    }

    /** Default data track options. */
    @NonNull
    public static final DataTrackOptions DEFAULT_DATA_TRACK_OPTIONS =
            new Builder()
                    .ordered(true)
                    .maxPacketLifeTime(DEFAULT_MAX_PACKET_LIFE_TIME)
                    .maxRetransmits(DEFAULT_MAX_RETRANSMITS)
                    .build();

    final boolean ordered;
    final int maxPacketLifeTime;
    final int maxRetransmits;
    final String name;

    private DataTrackOptions(Builder builder) {
        this.ordered = builder.ordered;
        this.maxPacketLifeTime = builder.maxPacketLifeTime;
        this.maxRetransmits = builder.maxRetransmits;
        this.name = builder.name;
    }

    /**
     * Build new {@link DataTrackOptions}.
     *
     * <p>All methods are optional.
     */
    public static class Builder {
        private boolean ordered = true;
        private int maxPacketLifeTime = DEFAULT_MAX_PACKET_LIFE_TIME;
        private int maxRetransmits = DEFAULT_MAX_RETRANSMITS;
        private String name = null;

        public Builder() {}

        /** Ordered transmission of messages. Default is {@code true}. */
        @NonNull
        public Builder ordered(boolean ordered) {
            this.ordered = ordered;
            return this;
        }

        /** Maximum retransmit time in milliseconds. */
        @NonNull
        public Builder maxPacketLifeTime(int maxPacketLifeTime) {
            this.maxPacketLifeTime = maxPacketLifeTime;
            return this;
        }

        /** Maximum number of retransmitted messages. */
        @NonNull
        public Builder maxRetransmits(int maxRetransmits) {
            this.maxRetransmits = maxRetransmits;
            return this;
        }

        /** Data track name. */
        @NonNull
        public Builder name(@Nullable String name) {
            this.name = name;
            return this;
        }

        /**
         * Builds the data track options.
         *
         * <p>Max packet life time and max retransmits are mutually exclusive. This means that only
         * one of these values can be set to a non default value at a time otherwise a {@link
         * IllegalStateException} occurs.
         */
        @NonNull
        public DataTrackOptions build() {
            Preconditions.checkArgument(maxPacketLifeTime >= DEFAULT_MAX_PACKET_LIFE_TIME);
            Preconditions.checkArgument(maxPacketLifeTime <= 65535);
            Preconditions.checkArgument(maxRetransmits >= DEFAULT_MAX_RETRANSMITS);
            Preconditions.checkArgument(maxRetransmits <= 65535);
            Preconditions.checkState(
                    maxRetransmits == DEFAULT_MAX_RETRANSMITS
                            || maxPacketLifeTime == DEFAULT_MAX_PACKET_LIFE_TIME);
            return new DataTrackOptions(this);
        }
    }
}
