package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/** A remote video track represents a remote video source. */
public class RemoteVideoTrack extends VideoTrack {
    private final String sid;
    private boolean isSwitchedOff;
    private @Nullable TrackPriority priority;
    private long nativeRemoteVideoTrackContext;

    RemoteVideoTrack(
            @NonNull tvi.webrtc.VideoTrack webRtcVideoTrack,
            @NonNull String sid,
            @NonNull String name,
            boolean enabled,
            boolean isSwitchedOff,
            @Nullable TrackPriority priority,
            long nativeRemoteVideoTrackContext) {
        super(webRtcVideoTrack, enabled, name);
        this.sid = sid;
        this.isSwitchedOff = isSwitchedOff;
        this.priority = priority;
        this.nativeRemoteVideoTrackContext = nativeRemoteVideoTrackContext;
    }

    /**
     * Returns the remote video track's server identifier. This value uniquely identifies the remote
     * video track within the scope of a {@link Room}.
     */
    @NonNull
    public String getSid() {
        return sid;
    }

    /** Return whether the track is switched off. */
    public boolean isSwitchedOff() {
        return isSwitchedOff;
    }

    /**
     * Get the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}. The default
     * subscriber priority is {@code null}, which indicates that the subscriber has not set a
     * priority for this {@link RemoteVideoTrack}.
     */
    public @Nullable TrackPriority getPriority() {
        return priority;
    }

    /**
     * Set the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}. Providing
     * {@code null} clears the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}.
     * This method is a no-op if the {@link RemoteVideoTrack} has been unsubscribed from.
     *
     * @param priority The priority to be set.
     */
    public void setPriority(@Nullable TrackPriority priority) {
        this.priority = priority;
        if (!isReleased()) {
            nativeSetPriority(nativeRemoteVideoTrackContext, priority);
        }
    }

    void setSwitchedOff(boolean isSwitchedOff) {
        this.isSwitchedOff = isSwitchedOff;
    }

    @Override
    synchronized void release() {
        if (!isReleased()) {
            super.release();
            nativeRelease(nativeRemoteVideoTrackContext);
            nativeRemoteVideoTrackContext = 0;
        }
    }

    @Override
    synchronized boolean isReleased() {
        return nativeRemoteVideoTrackContext == 0 && super.isReleased();
    }

    private native void nativeSetPriority(
            long nativeRemoteVideoTrackContext, TrackPriority priority);

    private native void nativeRelease(long nativeRemoteVideoTrackContext);
}
