package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * By default, the Video SDK uses {@link DefaultAudioDevice} to capture and render audio. The Video
 * SDK performs acoustic echo cancellation(AEC) and noise suppression(NS) using device hardware by
 * default.
 */
public final class DefaultAudioDevice implements AudioDevice {

    private boolean useHardwareAcousticEchoCanceler = true;
    private boolean useHardwareNoiseSuppressor = true;

    /**
     * Control if the built-in hardware AEC should be used or not. The default value is {@code
     * true}.
     *
     * @param useHardwareAcousticEchoCanceler true if audio device should use hardware AEC, false
     *     otherwise
     */
    private void setUseHardwareAcousticEchoCanceler(boolean useHardwareAcousticEchoCanceler) {
        this.useHardwareAcousticEchoCanceler = useHardwareAcousticEchoCanceler;
    }

    /**
     * Control if the built-in hardware noise suppressor should be used or not. The default value is
     * {@code true}.
     *
     * @param useHardwareNoiseSuppressor true if audio device should use hardware noise suppressor,
     *     false otherwise.
     */
    private void setUseHardwareNoiseSuppressor(boolean useHardwareNoiseSuppressor) {
        this.useHardwareNoiseSuppressor = useHardwareNoiseSuppressor;
    }

    /**
     * Indicates if the built-in hardware AEC is used or not. The default value is {@code true}.
     *
     * @return true if the built-in hardware AEC is used.
     */
    private boolean useHardwareAcousticEchoCanceler() {
        return this.useHardwareAcousticEchoCanceler;
    }

    /**
     * Indicates if the built-in hardware noise suppressor is used or not. The default value is
     * {@code true}.
     *
     * @return true if the built-in hardware noise suppressor is used.
     */
    private boolean useHardwareNoiseSuppressor() {
        return this.useHardwareNoiseSuppressor;
    }

    /** This method is a no-op that returns {@code null}. */
    @Override
    @Nullable
    public AudioFormat getCapturerFormat() {
        return null;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onInitCapturer() {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStartCapturing(@NonNull AudioDeviceContext audioDeviceContext) {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStopCapturing() {
        return false;
    }

    /** This method is a no-op that returns {@code null}. */
    @Override
    @Nullable
    public AudioFormat getRendererFormat() {
        return null;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onInitRenderer() {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStartRendering(@NonNull AudioDeviceContext audioDeviceContext) {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStopRendering() {
        return false;
    }
}
