/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import tvi.webrtc.VideoSink;

public abstract class VideoTrack implements Track {
    private static final Logger logger = Logger.getLogger(VideoTrack.class);

    private Set<VideoSink> videoSinks = new HashSet<>();
    private final tvi.webrtc.VideoTrack webRtcVideoTrack;
    private final String name;
    private boolean isEnabled;
    private boolean isReleased = false;

    VideoTrack(
            @NonNull tvi.webrtc.VideoTrack webRtcVideoTrack,
            boolean enabled,
            @NonNull String name) {
        this.isEnabled = enabled;
        this.name = name;
        this.webRtcVideoTrack = webRtcVideoTrack;
    }

    /**
     * Add a video sink to receive frames from the video track.
     *
     * @param videoSink video sink that receives video.
     */
    public synchronized void addSink(@NonNull VideoSink videoSink) {
        Preconditions.checkNotNull(videoSink, "Video sink must not be null");

        /*
         * Allow addSink to be called after the track has been released to avoid crashes
         * in cases where a developer mistakenly tries to render a track that has been removed.
         * This is different from LocalVideoTrack because developers do not control when a remote
         * video track is released.
         */
        if (!isReleased) {
            videoSinks.add(videoSink);

            webRtcVideoTrack.addSink(videoSink);
        } else {
            logger.w("Attempting to add sink to track that has been removed");
        }
    }

    /**
     * Remove a video sink to stop receiving video from the video track.
     *
     * @param videoSink the video sink that should no longer receives video.
     */
    public synchronized void removeSink(@NonNull VideoSink videoSink) {
        Preconditions.checkNotNull(videoSink, "Video sink must not be null");

        /*
         * Allow for removeSink to be called after the track has been released to avoid
         * crashes in cases where a developer mistakenly tries to stop renderering a track that has
         * been removed. This is different from LocalVideoTrack because developers do not control
         * when a remote video track is released.
         */
        if (!isReleased) {
            webRtcVideoTrack.removeSink(videoSink);
            videoSinks.remove(videoSink);
        } else {
            logger.w("Attempting to remove sink from track that has been removed");
        }
    }

    /**
     * The list of sinks receiving video from this video track. An empty list will be returned if
     * the video track has been released.
     */
    @NonNull
    public synchronized List<VideoSink> getSinks() {
        return new ArrayList<>(videoSinks);
    }

    /**
     * Check if this video track is enabled.
     *
     * @return true if track is enabled.
     */
    @Override
    public boolean isEnabled() {
        return isEnabled;
    }

    /**
     * Returns the video track name. A pseudo random string is returned if no track name was
     * specified.
     */
    @NonNull
    @Override
    public String getName() {
        return name;
    }

    synchronized void setEnabled(boolean isEnabled) {
        this.isEnabled = isEnabled;
    }

    synchronized void release() {
        if (!isReleased) {
            invalidateWebRtcTrack();
            videoSinks.clear();
            isReleased = true;
        }
    }

    synchronized boolean isReleased() {
        return isReleased;
    }

    synchronized void invalidateWebRtcTrack() {
        if (webRtcVideoTrack != null) {
            for (VideoSink videoSink : videoSinks) {
                webRtcVideoTrack.removeSink(videoSink);
            }
        }
    }

    /*
     * Used in video track tests to emulate behavior of a remote video track
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    synchronized tvi.webrtc.VideoTrack getWebRtcTrack() {
        return webRtcVideoTrack;
    }
}
