/*
 * Copyright (C) 2018 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import tvi.webrtc.VideoDecoderFactory;
import tvi.webrtc.VideoEncoderFactory;

/*
 * Provides options for creating a test MediaFactory instance. Useful for simulating media scenarios
 * on a device.
 */
@VisibleForTesting(otherwise = VisibleForTesting.NONE)
class MediaOptions {
    @SuppressWarnings("unused")
    @Nullable
    private final String audioFilePath;

    @Nullable final VideoEncoderFactory videoEncoderFactory;
    @Nullable final VideoDecoderFactory videoDecoderFactory;

    private MediaOptions(Builder builder) {
        this.audioFilePath = builder.audioFilePath;
        this.videoEncoderFactory = builder.videoEncoderFactory;
        this.videoDecoderFactory = builder.videoDecoderFactory;
    }

    static class Builder {
        @Nullable private String audioFilePath;
        @Nullable private VideoEncoderFactory videoEncoderFactory;
        @Nullable private VideoDecoderFactory videoDecoderFactory;

        /*
         * Provide a path to an audio file. Providing an audio file path configures the
         * MediaFactory instance to use an audio device module that captures audio from the
         * given file.
         */
        Builder audioFilePath(@NonNull String audioFilePath) {
            Preconditions.checkNotNull("audioFilePath should not be null", audioFilePath);
            this.audioFilePath = audioFilePath;
            return this;
        }

        Builder videoEncoderFactory(VideoEncoderFactory videoEncoderFactory) {
            this.videoEncoderFactory = videoEncoderFactory;
            return this;
        }

        Builder videoDecoderFactory(VideoDecoderFactory videoDecoderFactory) {
            this.videoDecoderFactory = videoDecoderFactory;
            return this;
        }

        MediaOptions build() {
            return new MediaOptions(this);
        }
    }
}
