/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * A local video track publication represents a {@link LocalVideoTrack} that has been shared to a
 * {@link Room}.
 */
@AccessedByNative
public class LocalVideoTrackPublication implements VideoTrackPublication {
    private final String sid;
    private final LocalVideoTrack localVideoTrack;
    private TrackPriority priority;
    private long nativeLocalVideoTrackPublicationContext;

    @AccessedByNative
    LocalVideoTrackPublication(
            @NonNull String sid,
            @NonNull LocalVideoTrack localVideoTrack,
            @NonNull TrackPriority priority,
            long nativeLocalVideoTrackPublicationContext) {
        Preconditions.checkNotNull(sid, "SID must not be null");
        Preconditions.checkArgument(!sid.isEmpty(), "SID must not be empty");
        Preconditions.checkNotNull(localVideoTrack, "Local video track must not be null");
        Preconditions.checkNotNull(priority, "priority should not be null");
        this.sid = sid;
        this.localVideoTrack = localVideoTrack;
        this.priority = priority;
        this.nativeLocalVideoTrackPublicationContext = nativeLocalVideoTrackPublicationContext;
    }

    /**
     * Returns the local video track's server identifier. This value uniquely identifies the local
     * video track within the scope of a {@link Room}.
     */
    @NonNull
    @Override
    public String getTrackSid() {
        return sid;
    }

    /**
     * Returns the name of the local video track. An empty string is returned if no name was
     * specified.
     */
    @NonNull
    @Override
    public String getTrackName() {
        return localVideoTrack.getName();
    }

    /** Check if local video track is enabled. */
    @Override
    public boolean isTrackEnabled() {
        return localVideoTrack.isEnabled();
    }

    /**
     * Unlike {@link VideoTrackPublication} or {@link RemoteVideoTrackPublication} this {@link
     * VideoTrack} is guaranteed to be non null.
     *
     * @return Returns the base video track object of the published local video track.
     */
    @NonNull
    @Override
    public VideoTrack getVideoTrack() {
        return localVideoTrack;
    }

    /** Returns the published local video track. */
    @NonNull
    public LocalVideoTrack getLocalVideoTrack() {
        return localVideoTrack;
    }

    /** Returns the {@link TrackPriority} of the published local video track. */
    @NonNull
    public TrackPriority getPriority() {
        return priority;
    }

    /**
     * Update the publisher's {@link TrackPriority} for this {@link LocalVideoTrack} after it has
     * been published. This method is a no-op if the {@link LocalVideoTrack} has been unpublished,
     * or the {@link LocalParticipant} has disconnected from the {@link Room}.
     *
     * @param priority The {@link TrackPriority} to be set.
     */
    public void setPriority(TrackPriority priority) {
        this.priority = priority;
        if (!isReleased()) {
            nativeSetPriority(nativeLocalVideoTrackPublicationContext, priority);
        }
    }

    /*
     * Releases native memory owned by local audio video publication.
     */
    synchronized void release() {
        if (!isReleased()) {
            nativeRelease(nativeLocalVideoTrackPublicationContext);
            nativeLocalVideoTrackPublicationContext = 0;
        }
    }

    synchronized boolean isReleased() {
        return nativeLocalVideoTrackPublicationContext == 0;
    }

    private native void nativeSetPriority(
            long nativeLocalVideoTrackPublicationContext, TrackPriority priority);

    private native void nativeRelease(long nativeLocalVideoTrackPublicationContext);
}
