package com.trx.neon.api.neonBeta.model;

import android.os.Parcel;
import android.os.Parcelable;

import java.util.UUID;

/**
 * Created by John on 7/1/2016.
 */
public class ElevationInfo implements Parcelable {

    public enum ElevationType {
        None,
        Floor,
        OnTerrain,
        Altitude,
        Underground,
        UndergroundFloor,
        BuildingFloor
    }

    public final ElevationType Type;
    public final UUID StructureID;
    public final double Value;

    private ElevationInfo(ElevationType type, double value, UUID structureID) {
        Type = type;
        Value = value;
        StructureID = structureID;
    }

    /**
     * Sets altitude to a particular floor
     * @param floor
     * @return
     */
    public static ElevationInfo OnFloor(int floor) {
        return new ElevationInfo(ElevationType.Floor, floor, null);
    }

    public static ElevationInfo OnFloor(UUID structureID, int floor)
    {
        return new ElevationInfo(ElevationType.BuildingFloor, floor, structureID);
    }

    /**
     * Sets altitude to the terrain
     * @return
     */
    public static ElevationInfo OnTerrain() {
        return new ElevationInfo(ElevationType.OnTerrain, 0, null);
    }

    /**
     * Sets altitude at a particular altitude with error bounds equal to the 2D error radius
     * @param altitude
     * @return
     */
    public static ElevationInfo AtAltitude(float altitude) {
        return new ElevationInfo(ElevationType.Altitude, altitude, null);
    }

    /**
     * No elevation information is known
     * @return
     */
    public static ElevationInfo None() {
        return new ElevationInfo(ElevationType.None, 0, null);
    }

    /**
     * Used for sending data across binders
     */
    public static final Parcelable.Creator<ElevationInfo> CREATOR = new Parcelable.Creator<ElevationInfo>() {
        @Override
        public ElevationInfo createFromParcel(Parcel in) {
            return new ElevationInfo(in);
        }

        @Override
        public ElevationInfo[] newArray(int size) {
            return new ElevationInfo[size];
        }
    };

    ElevationInfo(Parcel in)
    {
        int valueIndex = in.readInt();
        if (valueIndex < ElevationType.values().length)
        {
            Type = ElevationType.values()[in.readInt()];
            Value = in.readDouble();

            if(Type == ElevationType.BuildingFloor)
            {
                long mostSig = in.readLong();
                long leastSig = in.readLong();
                StructureID = new UUID(mostSig, leastSig);
            }
            else
            {
                StructureID = null;
            }
        }
        else
        {
            Type = ElevationType.None;
            Value = 0;
            StructureID = null;
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(Type.ordinal());
        dest.writeDouble(Value);

        if (Type == ElevationType.BuildingFloor)
        {
            long mostSig = StructureID.getMostSignificantBits();
            long leastSig = StructureID.getLeastSignificantBits();
            dest.writeLong(mostSig);
            dest.writeLong(leastSig);
        }
    }

    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        sb.append(Type.toString());
        if (Type == ElevationType.Floor || Type == ElevationType.UndergroundFloor) {
            sb.append(": ");
            sb.append(Value);
        } else if (Type == ElevationType.Altitude) {
            sb.append(": ");
            sb.append(Value);
            sb.append(" m");
        } else if (Type == ElevationType.BuildingFloor) {
            sb.append(": ");
            sb.append(Value);
            sb.append(" (" + StructureID + ")");
        }

        return sb.toString();
    }
}
