/******************************************************************************
 * 		
 * 	 Copyright 2015, TRX Systems, Inc.  All Rights Reserved.
 * 
 *   TRX Systems, Inc.
 *   7500 Greenway Center Drive, Suite 420
 *   Greenbelt, Maryland  20770
 *   
 *   Tel:    (301) 313-0053
 *   email:  info@trxsystems.com
 * 
 *****************************************************************************/
package com.trx.neon.api.neon;

import android.accounts.Account;
import android.app.Activity;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import com.trx.neon.api._internal.NeonImpl;
import com.trx.neon.api.neon.model.PositionConstraint;
import com.trx.neon.api.neon.model.interfaces.INeonEventListener;
import com.trx.neon.api.neon.model.interfaces.INeonLocationListener;
import com.trx.neon.api.neon.model.types.NeonLocationType;
import com.trx.neon.api.neon.model.types.NeonEventType;
import com.trx.neon.api.neonBeta.model.ElevationInfo;
import com.trx.neon.api.neonBeta.model.ManualConstraint;
import com.trx.neon.binder.OnLoginResult;

import java.util.EnumSet;

import static com.trx.neon.api._internal.NeonImpl.checkLocation;
import static com.trx.neon.api._internal.NeonImpl.neon;

/**
 * The main entry point for integrating Neon Location Services into your application.  
 * @author Dan
 */
public class Neon {
	/**
	 * Register an intent filter for this action to have your UI be started when the user touch pairs a tracking unit
	 */
	public static final String ACTIVITY_START_ON_TOUCH_PAIR = "com.trx.neon.framework.START_ON_TOUCH_PAIR";

	/**
	 * Neon location service is paired and ready to run
	 */
	public static final String NEON_READY_ON_TOUCH_PAIR = "com.trx.neon.framework.NEON_READY_ON_TOUCH_PAIR";

	/**
	 * Start this activity to let user manage Neon's preferences in a standard way
	 */
	public static final String ACTIVITY_SETTINGS = "com.trx.neon.framework.settings.SETTINGS";

	/**
	 * Register a listener for CURRENT location updates.
	 */
	public static void registerLocationUpdates(INeonLocationListener listener) {
		registerLocationUpdates(listener, NeonLocationType.CURRENT, "no tag");
	}

	/**
	 * Register a listener for location updates of the specified type
	 */
	public static void registerLocationUpdates(INeonLocationListener listener, NeonLocationType type) {
		neon.registerLocationUpdates(listener, type, "no tag");
	}

	/**
	 * Register a named listener for CURRENT location updates.
	 */
	public static void registerLocationUpdates(INeonLocationListener listener, String tag) {
		registerLocationUpdates(listener, NeonLocationType.CURRENT, tag);
	}

	/**
	 * Register a named listener for location updates of the specified type
	 */
	public static void registerLocationUpdates(INeonLocationListener listener, NeonLocationType type, String tag) {
		neon.registerLocationUpdates(listener, type, tag);
	}

	/**
	 * Register a listener for events of all types
	 */
	public static void registerEvents(INeonEventListener listener) {
		registerEvents(listener, EnumSet.allOf(NeonEventType.class));
	}

	/**
	 * Register a listener for the specified event types
	 */
	private static void registerEvents(INeonEventListener listener, EnumSet<NeonEventType> events) {
		neon.registerEvents(listener, events);
	}

	/**
	 * Checks if the Neon Service is already installed
	 * Return true if the Neon Service package was found
	 */
	public static boolean isNeonInstalled(Context c) {
		return NeonImpl.isNeonServiceInstalled(c);
	}

	/**
	 * Binds the Neon Location Services, ensuring that tracking is running and connecting all registered listeners
	 * All events will be posted to the main application thread.
	 * Returns true if Neon was successfully bound, false otherwise
	 */
	public static boolean bind(Context c) {
		return bind(c, new Handler(Looper.getMainLooper()));
	}

	/**
	 * Binds the Neon Location Services, ensuring that tracking is running and connecting all registered listeners
	 * All registered events will be posted to the specified handler.
	 * Returns true if Neon was successfully bound, false otherwise
	 */
	public static boolean bind(Context c, Handler eventHandler) {
		return neon.bind(c, eventHandler);
	}


	/**
	 * Logs-in with the specified account to the Neon Location Services.
	 */
	public static void login(Account account, OnLoginResult onLoginResult) {
		neon.login(account, onLoginResult);
	}



	/**
	 * Constrain the user's location.
	 *
	 */
	public static void addConstraint(PositionConstraint constraint) {
		checkLocation(constraint.getLatitude(), constraint.getLongitude());
		neon.addManualConstraint(new ManualConstraint(constraint.getLatitude(), constraint.getLongitude(), constraint.getError(), ElevationInfo.None()));
	}

	/**
	 * Unregister a previously registered listener
	 *
	 * @return true if listener was removed, false if it could not be found
	 */
	public static boolean unregisterLocationUpdates(INeonLocationListener listener) {
		return neon.unregisterLocationUpdates(listener);
	}

	/**
	 * Unregister a previously registered listener
	 *
	 * @return true if listener was removed, false if it could not be found
	 */
	public static boolean unregisterEvents(INeonEventListener listener) {
		return neon.unregisterEvents(listener);
	}

	/**
	 * Unbinds the Neon Location Services.  If this context is the last to unbind, then tracking will cease.
	 * All registered listeners will be disconnected until the next bind call,
	 * Events that have already been queued will continue to be processed, even after this function returns.
	 **/
	public static void unbind() {
		neon.unbind();
	}

	/**
	 * Get the version of the installed neon location service
	 **/
	public static String getLocationServiceVersion(Context c) {
		return neon.getLocationServiceVersion(c);
	}

	/**
	 * Interact with the user to download and install a new version of the Neon Location Services.
	 *
	 * @param visibleActivity A resumed Activity context used to start the interaction
	 */
	public static void upgradeNeonLocationServices(Activity visibleActivity) {
		neon.upgradeNeonLocationServices(visibleActivity);
	}

	/**
	 * Interact with the user to download and install a new version of the Neon Location Services.
	 *
	 * @param visibleActivity A resumed Activity context used to start the interaction
	 * @param mandatoryUpdate update will be mandatory
	 */
	public static void upgradeNeonLocationServices(Activity visibleActivity, boolean mandatoryUpdate) {
		neon.upgradeNeonLocationServices(visibleActivity, mandatoryUpdate);
	}

	/**
	 * Interact with the user to download and install a new version of the currently paired device firmware
	 *
	 * @param visibleActivity A resumed Activity context used to start the interaction
	 */
	public static void upgradeDeviceFirmware(Activity visibleActivity) {
		neon.upgradeDeviceFirmware(visibleActivity);
	}

	/**
	 * Interact with the user to download and install a new version of the currently paired device firmware
	 *
	 * @param visibleActivity A resumed Activity context used to start the interaction
	 * @param requestCode     A request code used for a call to startActivityForResult.  Results of the
	 *                        firmware upgrade process will be returned to the activity in an onActivityResult callback.
	 */
	public static void upgradeDeviceFirmware(Activity visibleActivity, int requestCode) {
		neon.upgradeDeviceFirmware(visibleActivity, requestCode);
	}

	/**
	 * Starts the LoginActivity, where users can login using their email and password or a google account
	 *
	 * @param visibleActivity onActivityResult will be called on this Activity when the Login Activity finishes
	 * @param requestCode     onActivityResult will be called this requestCode
	 * @param forceLogOut     forces LoginActivity to logout first, otherwise the activity will quit immediately when an account is already saved
	 * @return false if the LoginActivity could not be started, true otherwise
	 * in onActivityResult for this requestCode, if the resultCode is Activity.RESULT_CANCELED then the login was not done (pressed back button),
	 * if the resultCode is Activity.RESULT_OK then the login was successful and the account information is available via the data
	 * passed in onActivityResult :	Bundle extras = data.getExtras(); Account account = extras.getParcelable("account");
	 */
	public static boolean startLoginActivityForResult(int requestCode, Activity visibleActivity, boolean forceLogOut) {
		return neon.startLoginActivityForResult(requestCode, visibleActivity, forceLogOut);
	}
}
