package com.trustingsocial.apisdk;

import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;

import com.trustingsocial.apisdk.data.TVApiError;
import com.trustingsocial.apisdk.data.TVAuthorization;
import com.trustingsocial.apisdk.data.TVCallback;
import com.trustingsocial.apisdk.data.TVPollingCallback;
import com.trustingsocial.apisdk.data.TVPollingStatusResponse;
import com.trustingsocial.apisdk.data.TVResponseData;
import com.trustingsocial.apisdk.utils.DecryptiontInterceptor;
import com.trustingsocial.apisdk.utils.GsonUtils;
import com.trustingsocial.apisdk.utils.Utils;

import org.json.JSONObject;

import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import okhttp3.Call;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;


class ApiServiceHelper {
    private static final String TAG = ApiServiceImpl.class.getSimpleName();
    private static int PULLING_TIMEOUT = 30000;
    private static Integer PULLING_THRESHOLD = 1000;
    private final OkHttpClient mClient;
    private final String baseUrl;
    private HashMap<String, Integer> pullingMap = new HashMap<>();
    private IAuthorizationProvider authorizationProvider;

    ApiServiceHelper(String baseUrl, IAuthorizationProvider authorizationProvider) {
        this.baseUrl = baseUrl;
        this.authorizationProvider = authorizationProvider;
        try {
            // Create a trust manager that does not validate certificate chains
            X509TrustManager trustManager = new X509TrustManager() {
                @Override
                public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {

                }

                @Override
                public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {

                }

                @Override
                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[]{};
                }
            };
            final TrustManager[] trustAllCerts = new TrustManager[]{
                    trustManager
            };

            // Install the all-trusting trust manager
            final SSLContext sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
            // Create an ssl socket factory with our all-trusting manager
            final SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();
            OkHttpClient.Builder builder = new OkHttpClient.Builder();
            builder.sslSocketFactory(sslSocketFactory, trustManager);
            builder.writeTimeout(60, TimeUnit.SECONDS);
            builder.readTimeout(60, TimeUnit.SECONDS);
            builder.hostnameVerifier(new HostnameVerifier() {
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            });
            builder.addInterceptor(new DecryptiontInterceptor(authorizationProvider.getSecretKey()));
            this.mClient = builder.build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    <T> void requestJSON(String path, Object bodyData, TVCallback<T> callback, Class<T> type) {
        RequestBody body = RequestBody.create(Utils.JSON, GsonUtils.toJson(bodyData));
        Log.e(TAG, "body: " + GsonUtils.toJson(bodyData));
        request("POST", path, body, callback, type);
    }

    <T> void requestJSON(String path, Map<String, Object> params, TVCallback<T> callback, Class<T> type) {
        JSONObject parameter = new JSONObject(params);
        RequestBody body = RequestBody.create(Utils.JSON, parameter.toString());
        Log.e(TAG, "body: " + parameter.toString());
        request("POST", path, body, callback, type);
    }

    <T> void requestGet(String path, final TVCallback<T> callback, final Class<T> type) {
        request("GET", path, null, callback, type);
    }

    <T> void request(final String method, final String path, final RequestBody requestBody, final TVCallback<T> callback, final Class<T> type) {
        authorizationProvider.getAuthorization(method, path, new TVCallback<TVAuthorization>() {
            @Override
            public void onSuccess(TVAuthorization authorization) {
                TVApi.sApiCalls.add(doRequest(method, path, requestBody, authorization, callback, type));
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        });
    }

    private <T> Call doRequest(String method, String path, RequestBody requestBody, TVAuthorization authorization, final TVCallback<T> callback, final Class<T> type) {
        try {
            final Request request = buildRequest(method, path, requestBody, authorization);
            Call call = mClient.newCall(request);
            call.enqueue(new okhttp3.Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    Log.e(TAG, "onFailure");
                    callback.onError(Collections.singletonList(TVApiError.networkError()));
                }

                @Override
                public void onResponse(Call call, Response response) {
                    Log.e(TAG, "onResponse: " + call.request().url());
                    try {
                        if (type == byte[].class) {
                            handleDownloadResult(response, (TVCallback<byte[]>) callback);
                        } else {
                            handleResponse(response, callback, type);
                        }
                    } catch (Exception ex) {
                        callback.onError(Collections.singletonList(TVApiError.internalError()));
                        Log.e(TAG, ex.getMessage(), ex);
                    }

                }
            });
            return call;
        } catch (Exception ex) {
            callback.onError(Collections.singletonList(new TVApiError(TVApiError.ERROR_CODE_INTERNAL, ex.getMessage())));
            return null;
        }
    }

    private void handleDownloadResult(Response response, TVCallback<byte[]> callback) throws Exception {
        if (response.isSuccessful()) {
            byte[] byteRes = response.body().bytes();
            callback.onSuccess(byteRes);
        } else {
            String res = response.body().string().trim();
            Log.e(TAG, "response: " + res);
            TVResponseData responseData = GsonUtils.fromJson(res, TVResponseData.class);
            callback.onError(responseData.getErrors());
        }

    }

    private <T> void handleResponse(Response response, TVCallback<T> callback, Class<T> type) throws Exception {
        String res = response.body().string().trim();
        Log.e(TAG, "response: " + res);
        TVResponseData<T> responseData = GsonUtils.fromJson(res, GsonUtils.getType(TVResponseData.class, type));
        if (response.isSuccessful() && responseData.getErrors() == null && responseData.getData() != null) {
            callback.onSuccess(responseData.getData());
        } else {
            callback.onError(responseData.getErrors());
            Log.e(TAG, GsonUtils.getInstance().toJson(responseData));
        }
    }

    private Request buildRequest(String method, String path, RequestBody requestBody, TVAuthorization authorization) {
        String signature = "TV " + authorization.getApiKeyId() + ":" + authorization.getSignature();
        Log.e(TAG, "Authorization: " + signature);
        Log.e(TAG, "Timestamp: " + authorization.getTimestamp());

        Request.Builder requestBuilder = new Request.Builder()
                .url(baseUrl + path)
                .method(method, requestBody)
                .addHeader("Authorization", signature)
                .addHeader("X-TV-Timestamp", authorization.getTimestamp());
        IDeviceInfo deviceInfo = TVApi.getDeviceInfo();
        if (deviceInfo != null) {
            requestBuilder.addHeader(IDeviceInfo.HEADER_PLATFORM, deviceInfo.getPlatform());
            requestBuilder.addHeader(IDeviceInfo.HEADER_OS_VERSION, deviceInfo.getOSVersion());
            requestBuilder.addHeader(IDeviceInfo.HEADER_SDK_VERSION, deviceInfo.getSDKVersion());
            requestBuilder.addHeader(IDeviceInfo.HEADER_DEVICE_MODEL, deviceInfo.getDeviceModel());
        }

        String transactionId = TVApi.getTransactionId();
        if (!Utils.isEmptyString(transactionId)) {
            requestBuilder.addHeader("X-TV-Transaction", transactionId);
        }

        String channelId = TVApi.getChannelId();
        if (!Utils.isEmptyString(channelId )) {
            requestBuilder.addHeader("X-TV-Channel", channelId);
        }

        return requestBuilder.build();
    }

    <T extends TVPollingStatusResponse> void doPolling(final String requestId, final String apiPath, final TVPollingCallback<T> callback, final Class<T> type) {
        callback.onReceiveRequestId(requestId);
        HandlerThread handlerThread = new HandlerThread("Pooling_thread");
        handlerThread.start();
        final Handler handler = new Handler(handlerThread.getLooper());
        pullingMap.put(requestId, 0);
        final Runnable runnable = new Runnable() {
            @Override
            public void run() {
                final Runnable mRunnable = this;

                requestGet(apiPath + "/" + requestId, new TVCallback<T>() {
                    @Override
                    public void onSuccess(T data) {
                        if (data.getStatus() == TVPollingStatusResponse.Status.PROCESSING) {
                            Integer currentPullTime = pullingMap.get(requestId);
                            Integer newPullingTime = currentPullTime + PULLING_THRESHOLD;
                            pullingMap.put(requestId, newPullingTime);
                            if (newPullingTime < PULLING_TIMEOUT) {
                                handler.postDelayed(mRunnable, PULLING_THRESHOLD);
                            } else {
                                callback.onError(Collections.singletonList(TVApiError.timeoutError()));
                            }
                        } else if (data.getStatus() == TVPollingStatusResponse.Status.FAILURE) {
                            callback.onError(Collections.singletonList(TVApiError.timeoutError()));
                        } else {
                            callback.onSuccess(data);
                        }
                    }

                    @Override
                    public void onError(List<TVApiError> errors) {
                        callback.onError(errors);
                    }
                }, type);
            }
        };
        handler.postDelayed(runnable, PULLING_THRESHOLD);
    }
}
