package com.trustingsocial.apisdk;

import android.text.TextUtils;

import com.trustingsocial.apisdk.data.TVApiError;
import com.trustingsocial.apisdk.data.TVCallback;
import com.trustingsocial.apisdk.data.TVCardInfoRequest;
import com.trustingsocial.apisdk.data.TVCardInfoResponse;
import com.trustingsocial.apisdk.data.TVClientSettings;
import com.trustingsocial.apisdk.data.TVCompareFaceResponse;
import com.trustingsocial.apisdk.data.TVIDSanityResponse;
import com.trustingsocial.apisdk.data.TVIdTamperingResponse;
import com.trustingsocial.apisdk.data.TVPollingCallback;
import com.trustingsocial.apisdk.data.TVPollingResponse;
import com.trustingsocial.apisdk.data.TVSelfieSanityResponse;
import com.trustingsocial.apisdk.data.TVSyncCardInfoRequest;
import com.trustingsocial.apisdk.data.TVSyncImage;
import com.trustingsocial.apisdk.data.TVTransaction;
import com.trustingsocial.apisdk.data.TVUploadImageResponse;
import com.trustingsocial.apisdk.data.TVVerifyLivenessResponse;
import com.trustingsocial.apisdk.utils.GsonUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;

class ApiServiceImpl implements ApiService {

    private static final String PATH_SEPARATOR = "/";
    private static final String PATH_COMPARE_FACES = "/v1/compare_faces";
    private static final String PATH_UPLOAD_IMAGES = "/v1/images";
    private static final String PATH_VERIFY_LIVENESS = "/v1/verify_face_liveness";
    private static final String PATH_READ_ID_CARD_INFO = "/v1/read_id_card_info";
    private static final String PATH_CLIENT_SETTINGS = "/v1/client_settings";

    private static final String PATH_VERIFY_PORTRAIT_SANITY = "/v1/verify_portrait_sanity";
    private static final String PATH_VERIFY_ID_CARD_SANITY = "/v1/verify_id_card_sanity";

    private static final String PATH_VERIFY_LIVENESS_SYNC = "/v1/verify_face_liveness_sync";
    private static final String PATH_COMPARE_FACES_SYNC = "/v1/compare_faces_sync";
    private static final String PATH_VERIFY_ID_CARD_SANITY_SYNC = "/v1/verify_id_card_sanity_sync";
    private static final String PATH_VERIFY_PORTRAIT_SANITY_SYNC = "/v1/verify_portrait_sanity_sync";
    private static final String PATH_READ_ID_CARD_INFO_SYNC = "/v1/read_id_card_info_sync";
    private static final String PATH_DETECT_ID_TAMPERING_SYNC = "/v1/detect_id_card_tampering_sync";
    private static final String PATH_START_TRANSACTION = "/v1/transactions";
    private static final String PATH_DOWNLOAD_IMAGE = "/v1/images";

    private ApiServiceHelper apiHelper;

    ApiServiceImpl(String baseUrl, IAuthorizationProvider authorizationProvider) {
        this.apiHelper = new ApiServiceHelper(baseUrl, authorizationProvider);
    }

    @Override
    public void compareFaces(String image1Id, String image2Id, TVCallback<TVPollingResponse> callback) {
        Map<String, Object> textParams = new HashMap<>();
        if (!TextUtils.isEmpty(image1Id)) {
            textParams.put("image1_id", image1Id);
        }

        if (!TextUtils.isEmpty(image2Id)) {
            textParams.put("image2_id", image2Id);
        }

        apiHelper.requestJSON(PATH_COMPARE_FACES, textParams, callback, TVPollingResponse.class);
    }

    @Override
    public void checkCompareFaces(String requestId, TVCallback<TVCompareFaceResponse> callback) {
        apiHelper.requestGet(PATH_COMPARE_FACES + PATH_SEPARATOR + requestId, callback, TVCompareFaceResponse.class);
    }

    @Override
    public void pollCompareFaces(String image1Id, String image2Id, final TVPollingCallback<TVCompareFaceResponse> callback) {
        compareFaces(image1Id, image2Id, new TVCallback<TVPollingResponse>() {
            @Override
            public void onSuccess(final TVPollingResponse data) {
                apiHelper.doPolling(data.getRequestId(), PATH_COMPARE_FACES, callback, TVCompareFaceResponse.class);
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        });
    }

    @Override
    public void uploadImage(byte[] image, String label, Map<String, String> metadata, TVCallback<TVUploadImageResponse> callback) {
        MultipartBody.Builder requestBodyBuilder = new MultipartBody.Builder().setType(MultipartBody.FORM);
        if (!TextUtils.isEmpty(label)) {
            requestBodyBuilder.addFormDataPart("label", label);
        }
        requestBodyBuilder.addFormDataPart("file", System.currentTimeMillis() + "_" + label + ".jpg",
                RequestBody.create(MediaType.get("image/jpg"), image));
        Map<String, String> deviceInfoMap = new HashMap<>();
        deviceInfoMap.put(IDeviceInfo.HEADER_PLATFORM, TVApi.getDeviceInfo().getPlatform());
        deviceInfoMap.put(IDeviceInfo.HEADER_OS_VERSION, TVApi.getDeviceInfo().getOSVersion());
        deviceInfoMap.put(IDeviceInfo.HEADER_SDK_VERSION, TVApi.getDeviceInfo().getSDKVersion());
        deviceInfoMap.put(IDeviceInfo.HEADER_DEVICE_MODEL, TVApi.getDeviceInfo().getDeviceModel());
        if (metadata != null) {
            deviceInfoMap.putAll(metadata);
        }
        String deviceInfoJson = GsonUtils.toJson(deviceInfoMap);
        requestBodyBuilder.addFormDataPart("metadata", deviceInfoJson);
        apiHelper.request("POST", PATH_UPLOAD_IMAGES, requestBodyBuilder.build(), callback, TVUploadImageResponse.class);
    }

    @Override
    public void verifyLiveness(List<String> imageIds, TVCallback<TVPollingResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image_ids", imageIds);
        apiHelper.requestJSON(PATH_VERIFY_LIVENESS, params, callback, TVPollingResponse.class);
    }

    @Override
    public void pollVerifyLiveness(List<String> imageIds, final TVPollingCallback<TVVerifyLivenessResponse> callback) {
        verifyLiveness(imageIds, new TVCallback<TVPollingResponse>() {
            @Override
            public void onSuccess(final TVPollingResponse data) {
                apiHelper.doPolling(data.getRequestId(), PATH_VERIFY_LIVENESS, callback, TVVerifyLivenessResponse.class);
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        });
    }

    @Override
    public void checkVerifyLiveness(String requestId, TVCallback<TVVerifyLivenessResponse> callback) {
        apiHelper.requestGet(PATH_VERIFY_LIVENESS + PATH_SEPARATOR + requestId, callback, TVVerifyLivenessResponse.class);
    }

    @Override
    public void readIdCardInfo(TVCardInfoRequest readIdCardInfoRequest, TVCallback<TVPollingResponse> callback) {
        apiHelper.requestJSON(PATH_READ_ID_CARD_INFO, readIdCardInfoRequest, callback, TVPollingResponse.class);
    }

    @Override
    public void checkReadIdCardInfo(String requestId, TVCallback<TVCardInfoResponse> callback) {
        apiHelper.requestGet(PATH_READ_ID_CARD_INFO + PATH_SEPARATOR + requestId, callback, TVCardInfoResponse.class);
    }

    @Override
    public void pollReadIdCardInfo(TVCardInfoRequest readIdCardInfoRequest, final TVPollingCallback<TVCardInfoResponse> callback) {
        readIdCardInfo(readIdCardInfoRequest, new TVCallback<TVPollingResponse>() {
            @Override
            public void onSuccess(final TVPollingResponse data) {
                apiHelper.doPolling(data.getRequestId(), PATH_READ_ID_CARD_INFO, callback, TVCardInfoResponse.class);
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        });
    }

    @Override
    public void getClientSettings(TVCallback<TVClientSettings> callback) {
        apiHelper.requestGet(PATH_CLIENT_SETTINGS, callback, TVClientSettings.class);
    }

    @Override
    public void verifySelfieSanity(String imageId, final TVPollingCallback<TVSelfieSanityResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image_id", imageId);
        apiHelper.requestJSON(PATH_VERIFY_PORTRAIT_SANITY, params, new TVCallback<TVPollingResponse>() {
            @Override
            public void onSuccess(TVPollingResponse data) {
                apiHelper.doPolling(data.getRequestId(), PATH_VERIFY_PORTRAIT_SANITY, callback, TVSelfieSanityResponse.class);
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        }, TVPollingResponse.class);
    }

    @Override
    public void verifyIDSanity(String imageId, final TVPollingCallback<TVSelfieSanityResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image_id", imageId);
        apiHelper.requestJSON(PATH_VERIFY_ID_CARD_SANITY, params, new TVCallback<TVPollingResponse>() {
            @Override
            public void onSuccess(TVPollingResponse data) {
                apiHelper.doPolling(data.getRequestId(), PATH_VERIFY_ID_CARD_SANITY, callback, TVSelfieSanityResponse.class);
            }

            @Override
            public void onError(List<TVApiError> errors) {
                callback.onError(errors);
            }
        }, TVPollingResponse.class);
    }

    @Override
    public void syncVerifyLiveness(List<TVSyncImage> images, List<String> gestures, TVCallback<TVVerifyLivenessResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("images", TVSyncImage.map(images));
        params.put("gestures", gestures);
        apiHelper.requestJSON(PATH_VERIFY_LIVENESS_SYNC, params, callback, TVVerifyLivenessResponse.class);
    }

    @Override
    public void syncCompareFaces(TVSyncImage image1, TVSyncImage image2, TVCallback<TVCompareFaceResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image1", image1.toJSONObject());
        params.put("image2", image2.toJSONObject());
        apiHelper.requestJSON(PATH_COMPARE_FACES_SYNC, params, callback, TVCompareFaceResponse.class);
    }

    @Override
    public void syncVerifyIDSanity(TVSyncCardInfoRequest cardInfoRequest, TVCallback<TVIDSanityResponse> callback) {
        apiHelper.requestJSON(PATH_VERIFY_ID_CARD_SANITY_SYNC, cardInfoRequest, callback, TVIDSanityResponse.class);
    }

    @Override
    public void syncVerifySelfieSanity(TVSyncImage image, TVCallback<TVSelfieSanityResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image", image.toJSONObject());
        apiHelper.requestJSON(ApiServiceImpl.PATH_VERIFY_PORTRAIT_SANITY_SYNC, params, callback, TVSelfieSanityResponse.class);
    }

    @Override
    public void syncReadIdCardInfo(TVSyncCardInfoRequest cardInfoRequest, TVCallback<TVCardInfoResponse> callback) {
        apiHelper.requestJSON(PATH_READ_ID_CARD_INFO_SYNC, cardInfoRequest, callback, TVCardInfoResponse.class);
    }

    @Override
    public void syncDetectIdTampering(TVSyncImage image1, TVSyncImage image2, TVCallback<TVIdTamperingResponse> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("image", image1.toJSONObject());
        if (image2 != null) {
            params.put("image2", image2.toJSONObject());
        }
        apiHelper.requestJSON(PATH_DETECT_ID_TAMPERING_SYNC, params, callback, TVIdTamperingResponse.class);
    }

    @Override
    public void startTransaction(String refId, String channelName, TVCallback<TVTransaction> callback) {
        Map<String, Object> params = new HashMap<>();
        params.put("reference_id", refId);
        params.put("channel_name", channelName);
        apiHelper.requestJSON(PATH_START_TRANSACTION, params, callback, TVTransaction.class);
    }

    @Override
    public void endTransaction(String transactionId, TVCallback<TVTransaction> callback) {
        Map<String, Object> params = new HashMap<>();
        apiHelper.requestJSON(PATH_START_TRANSACTION + PATH_SEPARATOR + transactionId + PATH_SEPARATOR + "finish", params, callback, TVTransaction.class);
    }

    @Override
    public void downloadImage(String imageId, TVCallback<byte[]> callback) {
        apiHelper.requestGet(PATH_DOWNLOAD_IMAGE + PATH_SEPARATOR + imageId, callback, byte[].class);
    }
}


