package com.topimagesystems.util;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.content.res.Configuration;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.PointF;
import android.hardware.Camera;
import android.os.Build;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.Window;
import android.view.WindowManager;

import com.topimagesystems.camera.CameraConfigurationUtils;
import com.topimagesystems.controllers.imageanalyze.CameraConfigurationManager;
import com.topimagesystems.controllers.imageanalyze.CameraController;
import com.topimagesystems.controllers.imageanalyze.CameraManagerController;
import com.topimagesystems.intent.CaptureIntent;
import com.topimagesystems.micr.GenericBoundingBoxResult;

import org.opencv.core.Rect;


// TODO: Auto-generated Javadoc
/**
 * The Class UIUtils.
 */
public class UserInterfaceUtils {

	/** The Constant tag. */
	private final static String tag = Logger.makeLogTag("UIUtils");
	
	/** The stub. */
	private static String STUB = "payment";
	
	/** The check. */
	private static String CHECK = "check";
	
	/**
	 * Gets the display dimensions.
	 *
	 * @param context the context
	 * @return the display dimensions
	 */
	public static Point getDisplayDimensions(Context context) {
		Point size = new Point();
		WindowManager wm;
		if (context != null)
		 wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
		else{
			wm = (WindowManager) CameraController.getInstance().getSystemService(Context.WINDOW_SERVICE);
		}

		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
			getSizeAboveAPI13(size, wm);
		}
		else {
			getSizePriorApi13(size, wm);
		}

		if ((CameraManagerController.sessionType == CaptureIntent.SessionType.PORTRAIT && size.x > size.y) ||
				(CameraManagerController.sessionType != CaptureIntent.SessionType.PORTRAIT && size.x < size.y))
			return new Point(size.y, size.x);

		return size;
	}
	public static boolean hasHardwareAcceleration(Activity activity) {
		// Has HW acceleration been enabled manually in the current window?
		try {
			Window window = activity.getWindow();
			if (window != null) {
				if ((window.getAttributes().flags
						& WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED) != 0) {
					return true;
				}
			}

			// Has HW acceleration been enabled in the manifest?
			try {
				ActivityInfo info = activity.getPackageManager().getActivityInfo(
						activity.getComponentName(), 0);
				if ((info.flags & ActivityInfo.FLAG_HARDWARE_ACCELERATED) != 0) {
					return true;
				}
			} catch (PackageManager.NameNotFoundException e) {
				Log.e("hasHardwareAcceleration", "getActivityInfo(self) should not fail");
			}

			return false;
		}catch(Exception e){
			Logger.e("hasHardwareAcceleration",Log.getStackTraceString(e));
			return false;
		}
	}

	/**
	 * Gets the camera preview resolution.
	 *
	 * @param context the context
	 * @param parameters the parameters
	 * @return the camera preview resolution
	 */
	public static Point getCameraPreviewResolution(Context context, Camera.Parameters parameters) {
		if (CameraManagerController.invertedCamera){
			return CameraConfigurationUtils.findBestPreviewSizeValue(parameters,getDisplayDimensions(context));
		}
		else {
			return CameraConfigurationManager.getCameraPreviewResolution(parameters, getDisplayDimensions(context), context);
		}
	}

	/**
	 * Gets the version name.
	 *
	 * @param context the context
	 * @return the version name
	 */
	public static String getVersionName(Context context) {
			String app_ver = "unknown";
		
		try {
			app_ver = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName;
		}
		catch (NameNotFoundException e) {
			Logger.v(tag, e.getMessage());
		}
	
		return app_ver;
	}

	
	/**
	 * Gets the size prior api13.
	 *
	 * @param size the size
	 * @param wm the wm
	 * @return the size prior api13
	 */
	@SuppressWarnings("deprecation")
	private static void getSizePriorApi13(Point size, WindowManager wm) {
		Display d = wm.getDefaultDisplay();
		size.x = d.getWidth();
		size.y = d.getHeight();
	}

	/**
	 * Gets the size above ap i13.
	 *
	 * @param size the size
	 * @param wm the wm
	 * @return the size above ap i13
	 */
	@TargetApi(13)
	private static void getSizeAboveAPI13(Point size, WindowManager wm) {
		wm.getDefaultDisplay().getSize(size);
	}

	/**
	 * Gets the display metrics.
	 *
	 * @param context the context
	 * @return the display metrics
	 */
	public static DisplayMetrics getDisplayMetrics(Context context) {
		WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
		DisplayMetrics metrics = new DisplayMetrics();
		wm.getDefaultDisplay().getMetrics(metrics);
		return metrics;
	}

	/**
	 * Gets the API level.
	 *
	 * @return the API level
	 */
	public static int getAPILevel() {
		return Build.VERSION.SDK_INT;
	}
	
	/**
	 * Rect to bundle.
	 *
	 * @param rect the rect
	 * @return the bundle
	 */
	public static Bundle rectToBundle(Rect rect){
		Bundle bundle = new Bundle();
		if ( rect != null){
			bundle.putInt("x", rect.x);
			bundle.putInt("y", rect.y);
			bundle.putInt("width", rect.width);
			bundle.putInt("height", rect.height);
		}
		return bundle;
	}
	
	/**
	 * Bundle to rect.
	 *
	 * @param bundle the bundle
	 * @return the rect
	 */
	public static Rect bundleToRect(Bundle bundle){
		Rect rect = new Rect();
		if ( bundle != null){
			rect.x = bundle.getInt("x");
			rect.y = bundle.getInt("y");
			rect.width = bundle.getInt("width");
			rect.height = bundle.getInt("height");
		}
		return rect;
	}

	public static Rect getFitRectangle(int frameWidth, int frameHeight, int imageWidth, int imageHeight) {
		int x, y, width, height;
		if (frameWidth * imageHeight > frameHeight * imageWidth) {
			width = imageWidth * frameHeight / imageHeight;
			x = (frameWidth - width) / 2;
			height = frameHeight;
			y = 0;
		} else {
			width = frameWidth;
			x = 0;
			height = imageHeight * frameWidth / imageWidth;
			y = (frameHeight - height) / 2;
		}
		return new Rect(x, y, width, height);
	}

	public static Rect getFillRectangle(int frameWidth, int frameHeight, int imageWidth, int imageHeight) {
		int x, y, width, height;
		if (frameWidth * imageHeight > frameHeight * imageWidth) {
			width = frameWidth;
			x = 0;
			height = imageHeight * frameWidth / imageWidth;
			y = (frameHeight - height) / 2;
		} else {
			width = imageWidth * frameHeight / imageHeight;
			x = (frameWidth - width) / 2;
			height = frameHeight;
			y = 0;
		}
		return new Rect(x, y, width, height);
	}


	/**
	 * fill array with 4 pairs x,y according to bounding box. Order of points will be:
	 * top-left, top-right, bottom-right, bottom-left
	 *
	 * @param points
	 * @param bb
	 */
	public static void fillPointsArrayFromBoundingBox(float[] points, GenericBoundingBoxResult bb) {
		points[0] = bb.topLeftX;
		points[1] = bb.topLeftY;
		points[2] = bb.topRightX;
		points[3] = bb.topRightY;
		points[4] = bb.bottomRightX;
		points[5] = bb.bottomRightY;
		points[6] = bb.bottomLeftX;
		points[7] = bb.bottomLeftY;
	}

	public static void fillGenericBBUsingPointsArr(GenericBoundingBoxResult bb, float[] points) {
		bb.topLeftX = points[0];
		bb.topLeftY = points[1];
		bb.topRightX = points[2];
		bb.topRightY = points[3];
		bb.bottomLeftX = points[4];
		bb.bottomLeftY = points[5];
		bb.bottomRightX = points[6];
		bb.bottomRightY = points[7];
		bb.x = Math.round(Math.min(bb.topLeftX, bb.bottomLeftX));
		bb.y = Math.round(Math.min(bb.topLeftY, bb.topRightY));
		bb.width = Math.round(Math.max(bb.topRightX, bb.bottomRightX)) - bb.x + 1;
		bb.height = Math.round(Math.max(bb.bottomLeftY, bb.bottomRightY)) - bb.y + 1;
	}

	public static float[] convertPointsToRatio(float[] origPoints, PointF ratio) {
		if (origPoints == null)
			return null;


		float[] convertedPoints = new float[origPoints.length];
		Matrix scaleMat = new Matrix();
		scaleMat.setScale(ratio.x, ratio.y);
		scaleMat.mapPoints(convertedPoints, origPoints);



//		for (int i = 0; i < origPoints.length; i += 2) {
//			convertedPoints[i] = origPoints[i] * ratio.x;
//			convertedPoints[i+1] = origPoints[i+1] * ratio.y;
//		}

		return convertedPoints;
	}


	/**
	 * Get screen orientation
	 *
	 * @param context the context
	 * @return the orientation
	 */
	public static int getScreenOrientation(Context context) {
		WindowManager wm = (WindowManager) CameraController.getInstance().getSystemService(Context.WINDOW_SERVICE);
		Display display = wm.getDefaultDisplay();

		int orientation = Configuration.ORIENTATION_UNDEFINED;
		if (display.getWidth() == display.getHeight()) {
			orientation = Configuration.ORIENTATION_SQUARE;
		} else {
			if (display.getWidth() < display.getHeight()) {
				orientation = Configuration.ORIENTATION_PORTRAIT;
			} else {
				orientation = Configuration.ORIENTATION_LANDSCAPE;
			}
		}
		return orientation;
	}



	public static void sortPointsClockwise(PointF[] points, PointF center) {

		for (int i = 0; i < points.length - 1; i++) {
			PointF lowest = points[i];
			int currentIndex = i;

			for (int j = i + 1; j < points.length; j++) {

				if (comparePointCW(lowest, points[j], center)) {
					lowest = points[j];
					currentIndex = j;
				}
			}
			if (currentIndex != i) {
				points[currentIndex] = points[i];
				points[i] = lowest;
			}
		}

	}

	private static boolean comparePointCW(PointF a, PointF b, PointF center) {

		if (a.x - center.x >= 0 && b.x - center.x < 0) {
			return true;
		}
		if (a.x - center.x < 0 && b.x - center.x >= 0) {
			return false;
		}
		if (a.x - center.x == 0 && b.x - center.x == 0) {
			if (a.y - center.y >= 0 || b.y - center.y >= 0) {
				return a.y > b.y;
			}
			return b.y > a.y;
		}

		// compute the cross product of vectors (center -> a) x (center -> b)
		double det = (a.x - center.x) * (b.y - center.y) - (b.x - center.x) * (a.y - center.y);
		if (det < 0) {
			return true;
		}
		if (det > 0) {
			return false;
		}

		// points a and b are on the same line from the center
		// check which point is closer to the center
		double d1 = (a.x - center.x) * (a.x - center.x) + (a.y - center.y) * (a.y - center.y);
		double d2 = (b.x - center.x) * (b.x - center.x) + (b.y - center.y) * (b.y - center.y);
		return d1 > d2;
	}

}