/*
 * 
 */
package com.topimagesystems.util;

import android.content.Context;
import android.content.res.AssetManager;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.os.Build;
import android.os.Environment;
import android.util.Log;

import com.topimagesystems.R;
import com.topimagesystems.controllers.imageanalyze.CameraManagerController;
import com.topimagesystems.controllers.imageanalyze.RawImagesFlowManager;
import com.topimagesystems.data.SessionResultParams;

import org.opencv.core.Core;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.MatOfByte;
import org.opencv.core.MatOfInt;
import org.opencv.core.Rect;
import org.opencv.imgcodecs.Imgcodecs;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;


// TODO: Auto-generated Javadoc
/**
 * The Class FileUtils.
 */
public class FileUtils {

    /** The Constant tag. */
    private static final String tag = Logger.makeLogTag("FileUtils");

    /** The temp path. */
    public static String tempPath = ".mobiflow";

    /** The Constant tempDebugPath. */
    public static final String tempDebugPath = ".debugmobiflow";

    /** The Constant tempVideoPath. */
    private static final String tempVideoPath = tempDebugPath + File.separator + "video";

    /** The Constant tempVideoValidPath. */
    private static final String tempVideoValidPath = tempVideoPath + File.separator + "valid";

    private static final String tempVideoInValidPath = tempVideoPath + File.separator + "invalid";

    /** The external storage writeable. */
    private static boolean externalStorageAvailable, externalStorageWriteable;
    public static String internalStorageLocation;
    public static String logFilePath;

    /**
     * Make app dir exists.
     *
     * @param context the context
     */
    public static void makeAppDirExists(final Context context) {
        Thread thread1 = new Thread() {
            public void run() {
                File tempFileDir = getFolder(context, tempDebugPath);
                if (!tempFileDir.exists()) {
                    tempFileDir.mkdirs();
                    Logger.i("FileUtils", tempDebugPath + " created");
                }
                File fileDir = getFolder(context, tempPath);
                createFolder(context, tempPath);
                if (!fileDir.exists()) {
                    fileDir.mkdirs();
                    Logger.i("FileUtils", tempPath + " created");
                }
                createFolder(context, tempDebugPath);
                createFolder(context, tempPath);
            }
        };
        thread1.start();
    }

    /**
     * Gets the high res image path.
     *
     * @param context the context
     * @return the high res image path
     */
    public static String getHighResImagePath(Context context) {
        File fileDir = getFolder(context, tempPath);
        return fileDir.getAbsolutePath();
    }

    /**
     * Gets the test image path.
     *
     * @param context the context
     * @return the test image path
     */
    public static String getTestImagePath(Context context) {

        File fileDir = Environment.getExternalStorageDirectory();
        return fileDir.getAbsolutePath() + "/"+ tempPath;
    }

    /**
     * Gets the current time.
     *
     * @return the current time
     */
    public static String getCurrentTime() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yy-MM-dd_HH.mm.ss", Locale.US);
        String currentDateAndTime = dateFormat.format(new Date());
        return currentDateAndTime;

    }

    public static void clearSessionImages(Context context) {
        //Logger.i(tag, "clear High res images folder");
        // sort files
        clearFiles(context,FileUtils.internalStorageLocation+"/.mobiflow");
        clearFiles(context,FileUtils.internalStorageLocation+"/.debugmobiflow");
    }

    /**
     * Clear log file.
     *
     * @param folder the folder
     */
    public static void clearLogFile(String folder) {
        try {
            File file = new File(folder);

            if (file.delete()) {
                //System.out.println(file.getName() + " is deleted!");
            } else {
                //	System.out.println("Delete operation is failed.");
            }

        } catch (Exception e) {

            e.printStackTrace();
        }

    }

    /**
     * Checks if is minimum cpu speed.
     *
     * @return true, if is minimum cpu speed
     * @throws IOException
     */
    public static boolean isMinimumCPUSpeed() {

        int maxFreq = -1;
        RandomAccessFile reader = null;
        try {
            reader = new RandomAccessFile("/sys/devices/system/cpu/cpu0/cpufreq/stats/time_in_state", "r");

            boolean done = false;
            while (!done) {
                String line = reader.readLine();
                if (null == line) {
                    done = true;
                    break;
                }
                String[] splits = line.split("\\s+");
                assert (splits.length == 2);
                int timeInState = Integer.parseInt(splits[1]);
                if (timeInState > 0) {
                    int freq = Integer.parseInt(splits[0]) / 1000;
                    if (freq > maxFreq) {
                        maxFreq = freq;
                    }
                }
            }
            reader.close();

        } catch (IOException ex) {
            ex.printStackTrace();

        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
        }

        Logger.i(tag, "device CPU speed is " + maxFreq);
        if (maxFreq < 1000 && getNumCores() <= 1) {
            return false;
        }
        return true;
    }

    /**
     * Gets the num cores.
     *
     * @return the num cores
     */
    public static int getNumCores() {
        // Private Class to display only CPU devices in the directory listing
        class CpuFilter implements FileFilter {
            @Override
            public boolean accept(File pathname) {
                // Check if filename is "cpu", followed by a single digit number
                if (Pattern.matches("cpu[0-9]+", pathname.getName())) {
                    return true;
                }
                return false;
            }
        }

        try {
            // Get directory containing CPU info
            File dir = new File("/sys/devices/system/cpu/");
            // Filter to only list the devices we care about
            File[] files = dir.listFiles(new CpuFilter());
            // Return the number of cores (virtual CPU devices)
            Logger.i(tag, "device cores number is " + files.length);
            return files.length;

        } catch (Exception e) {
            // Default to return 1 core
            return 0;
        }
    }

    /**
     * Gets the device name.
     *
     * @return the device name
     */
    public static String getDeviceName() {
        String manufacturer = Build.MANUFACTURER;
        String model = Build.MODEL;
        if (model.startsWith(manufacturer)) {
            return capitalize(model);
        } else {
            return capitalize(manufacturer) + " " + model;
        }
    }

    /**
     * Capitalize.
     *
     * @param s the s
     * @return the string
     */
    private static String capitalize(String s) {
        if (s == null || s.length() == 0) {
            return "";
        }
        char first = s.charAt(0);
        if (Character.isUpperCase(first)) {
            return s;
        } else {
            return Character.toUpperCase(first) + s.substring(1);
        }
    }

    /**
     * Clear memory.
     */
    public static void clearMemory() {
        SessionResultParams.tiffFront = null;
        SessionResultParams.jpegBWFront = null;
        SessionResultParams.grayscaleFront = null;
        SessionResultParams.colorFront = null;
        SessionResultParams.originalFront = null;
        SessionResultParams.tiffBack = null;
        SessionResultParams.jpegBWBack = null;
        SessionResultParams.grayscaleBack = null;
        SessionResultParams.colorBack = null;
        SessionResultParams.originalBack = null;

    }

    /**
     * Write to file.
     *
     * @param array the array
     * @param path the path
     *
     * @return true if saved successfully, false otherwise
     */
    public static boolean writeToFile(byte[] array, String path) {
        FileOutputStream stream = null;
        boolean success = true;
        try {
            if (array == null || path == null) {
                Logger.i(tag, "failed to create image byte array from, image may be corrupted " + path);
                success =  false;
            }
            if (!whiteListFiles(path)) {
                Logger.i(tag, "Invalid file name" + path);
                success =  false;
            }
            stream = new FileOutputStream(path);
            if (stream != null && array.length > 0) {
                stream.write(array);
            }

        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return false;
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (stream != null) {
                    stream.close();
                }
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }

        }
        return success;
    }

    private static boolean whiteListFiles(String path) {
        try {

            String[] validationPath = path.split("_");
            if (!validationPath[0].contains("FRONT") && !validationPath[0].contains("BACK")) {
                return false;
            }

            if (!validationPath[1].matches("([0-9]{2})-([0-9]{2})-([0-9]{2})") || !validationPath[2].matches("([0-9]{2}).([0-9]{2}).([0-9]{2})")) {
                return false;
            }

        } catch (Exception e) {
            Logger.e(tag, Log.getStackTraceString(e));
            return false;
        }
        return true;
    }

    /**
     * Gets the device model.
     *
     * @return the device model
     */
    public static String getDeviceModel() {
        return Build.MODEL;
    }

    /**
     * Gets the device brand.
     *
     * @return the device brand
     */
    public static String getDeviceBrand() {
        String manufacturer = Build.MANUFACTURER;
        return capitalize(manufacturer);

    }

    /**
     * Checks if is samsung s5.
     *
     * @return true, if is samsung s5
     */
    public static boolean isSamsungS5() {
        try {
            return (CameraManagerController.deviceBrand.toLowerCase(Locale.US).contains("samsung") && CameraManagerController.deviceModal.toLowerCase(Locale.US).contains("g900"));
        } catch (Exception e) {
            Logger.e(tag, Log.getStackTraceString(e));
            return false;
        }

    }
	public static boolean isSonyDevice(){
		if(CameraManagerController.deviceBrand.toLowerCase(Locale.US).contains("sony")){
			return  true;
		}
		return false;
	}

    /**
     * Checks if is samsung s4.
     *
     * @return true, if is samsung s4
     */
    public static boolean isSamsungS4() {
        return (CameraManagerController.deviceBrand.toLowerCase(Locale.US).contains("samsung") && (CameraManagerController.deviceModal.toLowerCase(Locale.US).equals("gt-i9500")
                || CameraManagerController.deviceModal.toLowerCase(Locale.US).equals("shv-e300k") || CameraManagerController.deviceModal.toLowerCase(Locale.US).equals("gt-i9505")
                || CameraManagerController.deviceModal.toLowerCase(Locale.US).equals("gt-i9506") || CameraManagerController.deviceModal.toLowerCase(Locale.US).equals("sch-i545")));
    }

    /**
     * Clear temp files.
     *
     * @param context the context
     * @param path the dir path
     */
    public static void clearFiles(final Context context,final String path) {
        Thread thread1 = new Thread() {
            public void run() {
                try {
                    File cacheDir = new File(path);
                    if (cacheDir.exists()) {
                        List<File> filesList = Arrays.asList(cacheDir.listFiles());
                        if (filesList == null) {
                            return;
                        }
                        int filesNumber = filesList.size();
                        for (int i = 0; i < filesNumber; i++) {
                            File curr = filesList.get(i);
                            Logger.i(tag, "deleting file " + curr.getAbsolutePath());
                            curr.delete();
                        }
                    }


                } catch (Exception e) {
                    e.printStackTrace();

                }
            }
        };
        thread1.start();

    }

    /**
     * Read raw resource.
     *
     * @param context the context
     * @param resourceId the resource id
     * @param fileName the file name
     * @return the string
     */
    public static String readRawResource(Context context, int resourceId, String fileName) {
        return readRawResource(context, resourceId, "micr", fileName);
    }

    /**
     * Read raw resource.
     *
     * @param context the context
     * @param resourceId the resource id
     * @param folderName the folder name
     * @param fileName the file name
     * @return the string
     */
    public static String readRawResource(Context context, int resourceId, String folderName, String fileName) {
        String result = "";
        InputStream is = null;
        FileOutputStream os = null;
        //FileOutputStream os2 = null;
        if (fileName == null || folderName == null || resourceId == -1) {
            return null;
        }
        try {
            is = context.getResources().openRawResource(resourceId);
            File localAssetDir = context.getDir(folderName, Context.MODE_PRIVATE);
            File localFile = new File(localAssetDir, fileName);

            //	String path = Environment.getExternalStorageDirectory().getAbsolutePath() + "/" + ".mobiflow/";
            if (localFile != null) {
                os = new FileOutputStream(localFile);
            }
            //os2 = new FileOutputStream(path+fileName);
            int size = is.available();
            byte[] buffer = new byte[size];

            int bytesRead;

            if (is != null && buffer != null && buffer.length > 0) {
                while ((bytesRead = is.read(buffer)) != -1) {
                    if (bytesRead > 0) {
                        os.write(buffer, 0, bytesRead);
                        //		os2.write(buffer, 0, bytesRead);
                    }
                }

            }

            result = localFile.getAbsolutePath();
        } catch (Exception e) {
            Logger.e(tag, "Error readRawResource: " + e.getMessage());
        } finally {
            closeInputStream(is);
            closeOutputStream(os);
        }
        return result;
    }

    public static String readFromAsset(Context ctx, String fileToRead, String fileToSave) {

        AssetManager assetManager = ctx.getAssets();
        ByteArrayOutputStream outputStream = null;
        FileOutputStream os = null;
        if (fileToRead == null || fileToSave == null) {
            return null;
        }
        InputStream inputStream = null;
        try {
            inputStream = assetManager.open(fileToRead);
            outputStream = new ByteArrayOutputStream();
            byte buf[] = new byte[1024];
            int len;
            try {
                while ((len = inputStream.read(buf)) != -1) {
                    if (os != null && len > 0) {
                        os.write(buf, 0, len);
                    }
                }

            } catch (IOException e) {
            }
        } catch (IOException e) {
        } finally {

            try {
                if (inputStream != null) {
                    inputStream.close();
                }
                if (outputStream != null) {
                    outputStream.close();
                }
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
        if (outputStream != null) {
            return outputStream.toString();
        } else {
            return null;
        }

    }

    public static void readRaw(Context ctx, int res_id, String fileName) {

        InputStream is = ctx.getResources().openRawResource(res_id);
        InputStreamReader isr = new InputStreamReader(is);
        BufferedReader br = new BufferedReader(isr, 8192); // 2nd arg is buffer
        // size

        // More efficient (less readable) implementation of above is the
        // composite expression
		/*
		 * BufferedReader br = new BufferedReader(new InputStreamReader(
		 * this.getResources().openRawResource(R.raw.textfile)), 8192);
		 */

        try {
            String test;
            while (true) {
                test = br.readLine();
                // readLine() returns null if no more lines in the file
                if (test == null)
                    break;
            }
            isr.close();
            is.close();
            br.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    /**
     * Read ocr mask raw resources.
     *
     * @param context the context
     * @param folderName the folder name
     * @return the string
     */
    public static String readOCRMaskRawResources(Context context, String folderName) throws Throwable{
        String result = "";
        result = readRawResource(context, R.raw.ocr_mask0, folderName, "0.png");
        result = result.substring(0, result.indexOf("0.png"));
        readRawResource(context, R.raw.ocr_mask1, folderName, "1.png");
        readRawResource(context, R.raw.ocr_mask2, folderName, "2.png");
        readRawResource(context, R.raw.ocr_mask3, folderName, "3.png");
        readRawResource(context, R.raw.ocr_mask4, folderName, "4.png");
        readRawResource(context, R.raw.ocr_mask5, folderName, "5.png");
        readRawResource(context, R.raw.ocr_mask6, folderName, "6.png");
        readRawResource(context, R.raw.ocr_mask7, folderName, "7.png");
        readRawResource(context, R.raw.ocr_mask8, folderName, "8.png");
        readRawResource(context, R.raw.ocr_mask9, folderName, "9.png");
        readRawResource(context, R.raw.ocr_mask11, folderName, "11.png");
        readRawResource(context, R.raw.ocr_mask12, folderName, "12.png");
        readRawResource(context, R.raw.ocr_mask13, folderName, "13.png");
        readRawResource(context, R.raw.ocr_mask14, folderName, "14.png");
        readRawResource(context, R.raw.ocr_mask15, folderName, "15.png");
        return result;
    }

    public static String readOCRMRZRawResources(Context context, String folderName) throws Throwable {
        String result = "";
        result = readRawResource(context, R.raw.mrz0, folderName, "0.xml");
        result = result.substring(0, result.indexOf("0.xml"));
        readRawResource(context, R.raw.mrz0, folderName, "0.xml");
        readRawResource(context, R.raw.mrz1, folderName, "1.xml");
        readRawResource(context, R.raw.mrz2, folderName, "2.xml");
        readRawResource(context, R.raw.mrz3, folderName, "3.xml");
        readRawResource(context, R.raw.mrz4, folderName, "4.xml");
        readRawResource(context, R.raw.mrz5, folderName, "5.xml");
        readRawResource(context, R.raw.mrz6, folderName, "6.xml");
        readRawResource(context, R.raw.mrz7, folderName, "7.xml");
        readRawResource(context, R.raw.mrz8, folderName, "8.xml");
        readRawResource(context, R.raw.mrz9, folderName, "9.xml");
        readRawResource(context, R.raw.mrz10, folderName, "10.xml");
        readRawResource(context, R.raw.mrz11, folderName, "11.xml");
        readRawResource(context, R.raw.mrz12, folderName, "12.xml");
        readRawResource(context, R.raw.mrz13, folderName, "13.xml");
        readRawResource(context, R.raw.mrz14, folderName, "14.xml");
        readRawResource(context, R.raw.mrz15, folderName, "15.xml");
        readRawResource(context, R.raw.mrz16, folderName, "16.xml");
        readRawResource(context, R.raw.mrz17, folderName, "17.xml");
        readRawResource(context, R.raw.mrz18, folderName, "18.xml");
        readRawResource(context, R.raw.mrz19, folderName, "19.xml");
        readRawResource(context, R.raw.mrz20, folderName, "20.xml");
        readRawResource(context, R.raw.mrz21, folderName, "21.xml");
        readRawResource(context, R.raw.mrz22, folderName, "22.xml");
        readRawResource(context, R.raw.mrz23, folderName, "23.xml");
        readRawResource(context, R.raw.mrz24, folderName, "24.xml");
        readRawResource(context, R.raw.mrz25, folderName, "25.xml");
        readRawResource(context, R.raw.mrz26, folderName, "26.xml");
        readRawResource(context, R.raw.mrz27, folderName, "27.xml");
        readRawResource(context, R.raw.mrz28, folderName, "28.xml");
        readRawResource(context, R.raw.mrz29, folderName, "29.xml");
        readRawResource(context, R.raw.mrz30, folderName, "30.xml");
        readRawResource(context, R.raw.mrz31, folderName, "31.xml");
        readRawResource(context, R.raw.mrz32, folderName, "32.xml");
        readRawResource(context, R.raw.mrz33, folderName, "33.xml");
        readRawResource(context, R.raw.mrz34, folderName, "34.xml");
        readRawResource(context, R.raw.mrz35, folderName, "35.xml");
        readRawResource(context, R.raw.mrz36, folderName, "36.xml");
        //	Mat img = Highgui.imread(result);
        return result;
    }

    public static String readOCRPanRawResources(Context context, String folderName) throws Throwable {
        String result = "";
        //input.grayscaleImageCompression
        result = readRawResource(context, R.raw.pan0, folderName, "0.xml");
        result = result.substring(0, result.indexOf("0.xml"));
        readRawResource(context, R.raw.pan0, folderName, "0.xml");
        readRawResource(context, R.raw.pan1, folderName, "1.xml");
        readRawResource(context, R.raw.pan2, folderName, "2.xml");
        readRawResource(context, R.raw.pan3, folderName, "3.xml");
        readRawResource(context, R.raw.pan4, folderName, "4.xml");
        readRawResource(context, R.raw.pan5, folderName, "5.xml");
        readRawResource(context, R.raw.pan6, folderName, "6.xml");
        readRawResource(context, R.raw.pan7, folderName, "7.xml");
        readRawResource(context, R.raw.pan8, folderName, "8.xml");
        readRawResource(context, R.raw.pan9, folderName, "9.xml");
        readRawResource(context, R.raw.pan10, folderName, "10.xml");
        readRawResource(context, R.raw.pan11, folderName, "11.xml");
        readRawResource(context, R.raw.pan12, folderName, "12.xml");
        readRawResource(context, R.raw.pan13, folderName, "13.xml");
        readRawResource(context, R.raw.pan14, folderName, "14.xml");
        readRawResource(context, R.raw.pan15, folderName, "15.xml");
        readRawResource(context, R.raw.pan16, folderName, "16.xml");
        readRawResource(context, R.raw.pan17, folderName, "17.xml");
        readRawResource(context, R.raw.pan18, folderName, "18.xml");
        readRawResource(context, R.raw.pan19, folderName, "19.xml");
        readRawResource(context, R.raw.pan20, folderName, "20.xml");
        readRawResource(context, R.raw.pan21, folderName, "21.xml");
        readRawResource(context, R.raw.pan22, folderName, "22.xml");
        readRawResource(context, R.raw.pan23, folderName, "23.xml");
        readRawResource(context, R.raw.pan24, folderName, "24.xml");
        readRawResource(context, R.raw.pan25, folderName, "25.xml");
        readRawResource(context, R.raw.pan26, folderName, "26.xml");
        readRawResource(context, R.raw.pan27, folderName, "27.xml");
        readRawResource(context, R.raw.pan28, folderName, "28.xml");
        readRawResource(context, R.raw.pan29, folderName, "29.xml");
        readRawResource(context, R.raw.pan30, folderName, "30.xml");
        readRawResource(context, R.raw.pan31, folderName, "31.xml");
        readRawResource(context, R.raw.pan32, folderName, "32.xml");
        readRawResource(context, R.raw.pan33, folderName, "33.xml");
        readRawResource(context, R.raw.pan34, folderName, "34.xml");
        readRawResource(context, R.raw.pan35, folderName, "35.xml");
        readRawResource(context, R.raw.pan36, folderName, "36.xml");
        //	Mat img = Highgui.imread(result);
        return result;
    }

    /**
     * Store bitmap in temp folder.
     *
     * @param context the context
     * @param bitmap the bitmap
     * @return the string
     */
    public static String storeBitmapInTempFolder(Context context, Bitmap bitmap) {
        String filePath = getTempImageFileName(context);
        return storeBitmap(bitmap, filePath);
    }

    /**
     * Store bitmap in debug folder.
     *
     * @param context the context
     * @param bitmap the bitmap
     * @param rect the rect
     * @param fileName the file name
     * @return the string
     */
    public static String storeBitmapInDebugFolder(Context context, Bitmap bitmap, Rect rect, String fileName) {
        // if (CameraManagerController.isDebug==false)
        // return null;
        String filePath = getDebugFilePath(context, fileName);
        String result = "";
        if (bitmap != null && rect != null) {
            result = storeBitmap(bitmap, filePath);
            bitmap = drawRectOnBitmap(bitmap, rect);
            // storing bitmap with border
            int lastIndexOfDot = filePath.lastIndexOf(".");
            StringBuilder builder = new StringBuilder();
            builder.append(filePath.substring(0, lastIndexOfDot)).append("_border.").append(filePath.substring(lastIndexOfDot + 1, filePath.length()));
            filePath = builder.toString();
            storeBitmap(bitmap, filePath);
        } else if (bitmap != null) {
            result = storeBitmap(bitmap, filePath);
        }
        return result;
    }

    /**
     * Draw rect on bitmap.
     *
     * @param srcBitmap the src bitmap
     * @param rect the rect
     * @return the bitmap
     */
    private static Bitmap drawRectOnBitmap(Bitmap srcBitmap, Rect rect) {
        RectF targetRect = new RectF(rect.x, rect.y, rect.x + rect.width, rect.y + rect.height);
        Canvas canvas = new Canvas(srcBitmap);
        Paint paint = new Paint();
        paint.setColor(Color.RED);
        paint.setAntiAlias(true);
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(3);

        canvas.drawRect(targetRect, paint);
        return srcBitmap;
    }

    /**
     * Gets the debug file path.
     *
     * @param context the context
     * @param fileName the file name
     * @return the debug file path
     */
    private static String getDebugFilePath(Context context, String fileName) {
        File folder = getFolder(context, tempDebugPath);

        if (!fileName.endsWith(".jpeg") && !fileName.endsWith(".jpg")) {
            fileName += ".jpg";
        }

        String filePath = folder + File.separator + fileName;
        return filePath;
    }

    /**
     * Store video frame.
     *
     * @param context the context
     * @param bitmap the bitmap
     * @param fileName the file name
     * @param isValid the is valid
     * @return the string
     */
    public static String storeVideoFrame(Context context, Bitmap bitmap, String fileName, boolean isValid) {
        // if (CameraManagerController.isDebug)
        // return null;
        if (!fileName.endsWith(".jpeg") && !fileName.endsWith(".jpg")) {
            fileName += ".jpg";
        }

        String filePath = (isValid ? getRootDirForTempValidVideo(context) : getRootDirForTempInValidVideo(context)) + File.separator + fileName;
        return storeBitmap(bitmap, filePath);
    }

    /**
     * Store bitmap in temp folder.
     *
     * @param context the context
     * @param bitmap the bitmap
     * @param name the name
     * @return the string
     */
    public static String storeBitmapInTempFolder(Context context, Bitmap bitmap, String name) {
        if (!name.endsWith(".jpeg") && !name.endsWith(".jpg")) {
            name += ".jpg";
        }
        String filePath = getTempFilePath(context) + name;
        return storeBitmap(bitmap, filePath);
    }

    /**
     * Gets the bitmap.
     *
     * @param context the context
     * @param filePath the file path
     * @return the bitmap
     */
    public static Bitmap getBitmap(Context context, final String filePath) {
        if (!whiteListFiles(filePath)) {
            Logger.e(tag, "Invalid file name" + filePath);
            return null;
        }
        Bitmap bitmap = decodeFile(new File(filePath), -1, -1);
        return bitmap;
    }

    /**
     * Decode file.
     *
     * @param f the f
     * @param maxWidthResolution the max width resolution
     * @param maxHeightResolution the max height resolution
     * @return the bitmap
     */
    public static Bitmap decodeFile(File f, int maxWidthResolution, int maxHeightResolution) {
        FileInputStream fileInputStream = null;
        try {
            // decode image size
            BitmapFactory.Options o = new BitmapFactory.Options();
            o.inJustDecodeBounds = true;

            fileInputStream = new FileInputStream(f);
            BitmapFactory.decodeStream(fileInputStream, null, o);
            fileInputStream = null;

            // Find the correct scale value. It should be the power of 2.
            int width_tmp = o.outWidth, height_tmp = o.outHeight;
            int scale = 1;

            if (maxWidthResolution > -1 && maxHeightResolution > -1) {
                while (true) {
                    if (width_tmp / 2 < maxWidthResolution || height_tmp / 2 < maxHeightResolution)
                        break;

                    width_tmp /= 2;
                    height_tmp /= 2;
                    scale *= 2;
                }
            }

            // decode with inSampleSize
            BitmapFactory.Options o2 = new BitmapFactory.Options();
            o2.inSampleSize = scale;
            o2.inPreferredConfig = Config.RGB_565;

            fileInputStream = new FileInputStream(f);
            return BitmapFactory.decodeStream(fileInputStream, null, o2);
        } catch (Exception e) {
            Logger.e(tag, "FileUtils.decodeFile", e);
        } finally {
            close(fileInputStream);
        }
        return null;
    }

    /**
     * Close.
     *
     * @param fis the fis
     */
    private static void close(InputStream fis) {
        if (fis != null) {
            try {
                fis.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Gets the temp image file name.
     *
     * @param context the context
     * @return the temp image file name
     */
    private static String getTempImageFileName(Context context) {
        return getTempFilePath(context) + "temp.jpg";
    }

    /**
     * Gets the root dir for temp image.
     *
     * @param context the context
     * @return the root dir for temp image
     */
    public static File getRootDirForTempImage(Context context) {
        return getFolder(context, tempPath);
    }

    /**
     * Gets the root dir for temp valid video.
     *
     * @param context the context
     * @return the root dir for temp valid video
     */
    public static File getRootDirForTempValidVideo(Context context) {

        return getFolder(context, tempVideoValidPath);
    }

    /**
     * Gets the root dir for temp in valid video.
     *
     * @param context the context
     * @return the root dir for temp in valid video
     */
    public static File getRootDirForTempInValidVideo(Context context) {

        return getFolder(context, tempVideoInValidPath);
    }

    /**
     * Gets the folder.
     *
     * @param context the context
     * @param path the path
     * @return the folder
     */
    public static File getFolder(Context context, String path) {
        File cacheDir;
        cacheDir = new File(context.getFilesDir(), path); // write on device
        // application																// folder -																// internal
        // storage
        if (!cacheDir.exists())
            cacheDir.mkdirs();

        return cacheDir;
    }

    /**
     * Creates the folder.
     *
     * @param context the context
     * @param path the path
     */
    public static void createFolder(Context context, String path) {
        File internalDir = context.getDir(path, Context.MODE_PRIVATE);
        if (!internalDir.exists())
            internalDir.mkdirs();
    }

    /**
     * Gets the byte array image.
     *
     * @param context the context
     * @param filePath the file path
     * @return the byte array image
     */
    public static byte[] getByteArrayImage(Context context, final String filePath) {
        if (filePath == null) {
            return null;
        }
        if (!whiteListFiles(filePath)) {
            Logger.i(tag, "Invalid file name" + filePath);
            return null;
        }
        byte[] data = null;
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(filePath);
            data = new byte[fis.available()];
            while (fis.read(data) > -1) {
            }
            fis.close();
        } catch (FileNotFoundException e) {
            Logger.i(tag, "File not found: " + e.getMessage());

        } catch (IOException e) {
            Logger.i(tag, "Error accessing file: " + e.getMessage());
        } finally {
            closeInputStream(fis);
        }
        return data;
    }

    /**
     * Gets the temp file path.
     *
     * @param context the context
     * @return the temp file path
     */
    public static String getTempFilePath(Context context) {
        return getRootDirForTempImage(context) + File.separator;
    }

    /**
     * Gets the temp image path.
     *
     * @param context the context
     * @param imageName the image name
     * @return the temp image path
     */
    public static String getTempImagePath(Context context, String imageName) {
        return getTempFilePath(context) + imageName;
    }

    /**
     * /**.
     *
     * @param bitmap the bitmap
     * @param filePath the file path
     * @return the string
     */
    public static String storeBitmap(Bitmap bitmap, String filePath) {
        FileOutputStream out = null;
        try {
            File file = new File(filePath);
            out = new FileOutputStream(file);
            bitmap.compress(Bitmap.CompressFormat.JPEG, 80, out);
            return file.getAbsolutePath();
        } catch (Exception e) {
            Logger.e(tag, "Error storing a bitmap: " + e.getMessage());
        } finally {
            closeOutputStream(out);
        }
        return null;
    }

    /**
     * Store byte array image in temp folder.
     *
     * @param context the context
     * @param data the data
     * @param fileName the file name
     * @param isDebug the is debug
     * @return the string
     */
    public static String storeByteArrayImageInTempFolder(Context context, byte[] data, String fileName, boolean isDebug) {
        if (fileName == null) {
            return null;
        }
        FileOutputStream fos = null;
        String filePath = getTempFilePath(context) + fileName + ".jpg";
        //	String debugFilePath = getDebugFilePath(context, fileName + "_" + System.currentTimeMillis());

        File fileDir = getFolder(context, tempDebugPath);
        if (!fileDir.exists()) {
            fileDir.mkdirs();
        }
        try {
            fos = new FileOutputStream(filePath);
            if (fos != null && data.length > 0) {
                fos.write(data);

            }
            if (isDebug && fos != null && data.length > 0) {
                fos.write(data);
            }
        } catch (FileNotFoundException e) {
            Logger.i(tag, "File not found: " + e.getMessage());
        } catch (IOException e) {
            Logger.i(tag, "Error accessing file: " + e.getMessage());
        } finally {
            try {
                fos.close();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
        return filePath;
    }

    /**
     * Rotate mat.
     *
     * @param image the image
     * @param angle the angle
     * @return the mat
     */
    public static Mat rotateMat(Mat image, int angle) {

        Core.transpose(image, image);
        Core.flip(image, image, 1);
        return image;
    }

    public static Mat rotateMatInAngle(Mat src, int angle)
    {
        if(angle == 270 || angle == -90){
            // Rotate clockwise 270 degrees
            Core.transpose(src, src);
            Core.flip(src, src, 0);
        }else if(angle == 180 || angle == -180){
            // Rotate clockwise 180 degrees
            Core.flip(src, src, -1);
        }else if(angle == 90 || angle == -270){
            // Rotate clockwise 90 degrees
            Core.transpose(src, src);
            Core.flip(src, src, 1);
        }else if(angle == 360 || angle == 0){

        }
        return src;
    }

	/*
	 * public static byte[] rotateTiff(String imagePath, int angle) { byte[]
	 * origImage = getByteArray(imagePath); Bitmap
	 * bmp=BitmapFactory.decodeByteArray(origImage,0,origImage.length); Matrix
	 * matrix = new Matrix(); matrix.postRotate(90); Bitmap rotatedBitmap =
	 * Bitmap.createBitmap(bmp, 0, 0, bmp.getWidth(), bmp.getHeight(), matrix,
	 * true); int bytes = rotatedBitmap.getByteCount(); //or we can calculate
	 * bytes this way. Use a different value than 4 if you don't use 32bit
	 * images. //int bytes = b.getWidth()*b.getHeight()*4;
	 *
	 * ByteBuffer buffer = ByteBuffer.allocate(bytes); //Create a new buffer
	 * rotatedBitmap.copyPixelsToBuffer(buffer); //Move the byte data to the
	 * buffer
	 *
	 * byte[] array = buffer.array(); //Get the underlying array containing the
	 * data
	 *
	 * ImageView image=new ImageView(this); image.setImageBitmap(bmp);
	 *
	 *
	 * Mat image = convertByteImageToMat(origImage); int flipSide = angle > 0 ?
	 * 1 : 0; int number = Math.abs(angle / 90); Core.transpose(image, image);
	 * Core.flip(image, image, flipSide); Highgui.imwrite(imagePath, image);
	 * byte[] bytes = getByteArray(imagePath); return bytes;
	 *
	 * }
	 */

    /**
     * Rotate image.
     *
     * @param c the c
     * @param imagePath the image path
     * @param angle the angle
     * @param image the image
     * @return the byte[]
     */

    public static byte[] rotateImage(Context c, String imagePath, int angle, Mat image) {
        try {
            Mat rotateImage;
            if (image == null) {

                image = Imgcodecs.imread(imagePath, Imgcodecs.CV_LOAD_IMAGE_UNCHANGED);
            }

            angle = ((angle / 90) % 4) * 90;

            int flipSide = angle > 0 ? 1 : 0;
            int number = Math.abs(angle / 90);

            Core.transpose(image, image);
            Core.flip(image, image, flipSide);

            String extension = ".jpg";
            if (imagePath != null) {

                extension = imagePath.substring(imagePath.lastIndexOf("."), imagePath.length()).toLowerCase(Locale.US);
            }
            if (extension.equals(".tiff")) { // cannot get the byte from
                // imencode
                Imgcodecs.imwrite(imagePath, image);
                byte[] bytes = getByteArray(imagePath);
                return bytes;

            }
            MatOfByte buffer = new MatOfByte();
            Imgcodecs.imencode(extension, image, buffer);

            byte[] bytes = buffer.toArray();
            return bytes;

        } catch (Exception e) {

            Logger.e(tag, "failed to rotate image");
            return null;
        }

    }

    /**
     * Gets the byte array.
     *
     * @param filePath the file path
     * @return the byte array
     */

    public static byte[] getByteArray(final String filePath) {
        if (filePath == null) {
            return null;
        }
        if (!whiteListFiles(filePath)) {
            Logger.e(tag, "Invalid file name" + filePath);
            return null;
        }
        byte[] data = null;
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(filePath);
            data = new byte[fis.available()];
            while (fis.read(data) > -1) {
            }
            fis.close();
        } catch (FileNotFoundException e) {
            Logger.e(tag, "File not found: " + e.getMessage());
        } catch (IOException e) {
            Logger.e(tag, "Error accessing file: " + e.getMessage());
        } finally {
            closeInputStream(fis);
        }
        return data;
    }

    /**
     * Delete file.
     *
     * @param imagePath the image path
     * @return true, if successful
     */
    public static boolean deleteFile(final String imagePath) {
        if (StringUtils.isEmptyOrNull(imagePath)) {
            return false;
        }
        if (!whiteListFiles(imagePath)) {
            Logger.e(tag, "Invalid file name" + imagePath);
            return false;
        }

        try {
            File file = new File(imagePath);
            return file.delete();
        } catch (Exception e) {

        }

        return false;
    }

    /**
     * Close input stream.
     *
     * @param is the is
     */
    private static void closeInputStream(InputStream is) {
        if (is != null) {
            try {
                is.close();
            } catch (IOException e) {
                Logger.e(tag, "Error closing a stream: " + e.getMessage());
            }
        }
        is = null;
    }





    /**
     * Adds the to log file.
     *
     * @param value the value
     * @param c the c
     */
    public static void addToLogFile(String value, Context c) {
//		if (c == null || value == null) {
//			return;
//		}
//		BufferedWriter fos = null;
//		try {
//			File f = new File(logFilePath);
//			if (!f.exists()){
//				f.createNewFile();
//				Logger.i(tag, "creating new File " + FileUtils.logFilePath);
//			}
//			fos = new BufferedWriter(new FileWriter(logFilePath,true));
//			if (value != null && fos != null) {
//				fos.write(value);
//				fos.newLine();
//			}
//		} catch (FileNotFoundException e) {
//			Logger.e(tag, Log.getStackTraceString(e));
//		} catch (IOException e) {
//			Logger.e(tag, Log.getStackTraceString(e));
//		} finally {
//			close(fos);
//		}
        addToLogFile(null, value, c);
    }

    /**
     * Adds the to log file.
     *
     * @param tag the tag
     * @param value the value
     * @param c the c
     */
    public static void addToLogFile(String tag, String value, Context c) {
        addToLogFile(tag, value, c, logFilePath);
//		BufferedWriter fos = null;
//		if (c == null || value == null)
//			return;
//
//		try {
//
//			File f = new File(logFilePath);
//			if (!f.exists()){
//			f.createNewFile();
//				Logger.i((tag == null)? FileUtils.tag: tag,"creating new File " + FileUtils.logFilePath);
//			}
//			fos = new BufferedWriter(new FileWriter(FileUtils.logFilePath,true));
//			if (fos != null && value != null) {
//				fos.write(((tag == null)?FileUtils.tag :((new Date()).getTime() + " " + tag)) + " " + value);
//				fos.newLine();
//			}
//		} catch (FileNotFoundException e) {
//			Logger.e(FileUtils.tag, Log.getStackTraceString(e));
//		} catch (IOException e) {
//			Logger.e(FileUtils.tag, Log.getStackTraceString(e));
//		} finally {
//			close(fos);
//		}

        RawImagesFlowManager.handleLog((tag == null)? FileUtils.tag: tag, value);

    }

    public static void addToLogFile(String tag, String value, Context c, String path) {
        BufferedWriter fos = null;
        if (c == null || value == null || path == null)
            return;

        try {

            File f = new File(path);
            if (!f.exists()){
                f.createNewFile();
                Logger.i((tag == null)? FileUtils.tag: tag,"creating new File " +path);
            }
            fos = new BufferedWriter(new FileWriter(path,true));
            if (fos != null && value != null) {
                fos.write(((tag == null)?FileUtils.tag :((new Date()).getTime() + " " + tag)) + " " + value);
                fos.newLine();
            }
        } catch (FileNotFoundException e) {
            Logger.e(FileUtils.tag, Log.getStackTraceString(e));
        } catch (IOException e) {
            Logger.e(FileUtils.tag, Log.getStackTraceString(e));
        } finally {
            close(fos);
        }


    }


    public static void createTestLogFile(String tag, String value, String fileLocation, Context c) {
        BufferedWriter fos = null;
        if (c == null || value == null && tag != null) {
            return;
        }
        try {

            File f = new File(fileLocation);
            if (!f.exists()){
                Logger.i(tag, "creating new Test File " +fileLocation);
                f.createNewFile();
            }
            fos = new BufferedWriter(new FileWriter(fileLocation,true));
            if (fos != null && value != null && tag != null) {
                fos.write((new Date()).getTime() + " " + tag + " " + value);
                fos.newLine();
            }
        } catch (FileNotFoundException e) {
        } catch (IOException e) {
        } finally {
            close(fos);
        }
    }

    /**
     * Adds the to log file.
     *
     * @param tag the tag
     * @param value the value
     * @param fileName the file name
     * @param c the c
     */

    public static void modifySingleImgeLogFile(String tag, String value, String fileName, Context c) {
        if (c == null || fileName == null || tag == null) {
            return;
        }
        BufferedWriter fos = null;

        try {
            if (fileName != null) {
                int suffixLocation = 0;
                ;
                fileName = fileName.toLowerCase(Locale.US);
                if (fileName.endsWith("jpg")) {
                    suffixLocation = fileName.indexOf(".jpg");
                }
                if (fileName.endsWith("jpeg")) {
                    suffixLocation = fileName.indexOf(".jpeg");
                }
                String newFilePath = fileName.substring(fileName.lastIndexOf("/") + 1, suffixLocation) + "_result" + ".txt";
                File f = new File(newFilePath);
                if (!f.exists()) {
                    Logger.i(tag, "creating single Test File " + fileName);
                    fileName = fileName.substring(0, fileName.lastIndexOf("/") + 1);
                    fos = new BufferedWriter(new FileWriter(fileName + newFilePath, true));

                }
            }
            if (fos != null && value != null) {
                fos.write(value);
                fos.newLine();
            }
        } catch (FileNotFoundException e) {
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            close(fos);
        }
    }

    /**
     * Close.
     *
     * @param os the os
     */
    private static void close(BufferedWriter os) {
        if (os != null) {
            try {
                os.close();
            } catch (IOException e) {
                Logger.e(tag, "Error closing a stream: " + e.getMessage());
            }
        }
        os = null;
    }

    /**
     * Close output stream.
     *
     * @param os the os
     */
    private static void closeOutputStream(OutputStream os) {
        if (os != null) {
            try {
                os.close();
            } catch (IOException e) {
                Logger.e(tag, "Error closing a stream: " + e.getMessage());
            }
        }
        os = null;
    }

    /**
     * Rect to int array.
     *
     * @param rect the rect
     * @return the int[]
     */
    public static int[] rectToIntArray(Rect rect) {
        if (rect == null) {
            return null;
        }
        int arrayRect[] = new int[4];
        arrayRect[0] = rect.x;
        arrayRect[1] = rect.y;
        arrayRect[2] = rect.width;
        arrayRect[3] = rect.height;
        return arrayRect;
    }

    /**
     * Array to rect.
     *
     * @param arr the arr
     * @return the rect
     */
    public static Rect arrayToRect(int[] arr) {
        Rect rectResult = new Rect(arr[0], arr[1], arr[2], arr[3]);
        return rectResult;
    }

    /**
     * Convert byte image to mat.
     *
     * @param imageData the image data
     * @return the mat
     */
    public static Mat convertByteImageToMat(byte[] imageData) {
        Mat imageMat = new Mat(1, imageData.length, CvType.CV_16U);
        imageMat.put(0, 0, imageData);
        return imageMat;
    }

    /**
     * Convert byte image to mat.
     *
     * @param imageData the image data
     * @return the mat
     */
    public static Mat convertByteImageToMat(byte[] imageData, int cvType) {
        Mat imageMat = new Mat(1, imageData.length, cvType);
        imageMat.put(0, 0, imageData);
        return imageMat;
    }

    /**
     * Convert jpg mat to byte.
     *
     * @param result_mat the result_mat
     * @return the byte[]
     */
    public static byte[] convertJpgMatToByte(Mat result_mat) {
        MatOfByte buffer = new MatOfByte();
        Imgcodecs.imencode(".jpg", result_mat, buffer);

        byte[] bytes = buffer.toArray();
        return bytes;

    }

    public static byte[] convertJpgMatToByte(Mat result_mat, float compression) {
        MatOfByte buffer = new MatOfByte();
        MatOfInt params = new MatOfInt(Imgcodecs.IMWRITE_JPEG_QUALITY, (int) (compression * 100));

        Imgcodecs.imencode(".jpg", result_mat, buffer, params);

        byte[] bytes = buffer.toArray();
        return bytes;

    }

    /**
     * Last file modified.
     *
     * @param dir the dir
     * @param after the after
     * @return the file
     */
    public static File lastFileModified(String dir, String after) {
        File fl = new File(dir);
        File[] files = fl.listFiles(new FileFilter() {
            public boolean accept(File file) {
                return file.isFile();
            }
        });
        long lastMod = Long.MIN_VALUE;
        File choise = null;
        for (File file : files) {
            if (file.lastModified() > lastMod) {
                if (after != null && file.getPath().contains(after)) {

                } else {
                    choise = file;
                    lastMod = file.lastModified();
                }

            }

        }
        return choise;
    }

    /**
     * Check storage.
     *
     * @param c the c
     */
    private static void checkStorage(Context c) {
        // Get the external storage's state

        String state = Environment.getExternalStorageState();
        if (state.equals(Environment.MEDIA_MOUNTED) && checkWriteExternalPermission(c)) {
            // Storage is available and writeable
            externalStorageAvailable = externalStorageWriteable = true;
        } else if (state.equals(Environment.MEDIA_MOUNTED_READ_ONLY)) {
            // Storage is only readable
            externalStorageAvailable = true;
            externalStorageWriteable = false;
        } else {
            // Storage is neither readable nor writeable
            externalStorageAvailable = externalStorageWriteable = false;
        }
    }

    /**
     * Check write external permission.
     *
     * @param c the c
     * @return true, if successful
     */
    private static boolean checkWriteExternalPermission(Context c) {

        String permission = "android.permission.WRITE_EXTERNAL_STORAGE";
        int res = c.checkCallingPermission(permission);
        //int res = c.checkCallingOrSelfPermission(permission);
        return (res == c.getPackageManager().PERMISSION_GRANTED);
    }

    /**
     * Checks the state of the external storage.
     *
     * @param c the c
     * @return True if the external storage is available, false otherwise.
     */
    public boolean isExternalStorageAvailable(Context c) {
        checkStorage(c);
        return externalStorageAvailable;
    }

    /**
     * Checks the state of the external storage.
     *
     * @param c the c
     * @return True if the external storage is writeable, false otherwise.
     */
    public boolean isExternalStorageWriteable(Context c) {
        checkStorage(c);
        return externalStorageWriteable;
    }

    static public void copyFile(File source, File destination) throws IOException {
        InputStream input = new FileInputStream(source);
        OutputStream output = new FileOutputStream(destination);
        try {
            byte[] buffer = new byte[1024];
            int length;
            while ((length = input.read(buffer)) > 0) {
                output.write(buffer, 0, length);
            }
        }finally {
            closeInputStream(input);
            closeOutputStream(output);
        }
    }



}
