package com.topimagesystems.ui;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.graphics.Region;
import android.graphics.Shader;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;

import com.topimagesystems.R;
import com.topimagesystems.util.UserInterfaceUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by jhansi on 28/03/15.
 */
@TargetApi(Build.VERSION_CODES.HONEYCOMB)
public class SelectionCroppingView extends FrameLayout implements View.OnTouchListener{

    protected Context context;

    private SelectionCroppingView selectionCroppingView;

    private ImageView pointer1;
    private ImageView pointer2;
    private ImageView pointer3;
    private ImageView pointer4;
    private ImageView midPointer13;
    private ImageView midPointer12;
    private ImageView midPointer34;
    private ImageView midPointer24;
    private Paint rectPaint;
    private Paint movingPointerPaint = new Paint();
    private Drawable validPointerDrawable;
    private Drawable invalidPointerDrawable;
    private float pointerRadius;

    private boolean isZooming = false;
    private float magnifierScale;
    private int magnifierRadius;
    private int magnifierDistance;
    private PointF zoomPosition;
    private Matrix magnifierMatrix;
    private Bitmap magnifierBitmap;
    private BitmapShader magnifierShader;
    private Paint magnifierPaint = new Paint();
    private Path magnifierPath = new Path();
    private Drawable magnifierCircleDrawable;

    private boolean isValidRect = true;

    private boolean isPointsAlreadySet = false;

    private float minX;
    private float minY;
    private float maxX;
    private float maxY;

    private View movingView = null;

    private ISelectionListener selectionListener;

    public SelectionCroppingView(Context context) {
        super(context);
        this.context = context;
        init();
    }

    public SelectionCroppingView(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.context = context;
        init();
    }

    public SelectionCroppingView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        this.context = context;
        init();
    }

    public interface ISelectionListener {
        void onSelectionChanged();
        void onSelectionStarted();
    }

    private void init() {
        setDrawingCacheEnabled(false);
        selectionCroppingView = this;

        pointer1 = getImageView(0, 0);
        pointer1.post(new Runnable() {
            @Override
            public void run() {
                pointerRadius = pointer1.getHeight() / 2;
            }
        });
        pointer2 = getImageView(getWidth(), 0);
        pointer3 = getImageView(0, getHeight());
        pointer4 = getImageView(getWidth(), getHeight());
        midPointer13 = getImageView(0, getHeight() / 2);
        midPointer13.setOnTouchListener(new MidPointTouchListenerImpl(pointer1, pointer3));

        midPointer12 = getImageView(0, getWidth() / 2);
        midPointer12.setOnTouchListener(new MidPointTouchListenerImpl(pointer1, pointer2));

        midPointer34 = getImageView(0, getHeight() / 2);
        midPointer34.setOnTouchListener(new MidPointTouchListenerImpl(pointer3, pointer4));

        midPointer24 = getImageView(0, getHeight() / 2);
        midPointer24.setOnTouchListener(new MidPointTouchListenerImpl(pointer2, pointer4));

        addView(pointer1);
        addView(pointer2);
        addView(midPointer13);
        addView(midPointer12);
        addView(midPointer34);
        addView(midPointer24);
        addView(pointer3);
        addView(pointer4);

        rectPaint = new Paint();
        rectPaint.setColor(getResources().getColor(R.color.croppingValidColor));
        rectPaint.setStrokeWidth(2);
        rectPaint.setAntiAlias(true);

        movingPointerPaint.setColor(getResources().getColor(R.color.croppingPointerMoving));
        movingPointerPaint.setAntiAlias(true);

        zoomPosition = new PointF(0, 0);
        magnifierMatrix = new Matrix();

        Resources res = getResources();
        magnifierRadius = res.getDimensionPixelSize(R.dimen.cropping_magnifier_circle_size) / 2;
        magnifierDistance = res.getDimensionPixelOffset(R.dimen.cropping_magnifier_distance);
        magnifierScale = Float.valueOf(res.getString(R.string.cropping_magnifier_scale));
        magnifierCircleDrawable = res.getDrawable(R.drawable.cropping_magnifier);
        validPointerDrawable = res.getDrawable(R.drawable.cropping_pointer);
        invalidPointerDrawable = res.getDrawable(R.drawable.cropping_pointer_invalid);

        setBoundaries();
    }

    @Override
    protected void attachViewToParent(View child, int index, ViewGroup.LayoutParams params) {
        super.attachViewToParent(child, index, params);
    }

    @Override
    protected void detachViewFromParent(View child){
        if (magnifierBitmap != null) {
            if (!magnifierBitmap.isRecycled())
                magnifierBitmap.recycle();
            magnifierBitmap = null;
        }
        selectionListener = null;
        super.detachViewFromParent(child);
    }

    public void setSelectionListener (ISelectionListener listener) {
        selectionListener = listener;
    }


    public List<PointF> getPointsInsideBounds() {
        List<PointF> points = new ArrayList<PointF>();
        points.add(new PointF(pointer1.getX() + pointerRadius - minX, pointer1.getY() + pointerRadius - minY));
        points.add(new PointF(pointer2.getX() + pointerRadius - minX, pointer2.getY() + pointerRadius - minY));
        points.add(new PointF(pointer3.getX() + pointerRadius - minX, pointer3.getY() + pointerRadius - minY));
        points.add(new PointF(pointer4.getX() + pointerRadius - minX, pointer4.getY() + pointerRadius - minY));

        return points;
    }

    public Map<Integer, PointF> getPoints() {

        List<PointF> points = new ArrayList<PointF>();
        points.add(new PointF(pointer1.getX(), pointer1.getY()));
        points.add(new PointF(pointer2.getX(), pointer2.getY()));
        points.add(new PointF(pointer3.getX(), pointer3.getY()));
        points.add(new PointF(pointer4.getX(), pointer4.getY()));

        return getOrderedPoints(points);
    }

    public Map<Integer, PointF> getOrderedPoints(List<PointF> points) {
        PointF centerPoint = new PointF();
        int size = points.size();

        for (PointF pointF: points) {
            centerPoint.x += pointF.x / size;
            centerPoint.y += pointF.y / size;
        }

        PointF[] orderedPoints = points.toArray(new PointF[size]);
        UserInterfaceUtils.sortPointsClockwise(orderedPoints, centerPoint);

        double closestDistanceToTopLeft = Double.MAX_VALUE;
        int closestIndexToTopLeft = -1;
        for (int i = 0; i < orderedPoints.length; i++) {
            PointF pointF = orderedPoints[i];
            double distance = Math.sqrt(Math.pow(pointF.x, 2) +  Math.pow(pointF.y, 2));
            if (distance < closestDistanceToTopLeft) {
                closestDistanceToTopLeft = distance;
                closestIndexToTopLeft = i;
            }
            centerPoint.x += pointF.x / size;
            centerPoint.y += pointF.y / size;
        }

        Map<Integer, PointF> orderedPointsMap = new HashMap<>();
        orderedPointsMap.put(0, orderedPoints[closestIndexToTopLeft]); //top left
        orderedPointsMap.put(1, orderedPoints[(closestIndexToTopLeft + 1) % 4]); //top right
        orderedPointsMap.put(2, orderedPoints[(closestIndexToTopLeft + 3) % 4]); //bottom left
        orderedPointsMap.put(3, orderedPoints[(closestIndexToTopLeft + 2) % 4]); //bottom right

        return orderedPointsMap;
    }


    public boolean isConvex(Map<Integer, PointF> points)
    {
        PointF[] vertices = new PointF[points.size()];

        if (vertices.length < 4)
            return false;

        vertices[0] = points.get(0);
        vertices[1] = points.get(1);
        vertices[2] = points.get(3);
        vertices[3] = points.get(2);

        boolean sign = false;
        int n = vertices.length;
        for(int i = 0; i < n; i++)
        {
            double dx1 = vertices[(i + 2) % n].x - vertices[(i + 1) % n].x;
            double dy1 = vertices[(i + 2) % n].y - vertices[(i + 1) % n].y;
            double dx2 = vertices[i].x - vertices[(i + 1) % n].x;
            double dy2 = vertices[i].y - vertices[(i + 1) % n].y;
            double zcrossproduct = dx1 * dy2 - dy1 * dx2;
            if (i == 0)
                sign = zcrossproduct > 0;
            else if (sign != (zcrossproduct > 0))
                return false;
        }
        correctThePointsOrder(points);
        return true;
    }

    private void correctThePointsOrder(Map<Integer, PointF> pointFMap){
        if (pointerRadius == 0)
            pointerRadius = pointer1.getWidth() / 2;

        if (pointFMap == null) {
            hidePoints();
            return;
        }

        showPoints();
        if (pointFMap.size() == 4) {
            Object[] keys = pointFMap.keySet().toArray();
            for (Object key: keys) {
                PointF pointF = pointFMap.get(key);
                movePointInsideBoundaries(pointF);
            }
            setPointsCoordinates(pointFMap);
            isPointsAlreadySet = true;
            invalidate();
        }
    }

    public void setPoints(Map<Integer, PointF> pointFMap) {
        if (pointerRadius == 0)
            pointerRadius = pointer1.getWidth() / 2;

        if (pointFMap == null) {
            hidePoints();
            return;
        }

        showPoints();
        if (pointFMap.size() == 4) {
            Object[] keys = pointFMap.keySet().toArray();
            for (Object key: keys) {
                PointF pointF = pointFMap.get(key);
                // move coordinate relative to bounds
                pointF.x += minX;
                pointF.y += minY;
                // locating the center of the circle-image
                pointF.x -= pointerRadius;
                pointF.y -= pointerRadius;
                // verify the coordinate is inside boundaries
                movePointInsideBoundaries(pointF);
            }
            setPointsCoordinates(pointFMap);
            updateSelectionColorAndListener();
            isPointsAlreadySet = true;
            invalidate();
        }
    }

    private void movePointInsideBoundaries(PointF pointF) {
        pointF.x = Math.min(Math.max(pointF.x, minX - pointerRadius), maxX - pointerRadius);
        pointF.y = Math.min(Math.max(pointF.y, minY - pointerRadius), maxY - pointerRadius);
    }

    private void hidePoints() {
        pointer1.setVisibility(View.GONE);
        pointer2.setVisibility(View.GONE);
        pointer3.setVisibility(View.GONE);
        pointer4.setVisibility(View.GONE);

        midPointer12.setVisibility(View.GONE);
        midPointer13.setVisibility(View.GONE);
        midPointer24.setVisibility(View.GONE);
        midPointer34.setVisibility(View.GONE);
    }

    private void showPoints() {
        pointer1.setVisibility(View.VISIBLE);
        pointer2.setVisibility(View.VISIBLE);
        pointer3.setVisibility(View.VISIBLE);
        pointer4.setVisibility(View.VISIBLE);

        midPointer12.setVisibility(View.VISIBLE);
        midPointer13.setVisibility(View.VISIBLE);
        midPointer24.setVisibility(View.VISIBLE);
        midPointer34.setVisibility(View.VISIBLE);
    }

    private void setPointsCoordinates(Map<Integer, PointF> pointFMap) {
        pointer1.setX(pointFMap.get(0).x);
        pointer1.setY(pointFMap.get(0).y);

        pointer2.setX(pointFMap.get(1).x);
        pointer2.setY(pointFMap.get(1).y);

        midPointer12.setX((pointFMap.get(0).x + pointFMap.get(1).x) / 2);
        midPointer12.setY((pointFMap.get(0).y + pointFMap.get(1).y) / 2);

        pointer3.setX(pointFMap.get(2).x);
        pointer3.setY(pointFMap.get(2).y);

        midPointer13.setX((pointFMap.get(0).x + pointFMap.get(2).x) / 2);
        midPointer13.setY((pointFMap.get(0).y + pointFMap.get(2).y) / 2);

        pointer4.setX(pointFMap.get(3).x);
        pointer4.setY(pointFMap.get(3).y);

        midPointer24.setX((pointFMap.get(1).x + pointFMap.get(3).x) / 2);
        midPointer24.setY((pointFMap.get(1).y + pointFMap.get(3).y) / 2);

        midPointer34.setX((pointFMap.get(2).x + pointFMap.get(3).x) / 2);
        midPointer34.setY((pointFMap.get(2).y + pointFMap.get(3).y) / 2);
    }

    private void drawPointerOnMagnifierCanvas(Canvas canvas, PointF pointF) {
        int l = (int) (pointF.x - pointerRadius);
        int t = (int) (pointF.y - pointerRadius);
        int r = (int) (pointF.x + pointerRadius);
        int b = (int) (pointF.y + pointerRadius);

        Drawable d = isValidRect? validPointerDrawable : invalidPointerDrawable;
        d.setBounds(l, t, r, b);
        d.draw(canvas);
    }

    private final float pointerDrawRatio =
            getResources().getDimension(R.dimen.cropping_selection_circle_size) /
            getResources().getDimension(R.dimen.cropping_selection_circle_extra_size);

    private void drawPointerOnMagnifierCanvas(Canvas canvas, PointF pointF, boolean isMoving) {
        drawPointerOnMagnifierCanvas(canvas, pointF);
        if (isMoving)
            canvas.drawCircle(pointF.x, pointF.y, pointerRadius * pointerDrawRatio * 0.8f, movingPointerPaint);

    }

    private PointF getCenterPointFromImage(ImageView circle) {
        float xPosition = circle.getX() + (circle.getWidth() / 2);
        float yPosition = circle.getY() + (circle.getHeight() / 2);

        return new PointF(xPosition, yPosition);

    }

    private PointF getCenterPointFromImage(ImageView circle, float scaling, PointF scalingPivot) {
        float xPosition = scalingPivot.x - (scaling*(scalingPivot.x - (circle.getX() + (circle.getWidth() / 2))));
        float yPosition = scalingPivot.y - (scaling*(scalingPivot.y - (circle.getY() + (circle.getHeight() / 2))));

        return new PointF(xPosition, yPosition);

    }

    protected void drawSelectionOnMagnifierCanvas(Canvas canvas) {
        PointF p1 = getCenterPointFromImage(pointer1, magnifierScale, zoomPosition);
        PointF p2 = getCenterPointFromImage(pointer2, magnifierScale, zoomPosition);
        PointF p3 = getCenterPointFromImage(pointer3, magnifierScale, zoomPosition);
        PointF p4 = getCenterPointFromImage(pointer4, magnifierScale, zoomPosition);

        drawPointerOnMagnifierCanvas(canvas, p1, movingView == pointer1);
        drawPointerOnMagnifierCanvas(canvas, p2, movingView == pointer2);
        drawPointerOnMagnifierCanvas(canvas, p3, movingView == pointer3);
        drawPointerOnMagnifierCanvas(canvas, p4, movingView == pointer4);

        drawPointerOnMagnifierCanvas(canvas, getCenterPointFromImage(midPointer13, magnifierScale, zoomPosition));
        drawPointerOnMagnifierCanvas(canvas, getCenterPointFromImage(midPointer24, magnifierScale, zoomPosition));
        drawPointerOnMagnifierCanvas(canvas, getCenterPointFromImage(midPointer34, magnifierScale, zoomPosition));
        drawPointerOnMagnifierCanvas(canvas, getCenterPointFromImage(midPointer12, magnifierScale, zoomPosition));

        canvas.drawLine(p1.x, p1.y, p3.x, p3.y, rectPaint);
        canvas.drawLine(p1.x, p1.y, p2.x, p2.y, rectPaint);
        canvas.drawLine(p2.x, p2.y, p4.x, p4.y, rectPaint);
        canvas.drawLine(p3.x, p3.y, p4.x, p4.y, rectPaint);
    }

    protected void drawSelectionOnBasicCanvas(Canvas canvas) {
        PointF p1center = getCenterPointFromImage(pointer1);
        PointF p2center = getCenterPointFromImage(pointer2);
        PointF p3center = getCenterPointFromImage(pointer3);
        PointF p4center = getCenterPointFromImage(pointer4);

        canvas.drawLine(p1center.x, p1center.y, p3center.x, p3center.y, rectPaint);
        canvas.drawLine(p1center.x, p1center.y, p2center.x, p2center.y, rectPaint);
        canvas.drawLine(p2center.x, p2center.y, p4center.x, p4center.y, rectPaint);
        canvas.drawLine(p3center.x, p3center.y, p4center.x, p4center.y, rectPaint);

        midPointer13.setX((pointer3.getX() + pointer1.getX()) / 2);
        midPointer13.setY((pointer3.getY() + pointer1.getY()) / 2);
        midPointer24.setX((pointer4.getX() + pointer2.getX()) / 2);
        midPointer24.setY((pointer4.getY() + pointer2.getY()) / 2);
        midPointer34.setX((pointer4.getX() + pointer3.getX()) / 2);
        midPointer34.setY((pointer4.getY() + pointer3.getY()) / 2);
        midPointer12.setX((pointer2.getX() + pointer1.getX()) / 2);
        midPointer12.setY((pointer2.getY() + pointer1.getY()) / 2);
    }

    private boolean magnifierLocatedLeft = true;
    private boolean magnifierLocatedTop = true;

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);

        // drawing selection
        canvas.translate(0, 0);
        drawSelectionOnBasicCanvas(canvas);

        if (!isZooming || magnifierBitmap == null || magnifierShader == null)
            return;

        magnifierMatrix.reset();
        magnifierMatrix.postScale(magnifierScale, magnifierScale, zoomPosition.x, zoomPosition.y);
        magnifierPaint.getShader().setLocalMatrix(magnifierMatrix);

        // drawing magnifier in the corner

        // the magnifier will be in the **same** left/right position as the moving point
        float w = getWidth();
        if (magnifierLocatedLeft && zoomPosition.x > w*(2f/3f))
            magnifierLocatedLeft = false;
        else if (!magnifierLocatedLeft && zoomPosition.x < w*(1.0f/3f))
            magnifierLocatedLeft = true;
        float translateX = magnifierLocatedLeft ? (-zoomPosition.x + magnifierRadius) : (w - zoomPosition.x - magnifierRadius);

        // the magnifier will be in the **opposite** top/bottom position as the moving point
        float h = getHeight();
        if (magnifierLocatedTop && zoomPosition.y < h*(1.0f/3f))
            magnifierLocatedTop = false;
        else if (!magnifierLocatedTop && zoomPosition.y > h*(2f/3f))
            magnifierLocatedTop = true;
        float translateY = magnifierLocatedTop ? (-zoomPosition.y + magnifierRadius) : (h - zoomPosition.y - magnifierRadius);

        canvas.translate(translateX, translateY);
        
        canvas.drawCircle(zoomPosition.x, zoomPosition.y, magnifierRadius, magnifierPaint);
        magnifierPath.reset();
        magnifierPath.addCircle(zoomPosition.x, zoomPosition.y, magnifierRadius, Path.Direction.CCW);
        canvas.clipPath(magnifierPath, Region.Op.INTERSECT);
        drawSelectionOnMagnifierCanvas(canvas);

        int l = (int) (zoomPosition.x - magnifierRadius);
        int t = (int) (zoomPosition.y - magnifierRadius);
        int r = (int) (zoomPosition.x + magnifierRadius);
        int b = (int) (zoomPosition.y + magnifierRadius);

        magnifierCircleDrawable.setBounds(l, t, r, b);
        magnifierCircleDrawable.draw(canvas);
    }

    private ImageView getImageView(int x, int y) {
        ImageView imageView = new ImageView(context);
        LayoutParams layoutParams = new LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        imageView.setLayoutParams(layoutParams);
        imageView.setImageResource(R.drawable.cropping_pointer);
        imageView.setX(x - 16);
        imageView.setY(y - 16);
        imageView.setOnTouchListener(new TouchListenerImpl());
        return imageView;
    }

    public void setBoundaries() {
        minX = 0;
        minY = 0;
        maxX = getWidth();
        maxY = getHeight();
    }

    public void setBoundaries(RectF rectF) {
        minX = rectF.left;
        minY = rectF.top;
        maxX = rectF.right;
        maxY = rectF.bottom;
    }

    public RectF getBoundaries() {
        return new RectF(minX, minY, maxX, maxY);
    }

    @Override
    public boolean onTouch(View v, MotionEvent event) {
        return false;
    }

    public void setBitmap(Bitmap bitmap) {
        if (bitmap == null)
            return;

        if (magnifierBitmap != null) {
            if (!magnifierBitmap.isRecycled())
                magnifierBitmap.recycle();
            magnifierBitmap = null;
        }

        magnifierBitmap = bitmap;
        magnifierShader = new BitmapShader(magnifierBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
        magnifierPaint.setShader(magnifierShader);
        magnifierPaint.setAntiAlias(true);
    }

    public boolean isPointsAlreadySet() {
        return isPointsAlreadySet;
    }

    private class MidPointTouchListenerImpl implements OnTouchListener {

        PointF DownPT = new PointF(); // Record Mouse Position When Pressed Down
        PointF StartPT = new PointF(); // Record Start Position of 'img'

        private ImageView mainPointer1;
        private ImageView mainPointer2;

        public MidPointTouchListenerImpl(ImageView mainPointer1, ImageView mainPointer2) {
            this.mainPointer1 = mainPointer1;
            this.mainPointer2 = mainPointer2;
        }

        @Override
        public boolean onTouch(View v, MotionEvent event) {
            int eid = event.getAction();

            switch (eid) {
                case MotionEvent.ACTION_MOVE:

                    PointF mv = new PointF(event.getX() - DownPT.x, event.getY() - DownPT.y);

                    mainPointer2.setY((int) (mainPointer2.getY() + mv.y));
                    mainPointer1.setY((int) (mainPointer1.getY() + mv.y));
                    v.setY((mainPointer2.getY() + mainPointer1.getY())/2);

                    mainPointer2.setX((int) (mainPointer2.getX() + mv.x));
                    mainPointer1.setX((int) (mainPointer1.getX() + mv.x));
                    v.setX((mainPointer2.getX() + mainPointer1.getX())/2);
                    break;
                case MotionEvent.ACTION_DOWN:
                    DownPT.x = event.getX();
                    DownPT.y = event.getY();

                    StartPT = new PointF(v.getX(), v.getY());
                    if (selectionListener != null)
                        selectionListener.onSelectionStarted();
                    break;
                case MotionEvent.ACTION_UP:
                    PointF location = new PointF(mainPointer1.getX()*1.0f, mainPointer1.getY()*1.0f);

                    movePointInsideBoundaries(location);
                    mainPointer1.setX(location.x);
                    mainPointer1.setY(location.y);

                    location = new PointF(mainPointer2.getX()*1.0f, mainPointer2.getY()*1.0f);

                    movePointInsideBoundaries(location);
                    mainPointer2.setX(location.x);
                    mainPointer2.setY(location.y);

                    isZooming = false;
                    invalidate();
                    updateSelectionColorAndListener();

                    break;
                default:
                    break;
            }
            isZooming = false;

            selectionCroppingView.invalidate();
            return true;
        }
    }

    private boolean isValidShape(Map<Integer, PointF> pointFMap) {
        return isConvex(pointFMap);
    }

    public boolean isValid() {
        return isValidRect;
    }


    private class TouchListenerImpl implements OnTouchListener {

        PointF DownPT = new PointF(); // Record Mouse Position When Pressed Down
        PointF StartPT = new PointF(); // Record Start Position of 'img'

        @Override
        public boolean onTouch(View v, MotionEvent event) {

            int eid = event.getAction() & MotionEvent.ACTION_MASK;
            switch (eid) {
                case MotionEvent.ACTION_MOVE:
                    PointF mv = new PointF(event.getX() - DownPT.x, event.getY() - DownPT.y);
                    v.setX((int) (StartPT.x + mv.x));
                    v.setY((int) (StartPT.y + mv.y));
                    StartPT = new PointF(v.getX(), v.getY());
                    isZooming = true;
                    break;
                case MotionEvent.ACTION_DOWN:
                    DownPT.x = event.getX();
                    DownPT.y = event.getY();
                    StartPT = new PointF(v.getX(), v.getY());
                    if (selectionListener != null)
                        selectionListener.onSelectionStarted();
                    isZooming = true;
                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    PointF location = new PointF(v.getX()*1.0f, v.getY()*1.0f);

                    movePointInsideBoundaries(location);

                    v.setX(location.x);
                    v.setY(location.y);

                    isZooming = false;
                    invalidate();
                    updateSelectionColorAndListener();
                    break;
                default:
                    isZooming = false;
                    break;
            }

            if (isZooming) {
                movingView = v;
                zoomPosition.x = v.getX() + pointerRadius;
                zoomPosition.y = v.getY() + pointerRadius;
            }

            selectionCroppingView.invalidate();
            return true;
        }

    }

    private void updateSelectionColorAndListener() {
        isValidRect = isValidShape(getPoints());
        rectPaint.setColor(getResources().getColor(isValidRect? R.color.croppingValidColor: R.color.croppingInvalidColor));
        pointer1.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        pointer2.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        pointer3.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        pointer4.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        midPointer12.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        midPointer34.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        midPointer24.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);
        midPointer13.setImageResource(isValidRect? R.drawable.cropping_pointer : R.drawable.cropping_pointer_invalid);

        if (selectionListener != null)
            selectionListener.onSelectionChanged();
    }


}
