package com.topimagesystems.credit;

/* CardScanner.java
 * See the file "LICENSE.md" for the full license governing this code.
 */

import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.graphics.ImageFormat;
import android.graphics.Rect;
import android.graphics.YuvImage;
import android.hardware.Camera;
import android.hardware.Camera.Parameters;
import android.os.Bundle;
import android.os.Message;

import com.topimagesystems.Constants;
import com.topimagesystems.controllers.imageanalyze.CameraController;
import com.topimagesystems.controllers.imageanalyze.CameraManagerController;
import com.topimagesystems.controllers.imageanalyze.CameraTypes;
import com.topimagesystems.data.SessionResultParams;
import com.topimagesystems.intent.CaptureIntent;
import com.topimagesystems.util.FileUtils;
import com.topimagesystems.util.ImageUtils;
import com.topimagesystems.util.Logger;

import org.opencv.android.Utils;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

public class CardScanner implements Camera.PreviewCallback, Camera.AutoFocusCallback {

    public static final String EXTRA_LANGUAGE_OR_LOCALE = "com.topimagesystems.credit.languageOrLocale";

    private static final String TAG = CardScanner.class.getSimpleName();

    private static final float MIN_FOCUS_SCORE = 4;

//    private static final int DEFAULT_UNBLUR_DIGITS = -1; // no blur per default
    private static final int DEFAULT_UNBLUR_DIGITS = 4; // last 4 digits

    static final int ORIENTATION_PORTRAIT = 1;

    // these values MUST match those in dmz_constants.h
    static final int CREDIT_CARD_TARGET_WIDTH = 428; // kCreditCardTargetWidth
    static final int CREDIT_CARD_TARGET_HEIGHT = 270; // kCreditCardTargetHeight

    // NATIVE
    public static native boolean nUseNeon();

    public static native boolean nUseTegra();

    public static native boolean nUseX86();

    private native void nSetup(boolean shouldDetectOnly, float minFocusScore);

    private native void nSetup(boolean shouldDetectOnly, float minFocusScore, int unBlur);

    private native void nResetAnalytics();

    private native void nGetGuideFrame(int orientation, int previewWidth, int previewHeight, Rect r);

    private native void nScanFrame(byte[] data, int frameWidth, int frameHeight, int orientation,
                                   DetectionInfo dinfo, Bitmap resultBitmap, boolean scanExpiry);

    private native int nGetNumFramesScanned();

    private native void nCleanup();

    private Bitmap detectedBitmap;

    private static boolean manualFallbackForError;

    // member data
    private int mUnblurDigits = DEFAULT_UNBLUR_DIGITS;

    private int mFrameOrientation = ORIENTATION_PORTRAIT;

    private long mAutoFocusStartedAt;
    private long mAutoFocusCompletedAt;

    private byte[] mPreviewBuffer;

    // accessed by test harness subclass.
    protected boolean useCamera = true;
    private static boolean initialized = false;

    // ------------------------------------------------------------------------
    // STATIC INITIALIZATION
    // ------------------------------------------------------------------------

    public static void loadingLibs() throws Exception {
//        Logger.i(Util.PUBLIC_LOG_TAG, "card " + BuildConfig.PRODUCT_VERSION + " " + BuildConfig.BUILD_TIME);

        if (initialized)
            return;
        String errMsg = "";

        try {
            loadLibrary("cardioDecider");
            Logger.d(Util.PUBLIC_LOG_TAG, "Loaded card decider library.");
            Logger.d(Util.PUBLIC_LOG_TAG, "    nUseNeon(): " + nUseNeon());
            Logger.d(Util.PUBLIC_LOG_TAG, "    nUseTegra():" + nUseTegra());
            Logger.d(Util.PUBLIC_LOG_TAG, "    nUseX86():  " + nUseX86());

            if (nUseNeon()) {
                loadLibrary("cardioRecognizer");
                Logger.i(Util.PUBLIC_LOG_TAG, "Loaded card.io NEON library");
            } else if (nUseX86()) {
                loadLibrary("cardioRecognizer");
                Logger.i(Util.PUBLIC_LOG_TAG, "Loaded card.io x86 library");
            } else if (nUseTegra()) {
                loadLibrary("cardioRecognizer_tegra2");
                Logger.i(Util.PUBLIC_LOG_TAG, "Loaded card.io Tegra2 library");
            } else {
                Logger.w(Util.PUBLIC_LOG_TAG,
                        "unsupported processor - card-io scanning requires ARMv7 or x86 architecture");
                manualFallbackForError = true;
                throw new Exception("unsupported processor - card-io scanning requires ARMv7 or x86 architecture");
            }
        } catch (UnsatisfiedLinkError e) {
            String error = "Failed to load native library: " + e.getMessage();
            Logger.e(Util.PUBLIC_LOG_TAG, error);
            errMsg = e.getMessage();
            manualFallbackForError = true;
        } catch (Error e) {
            String error = "Failed to load native library: " + e.getMessage();
            Logger.e(Util.PUBLIC_LOG_TAG, error);
            errMsg = e.getMessage();
            manualFallbackForError = true;
        } catch (Exception e) {
            String error = "Failed to load native library: " + e.getMessage();
            Logger.e(Util.PUBLIC_LOG_TAG, error);
            errMsg = e.getMessage();
            manualFallbackForError = true;
        }
        synchronized (CardScanner.class) {
            initialized = !manualFallbackForError;
        }
        if (manualFallbackForError) {
            throw new Exception(errMsg);
        }
    }

    /**
     * Custom loadLibrary method that first tries to load the libraries from the built-in libs
     * directory and if it fails, tries to use the alternative libs path if one is set.
     *
     * No checks are performed to ensure that the native libraries match the cardIO library version.
     * This needs to be handled by the consuming application.
     */
    private static void loadLibrary(String libraryName) throws UnsatisfiedLinkError {
        System.loadLibrary(libraryName);
    }

    private static boolean usesSupportedProcessorArch() {
        return nUseNeon() || nUseTegra() || nUseX86();
    }

    static boolean processorSupported() {
        return (!manualFallbackForError && (usesSupportedProcessorArch()));
    }

    public CardScanner(int currentFrameOrientation) {
        if (!initialized) {
            try {
                loadingLibs();
            } catch (Exception e) {
                return;
            }
        }
        mUnblurDigits = DEFAULT_UNBLUR_DIGITS;
        mFrameOrientation = currentFrameOrientation;
    }

    public void release() {

        if (mPreviewBuffer != null)
            mPreviewBuffer = null;

    }


    public void pauseScanning(Camera camera) {
        camera.addCallbackBuffer(null);
        camera.stopPreview();
        mPreviewBuffer = null;
    }

    public void endScanning(Camera camera) {
        pauseScanning(camera);
//            camera.release();
        nCleanup();

        mPreviewBuffer = null;
    }

    /**
     * Handles processing of each frame.
     * <p/>
     * This method is called by Android, never directly by application code.
     */
    private static boolean processingInProgress = false;

    public void resumeSession(int bufferSize, Camera camera){
        nSetup(false, MIN_FOCUS_SCORE, mUnblurDigits);
        mPreviewBuffer = new byte[bufferSize];
        camera.addCallbackBuffer(mPreviewBuffer);
    }

    private int validFrameCount = 0;

    @Override
    public void onPreviewFrame(byte[] data, Camera camera) {

        if (data == null) {
            Logger.w(TAG, "frame is null! skipping");
            return;
        }

        Parameters p = null;
        try {
            p = camera.getParameters();
        } catch (Exception e) {
            Logger.w(TAG, "camera is released! skipping");
            camera.addCallbackBuffer(null);
            return;
        }
        int format = p.getPreviewFormat();
        int pWidth = p.getPreviewSize().width;
        int pHeight = p.getPreviewSize().height;

        if (processingInProgress) {
            Logger.e(TAG, "processing in progress.... dropping frame");
            // return frame buffer to pool
            if (camera != null) {
                camera.addCallbackBuffer(data);
            }
            return;
        }
        if (detectedBitmap == null) {
            detectedBitmap = Bitmap.createBitmap(CREDIT_CARD_TARGET_WIDTH,
                    CREDIT_CARD_TARGET_HEIGHT, Bitmap.Config.ARGB_8888);
        }

        processingInProgress = true;
        DetectionInfo dInfo = new DetectionInfo();
        nScanFrame(data, pWidth, pHeight, mFrameOrientation, dInfo, detectedBitmap, validFrameCount < 10);
//        nScanFrame(data, pWidth, pHeight, mFrameOrientation, dInfo, detectedBitmap, true);

        boolean sufficientFocus = (dInfo.focusScore >= MIN_FOCUS_SCORE);

        if (!sufficientFocus) {
            triggerAutoFocus(camera);
            validFrameCount = 0;
        } else if (dInfo.predicted()) {
            Logger.d(TAG, "detected card: " + dInfo.creditCard());
            saveVideoMat(data, camera);
            ByteArrayOutputStream scaledCardBytes = new ByteArrayOutputStream();
            detectedBitmap.compress(Bitmap.CompressFormat.JPEG, 100, scaledCardBytes);
            SessionResultParams.colorFront = scaledCardBytes.toByteArray();
            sendResults(dInfo.creditCard());
            if (camera != null) {
                mPreviewBuffer = null;
                camera = null;
            }
        }
        else if (dInfo.detected())
            validFrameCount++;
        else
            validFrameCount = 0;

        // give the image buffer back to the camera, AFTER we're done reading
        // the image.
        if (camera != null) {
            camera.addCallbackBuffer(data);
        }
        processingInProgress = false;

    }

    private synchronized void sendResults(CreditCard creditCard) {
        CameraController cameraController = CameraController.getInstance();
        if (cameraController == null || cameraController.getHandler() == null)
            return;

        cameraController.getHandler().removeAllMessages();
        Bundle messageBundle = new Bundle();
        Message message = Message.obtain(cameraController.getHandler(), CameraTypes.MESSAGE_CREDIT_CARD_RESULT);
        if (message != null) {
            message.obj = true;
            messageBundle.putDouble(Constants.INTENT_ORIENTATION, mFrameOrientation);
            messageBundle.putBoolean(Constants.INTENT_PROCEED_WITH_PROCESSING, false);
            messageBundle.putParcelable(Constants.CREDIT_CARD_DATA, creditCard);
            message.setData(messageBundle);
            message.sendToTarget();
        }

    }

    // used by JNI layer
    void onEdgeUpdate(DetectionInfo dInfo) {
        CameraController cameraController = CameraController.getInstance();

        if (cameraController == null || cameraController.cameraOverlayView ==  null)
            return;

        if (dInfo.detected())
            CameraController.getInstance().cameraOverlayView.setConfirmationIndicators();
        else {
            CameraController.getInstance().cameraOverlayView.setNonConfirmationIndicators();
        }

    }

    private void saveVideoMat(byte[] data, Camera camera) {
        CameraController cameraController = CameraController.getInstance();

        if (cameraController == null || CameraManagerController.getOcrAnalyzeSession(cameraController) == null)
            return;

        Camera.Size size = camera.getParameters().getPreviewSize();

        int width = size.width;
        int height = size.height;

        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            YuvImage yuvImage = new YuvImage(data, ImageFormat.NV21, width, height, null);
            yuvImage.compressToJpeg(new Rect(0, 0, width, height), 100, baos);
            byte[] output = baos.toByteArray();
            Bitmap outputBmp = ImageUtils.decodeByteArray(output);
            Mat mat = new Mat();
            Utils.bitmapToMat(outputBmp, mat);
            Mat matRgba = new Mat();
            Imgproc.cvtColor(mat, matRgba, Imgproc.COLOR_BGR2RGBA, 0);
            if (CameraManagerController.deviceName.equals("LGE Nexus 5X")) {
                matRgba = FileUtils.rotateMat(matRgba, 90);
                matRgba = FileUtils.rotateMat(matRgba, 90);
            }
            if (CameraManagerController.sessionType == CaptureIntent.SessionType.PORTRAIT)
                matRgba = FileUtils.rotateMat(matRgba, 90);
            CameraManagerController.getOcrAnalyzeSession(cameraController).setVideoMat(matRgba);
            baos.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public Rect getGuideFrame(int orientation, int previewWidth, int previewHeight) {

        final int ORIENTATION_PORTRAIT = 1;
        final int ORIENTATION_PORTRAIT_UPSIDE_DOWN = 2;
        final int ORIENTATION_LANDSCAPE_RIGHT = 3;
        final int ORIENTATION_LANDSCAPE_LEFT = 4;

        switch (orientation) {
            case ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE:
                setDeviceOrientation(ORIENTATION_LANDSCAPE_RIGHT);
                // switching between width/height
                int tmp = previewWidth;
                previewWidth = previewHeight;
                previewHeight = tmp;
                break;
            case ActivityInfo.SCREEN_ORIENTATION_PORTRAIT:
                setDeviceOrientation(orientation);
                break;
        }
        orientation = getDeviceOrientation();

        Rect r = null;
        if (processorSupported()) {
            r = new Rect();
            nGetGuideFrame(orientation, previewWidth, previewHeight, r);
        }

        if (orientation == ORIENTATION_LANDSCAPE_RIGHT || orientation == ORIENTATION_LANDSCAPE_LEFT)
            if (r != null) {
                r.set(r.top, r.left, r.bottom, r.right);
            }

        return r;
    }

    void setDeviceOrientation(int orientation) {
        mFrameOrientation = orientation;
    }

    int getDeviceOrientation() {
        return mFrameOrientation;
    }

//    Map<String, Object> getAnalytics() {
//        HashMap<String, Object> analytics = new HashMap<String, Object>(11);
//
//        analytics.put("num_frames_scanned", Integer.valueOf(nGetNumFramesScanned()));
//        analytics.put("num_frames_skipped", Integer.valueOf(numFramesSkipped));
//
//        analytics.put("elapsed_time", Double.valueOf((System.currentTimeMillis() - captureStart) / 1000));
//
//        analytics.put("num_manual_refocusings", Integer.valueOf(numManualRefocus));
//        analytics.put("num_auto_triggered_refocusings", Integer.valueOf(numAutoRefocus));
//        analytics.put("num_manual_torch_changes", Integer.valueOf(numManualTorchChange));
//        return analytics;
//    }

    // ------------------------------------------------------------------------
    // CAMERA CONTROL & CALLBACKS
    // ------------------------------------------------------------------------

    /**
     * Invoked when autoFocus is complete
     * <p/>
     * This method is called by Android, never directly by application code.
     */
    @Override
    public void onAutoFocus(boolean success, Camera camera) {
        mAutoFocusCompletedAt = System.currentTimeMillis();
    }

    /**
     * True if autoFocus is in progress
     */
    boolean isAutoFocusing() {
        return mAutoFocusCompletedAt < mAutoFocusStartedAt || (System.currentTimeMillis() - mAutoFocusStartedAt) < 1000;
    }

    // ------------------------------------------------------------------------
    // MISC CAMERA CONTROL
    // ------------------------------------------------------------------------

    /**
     * Tell Preview's camera to trigger autofocus.
     *
     */
    void triggerAutoFocus(Camera camera) {
        if (useCamera && !isAutoFocusing()) {
            try {
                mAutoFocusStartedAt = System.currentTimeMillis();
                camera.autoFocus(this);
            } catch (RuntimeException e) {
                Logger.w(TAG, "could not trigger auto focus: " + e);
            }
        }
    }
}
