package com.topimagesystems.controllers.imageanalyze;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Point;
import android.os.Environment;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import com.topimagesystems.R;
import com.topimagesystems.data.SessionResultParams;
import com.topimagesystems.intent.CaptureIntent;
import com.topimagesystems.util.FileUtils;
import com.topimagesystems.util.ImageUtils;
import com.topimagesystems.util.Logger;
import com.topimagesystems.util.StringUtils;

import org.opencv.core.Mat;
import org.opencv.core.MatOfInt;
import org.opencv.imgcodecs.Imgcodecs;
import org.opencv.imgproc.Imgproc;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;

/**
 * Created by ruthlosser on 12/07/16.
 */
public class RawImagesFlowManager {

    /** No special flow
     */
    public final static int NONE = -1;

    /** Special flow when frames from the camera feed will be saved as image files
     */
    public final static int SAVE_MODE = 0;

    /** Special flow when saved image files will be read as camera frames
     */
    public final static int LOAD_MODE = 1;

    private final static String logTag = "**Automation**";

    private final static boolean debugImages = false;

    private static String originalImagesPath = Environment.getExternalStorageDirectory().getAbsolutePath() + File.separator + "qa_session";

    static RawImagesFlowManager _instance;
    private int processMode = NONE;

    private File originalImagesDir = null;
    private File debugDir = null;
    private CameraController cameraControllerActivity;
    private Context context;

    /*
     function in two ways:
     (1) Counts the saved images
     (2) Calculate to total images in load mode
      */
    private int totalNumberOfImages = 0;

    // load mode class members
    private int loadingCounter = 0;
    private File[] originalSessionImages;
    private TextView imageProcessTextView;
    private TextView imageCounterTextView;
    private ProgressBar progressBarView;

    private boolean waitingForStillImage = false;

    private LoadStateHandlerThread loadStateHandlerThread = new LoadStateHandlerThread();

    private RawImagesFlowManager() {

    }

    public static boolean isActive() {
        return _instance != null;
    }

    static boolean isSaveMode() {
        return _instance != null && _instance.processMode == RawImagesFlowManager.SAVE_MODE;
    }

    static boolean isLoadMode() {
        return _instance != null && _instance.processMode == RawImagesFlowManager.LOAD_MODE;
    }

    public synchronized static void handleLog(String tag, String text) {
        if (isLoadMode() || isSaveMode()) {
            if (text == null)
                return;

            FileUtils.addToLogFile(tag, text, _instance.context, _instance.originalImagesDir.getAbsolutePath() + File.separator + (isLoadMode()? "load_":"save_") + "log.txt");
        }
    }



    public static void handleAlertLog(String title, String message) {
        handleLog("Alert Message", "\nTitle:" + title + "\nMessage:" + message);
    }



    protected static void handleDebugMat(Mat mat) {
        if (_instance != null && debugImages) {
            byte[] b = FileUtils.convertJpgMatToByte(mat);
            FileUtils.storeBitmap(ImageUtils.decodeByteArray(b), _instance.debugDir.getAbsolutePath() + File.separatorChar + "saveStill.jpeg");
        }
    }

    public synchronized static void init(final int processMode, final String directoryPath, final Context context) {
        if (processMode == NONE) {
            if (_instance != null) {
                if (_instance.loadStateHandlerThread != null && _instance.loadStateHandlerThread.isAlive()){
                    _instance.loadStateHandlerThread.quit();
                    _instance.loadStateHandlerThread = null;
                }
                _instance = null;
            }
            return;
        }
        if (_instance == null)
            _instance = new RawImagesFlowManager();
        _instance.context = context;

        handleLog(logTag, "init with process mode:" + ((processMode == SAVE_MODE)? "Save": "Load"));

        _instance.processMode = processMode;
        _instance.totalNumberOfImages = 0;
        _instance.loadingCounter = 0;

        if (processMode == SAVE_MODE || processMode == LOAD_MODE) {
            Thread thread = new Thread() {
                public void run() {
                    _instance.ensureFileSystemDirectories(processMode, (directoryPath == null)? originalImagesPath: directoryPath, context);
                }
            };
            thread.start();
        }
    }

    synchronized private void ensureFileSystemDirectories(int processMode, String directoryPath, Context context) {
        originalImagesDir = new File(directoryPath);
        if (!originalImagesDir.exists())
            originalImagesDir.mkdirs();

        debugDir = new File(Environment.getExternalStorageDirectory().getAbsolutePath()
                + File.separator + FileUtils.tempDebugPath);

        // init save mode (delete previous images)
        if (processMode == SAVE_MODE)
            FileUtils.clearFiles(context, originalImagesDir.getAbsolutePath());
        else if (processMode == LOAD_MODE) {
            // init load mode (delete previous log file)
            File loadLog = new File(originalImagesDir.getAbsolutePath() + File.separator + "load_log.txt");
            if (loadLog.exists()) {
                loadLog.delete();
                Logger.i(logTag, "deleting -> " + originalImagesDir.getAbsolutePath() + File.separator + "load_log.txt");
            }

        }

    }



    /**********************************
     *
     *       Methods for SAVE MODE
     *
     **********************************/
    synchronized void handleSave(Mat originalMat) {
        handleSave(originalMat, false);
    }

    synchronized void handleSave(Mat originalMat, boolean isStillsImage) {
        if (originalImagesDir == null || !originalImagesDir.exists())
            return;

        if (isStillsImage && !CameraManagerController.isStillMode)
            return;

        Mat rgbaMatOut = new Mat();
        if (isStillsImage &&
                !(CameraManagerController.imageType == CaptureIntent.TISDocumentType.FULL_PAGE) &&
                !(CameraManagerController.sessionType == CaptureIntent.SessionType.PORTRAIT)) {
            Imgproc.cvtColor(originalMat, rgbaMatOut, Imgproc.COLOR_BGR2RGBA, 0);
        }
        else {
            rgbaMatOut = originalMat;
        }

        totalNumberOfImages++;

        String filePath = null;
        boolean saveSuccess = false;
        if (isStillsImage) {
            filePath = originalImagesDir.getAbsolutePath() + File.separator +
                    StringUtils.getNumberWithLeadingZeros(totalNumberOfImages, 5) +  "_stills.jpeg";
            saveSuccess = Imgcodecs.imwrite(filePath, rgbaMatOut);
            // for older OpenCV:
//            saveSuccess = Highgui.imwrite(filePath, rgbaMatOut);
        }
        else {
            filePath = originalImagesDir.getAbsolutePath() + File.separator +
                    StringUtils.getNumberWithLeadingZeros(totalNumberOfImages, 5) +  ".png";
            MatOfInt mPngCompressionRate = new MatOfInt(Imgcodecs.CV_IMWRITE_PNG_COMPRESSION,0);
            saveSuccess = Imgcodecs.imwrite(filePath, rgbaMatOut, mPngCompressionRate);
            // for older OpenCV:
//            MatOfInt mPngCompressionRate = new MatOfInt(Highgui.CV_IMWRITE_PNG_COMPRESSION,0);
//            saveSuccess = Highgui.imwrite(filePath, rgbaMatOut, mPngCompressionRate);
        }

        if (!saveSuccess)
            totalNumberOfImages--;
        else
            handleLog(logTag, "image saved in path:" + filePath);
    }

    /**********************************
     *
     *      Methods for LOAD MODE
     *
     **********************************/
    void startLoadImagesFlow(CameraController cameraController) {
        this.cameraControllerActivity = cameraController;

        // check initiation
        if (originalImagesDir == null || !originalImagesDir.exists() || !originalImagesDir.isDirectory()) {
            Toast.makeText(cameraControllerActivity, "Images directory not found", Toast.LENGTH_LONG).show();
            handleLog(logTag, "Images directory not found");
            stopRunning();
            return;
        }

        // get views
        imageProcessTextView = (TextView) cameraControllerActivity.findViewById(R.id.ImageNumberTitle);
        imageCounterTextView = (TextView) cameraControllerActivity.findViewById(R.id.ImageNumberCounter);
        progressBarView = (ProgressBar) cameraControllerActivity.findViewById(R.id.progress_bar);

        // ensure message handler
        if (cameraControllerActivity.getHandler() == null) {
            if (CameraManagerController.isDynamicCapture)
                cameraControllerActivity.handler = new DynamicCaptureCameraController.DynamicCameraActivityHandler(DynamicCaptureCameraController.getInstance(), true);
            else
                cameraControllerActivity.handler = new CameraController.CameraActivityHandler(CameraController.getInstance(), true);
            CameraController.getInstance().mobiCHECKOCR.setHandler(cameraControllerActivity.handler);
        }

        // ensure loading thread handler
        if (loadStateHandlerThread == null)
            loadStateHandlerThread = new LoadStateHandlerThread();
        else
            loadStateHandlerThread.cleanQueue();
        loadStateHandlerThread.start();

        // get still and video files
        File[] files = originalImagesDir.listFiles();
        sortFileArrayByName(files);
        ArrayList<File> imageFiles = new ArrayList<File>();
        for (File file: files) {
            String fileName = file.getName();
            if (!fileName.endsWith(".png") && !fileName.endsWith(".jpeg"))
                continue;
            imageFiles.add(file);
        }

        originalSessionImages = imageFiles.toArray(new File[imageFiles.size()]);

        sortFileArrayByName(originalSessionImages);

        // update ui
        updateProgressView();

        handleLog(logTag, "starting load images flow with " + (originalSessionImages.length) + " images");
    }

    private boolean isStillImage(File file) {
        return file.getName().endsWith("_stills.jpeg");

    }


    private void sortFileArrayByName(File[] files) {
        if (files != null && files.length > 1) {
            Arrays.sort(files, new Comparator<File>() {
                @Override
                public int compare(File lhs, File rhs) {
                    return lhs.getName().compareToIgnoreCase(rhs.getName());
                }
            });
        }
    }

    boolean hasMoreVideoFrames() {
        return originalSessionImages != null && loadingCounter < originalSessionImages.length;
    }


    synchronized void loadNextImage() {
        if (originalSessionImages != null &&
                loadingCounter < originalSessionImages.length) {

            if (isStillImage(originalSessionImages[loadingCounter])) {
                if (waitingForStillImage)
                    loadNextStillImage();
                else {

                    handleLog(logTag, "No more video images to load before the next Still image.... finishing..");
                    stopRunning();
                    return;
                }
            }
            else {
                loadStateHandlerThread.processVideoImage(originalSessionImages[loadingCounter]);
            }


        } else if (loadingCounter >= originalSessionImages.length && CameraController.getInstance().cameraSessionManager.getState() != CameraSessionManager.State.CAPTURING_IMAGE) {

            handleLog(logTag, "No more images to load.... finishing..");
            stopRunning();
        }
    }

    private void stopRunning() {
        if (loadStateHandlerThread != null)
            loadStateHandlerThread.quit();
        if (cameraControllerActivity != null && !cameraControllerActivity.isFinishing()) {
            cameraControllerActivity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    cameraControllerActivity.setResult(Activity.RESULT_CANCELED);
                    cameraControllerActivity.finish();
                }
            });
        }

    }



    public class LoadStateHandlerThread extends HandlerThread {

        private Handler mHandler;

        public LoadStateHandlerThread() {
            super("LoadStateHandlerThread", android.os.Process.THREAD_PRIORITY_BACKGROUND);
        }

        @Override
        protected void onLooperPrepared() {
            super.onLooperPrepared();
            mHandler = new Handler(getLooper()) {
                @Override
                public void handleMessage(Message msg) {
                    if (CameraController.getInstance() == null || CameraController.getInstance().getHandler() == null) {
                        cleanQueue();
                        return;
                    }

                    File file;
                    String filePath;
                    switch(msg.what) {
                        case 1:
                            if (CameraSessionManager.getInstance().previewCallback == null) {
                                cleanQueue();
                                return;
                            }
                            file = (File) msg.obj;
                            filePath = file.getAbsolutePath();

                            handleLog(logTag, "loading image file:" + filePath);

                            Bitmap bmp = BitmapFactory.decodeFile(filePath);

                            Mat bgrMat = Imgcodecs.imread(filePath);
                            Mat rgbaMatOut = new Mat();

                            // For older Open CV
//                            Mat bgrMat = Highgui.imread(filePath);
//                            Mat rgbaMatOut = new Mat();

                            Imgproc.cvtColor(bgrMat, rgbaMatOut, Imgproc.COLOR_BGR2RGBA, 0);

                            if (debugImages) {
                                Long tsLong = System.currentTimeMillis() / 1000;
                                Imgcodecs.imwrite(debugDir + File.separator + tsLong.toString() + file.getName(), bgrMat);
                                // For older Open CV
//                                Highgui.imwrite(debugDir + File.separator + tsLong.toString() + file.getName(), bgrMat);
                                handleLog(logTag, "loading image saved for debug in:" + debugDir + File.separator + tsLong.toString() + file.getName());
                            }

                            synchronized (_instance) {
                                loadingCounter++;
                            }
                            CameraSessionManager.getInstance().previewCallback.processFrame(null, null, bgrMat.width(), bgrMat.height(), rgbaMatOut);

                            updateProgressView();
                            break;
                        case 2:
                            // get and store file data
                            file = (File) msg.obj;
                            filePath = file.getAbsolutePath();

                            byte[] data = null;
                            FileInputStream fis = null;
                            boolean errorOccur = false;
                            try {
                                fis = new FileInputStream(filePath);
                                data = new byte[fis.available()];
                                while (fis.read(data) > -1) {
                                }
                                fis.close();
                            } catch (FileNotFoundException e) {
                                errorOccur = true;
                                handleLog(logTag, "File not found: " + e.getMessage());
                            } catch (IOException e) {
                                handleLog(logTag, "Error accessing file: " + e.getMessage());
                                errorOccur = true;
                            } finally {
                                if (fis != null) {
                                    try {
                                        fis.close();
                                    } catch (IOException e) {
                                        Logger.e(logTag, "Error closing a stream: " + e.getMessage());
                                        errorOccur = true;
                                    }
                                }

                            }

                            if (errorOccur) {
                                stopRunning();
                                return;
                            }
                            boolean isFront = CameraManagerController.getOcrAnalyzeSession(cameraControllerActivity).captureMode == CameraTypes.CaptureMode.FRONT;
                            if (isFront)
                                SessionResultParams.originalFront = data;
                            else
                                SessionResultParams.originalBack = data;

                            handleLog(logTag, "loading still image:" + filePath);


                            // get resolution
                            Point cameraResolution = CameraConfigurationManager.cameraCaptureResolution;

                            // send message
                            Message message = CameraController.getInstance().getHandler().obtainMessage(CameraTypes.MESSAGE_PROCESS_CAPTURED_IMAGE);
                            message.obj = PreviewCallback.getImagePath(isFront);//filePath;
                            message.arg1 = cameraResolution.x;
                            message.arg2 = cameraResolution.y;
                            message.sendToTarget();

                            // update progress
                            synchronized (_instance){
                                loadingCounter++;
                            }
                            updateProgressView();
                            break;
                    }
                }
            };
        }

        protected void processVideoImage(File imageFile) {
            Message msg = Message.obtain();
            msg.what = 1;
            msg.obj = imageFile;
            mHandler.sendMessage(msg);
        }
        protected void processStillImage(File imageFile) {
            Message msg = Message.obtain();
            msg.what = 2;
            msg.obj = imageFile;
            mHandler.sendMessage(msg);
        }

        protected void cleanQueue() {
            if (mHandler != null)
                mHandler.removeCallbacksAndMessages(null);
        }
    }

    protected synchronized void waitForStillImage() {
        waitingForStillImage = true;
        loadNextImage();
    }

    private synchronized void loadNextStillImage() {
        if (originalSessionImages != null &&
                loadingCounter < originalSessionImages.length  &&
                isStillImage(originalSessionImages[loadingCounter]) &&
                waitingForStillImage &&
                CameraController.getInstance().cameraSessionManager.getState() != CameraSessionManager.State.CAPTURING_IMAGE) {
            waitingForStillImage = false;
            // change session state
            CameraController.getInstance().cameraSessionManager.setState(CameraSessionManager.State.CAPTURING_IMAGE);

            loadStateHandlerThread.processStillImage(originalSessionImages[loadingCounter]);
        }
    }


    private void updateProgressView() {
        if (cameraControllerActivity != null && !cameraControllerActivity.isFinishing())
            cameraControllerActivity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    progressBarView.setMax(originalSessionImages.length);
                    imageProcessTextView.setText("processing image");
                    imageCounterTextView.setText(String.format("%s/%s", loadingCounter, originalSessionImages.length));
                    progressBarView.setProgress(loadingCounter);
                }
            });
    }




}
