/*
 * 
 */
package com.topimagesystems.controllers.imageanalyze;

import android.content.Context;
import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.Point;
import android.hardware.Camera;
import android.os.Bundle;

import com.topimagesystems.Common.OCRType;
import com.topimagesystems.Constants;
import com.topimagesystems.controllers.imageanalyze.BarcodeReader.BarcodeResult;
import com.topimagesystems.controllers.imageanalyze.CameraTypes.CaptureMode;
import com.topimagesystems.controllers.imageanalyze.CameraTypes.OCRAnalyzeErrorCode;
import com.topimagesystems.credit.CardScanner;
import com.topimagesystems.data.SessionResultParams;
import com.topimagesystems.intent.CaptureIntent.SessionType;
import com.topimagesystems.intent.IQASettingsIntent;
import com.topimagesystems.micr.OCRCommon.ErrorCode;
import com.topimagesystems.micr.OCRResult;
import com.topimagesystems.util.Logger;
import com.topimagesystems.util.StringUtils;
import com.topimagesystems.util.UserInterfaceUtils;

import org.opencv.android.Utils;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.Rect;
import org.opencv.imgcodecs.Imgcodecs;
import org.opencv.imgproc.Imgproc;

import java.io.ByteArrayOutputStream;

// TODO: Auto-generated Javadoc
/**
 * The Class OCRAnalyzeSession.
 */
public class OCRAnalyzeSession {

	/** The analyze error code. */
	public OCRAnalyzeErrorCode analyzeErrorCode;

	/** The ocr error code. */
	public ErrorCode ocrErrorCode;

	/** The ocr error message. */
	public String ocrErrorMessage;

	/** The capture mode. */
	public CaptureMode captureMode;

	/** The check rect. */
	public Rect checkRect;

	/** The video bounding box rect. */
	private Rect videoBoundingBoxRect;

	/** The stills bounding box rect. */
	private Rect stillsBoundingBoxRect;

	/** The orientation. */
	public double orientation;

	/** The front image rect. */
	public Rect frontImageRect;

	/** The is front successfull. */
	public boolean isFrontSuccessfull = false;

	/** The is back successfull. */
	public boolean isBackSuccessfull = false;
	private static String TAG = "OCRAnalyzeSession";

	/** The check boundaries rect. */
	private CheckBoundaries checkBoundariesRect;
	//private static CheckBoundaries checkBoundariesRectBckup;

	/** The check boundaries rect disp. */
	private CheckBoundaries checkBoundariesRectDisp;

	/** The ocr error counter. */
	private int ocrErrorCounter;

	/** The max number of retries. */
	public int maxNumberOfRetries;

	/** The front retries. */
	public int frontRetries = -1;

	/** The back retries. */
	public int backRetries = -1;
	// public Mat currentMat;
	/** The current micr type. */
	public OCRType currentMICRType;

	/** The min ratio hw. */
	public float minRatioHW;

	/** The max ratio hw. */
	public float maxRatioHW;

	/** The is send image as is. */
	public boolean isSendImageAsIs = false;

	/** The is manual capture. */
	public boolean isManualCapture = false;

	/** The is torch on. */
	public boolean isTorchOn = false;

	/** The is use custom algorithm on back. */
	public boolean isUseCustomAlgorithmOnBack;

	/** The is binarize bak same as front. */
	public boolean isBinarizeBakSameAsFront;

	/** The output height in inch. */
	public float outputHeightInInch;

	/** The output width in inch. */
	public float outputWidthInInch;

	/** The min ratio hw back. */
	public float minRatioHWBack;

	/** The max ratio hw back. */
	public float maxRatioHWBack;

	/** The txt valid from. */
	public int txtValidFrom; // TODO: These settings should be handled inside
	// the code.
	/** The txt valid to. */
	public int txtValidTo;

	/** The is iqa enabled. */
	public boolean isIQAEnabled;

	/** The is auto capture front. */
	public boolean isAutoCaptureFront;

	/** The is auto capture back. */
	public boolean isAutoCaptureBack;

	/** The is iqa passed front. */
	public boolean isIQAPassedFront;

	/** The is iqa passed back. */
	public boolean isIQAPassedBack;

	/** The font jpeg after binarization path. */
	public String fontJpegAfterBinarizationPath;

	/** The back jpeg after binarization path. */
	public String backJpegAfterBinarizationPath;

	/** The is blured enabled. */
	public boolean isBluredEnabled;

	/** The video mat. */
	private Mat videoMat;



	private Mat croppedFrontStillsMat;



	private Mat croppedBackStillsMat;
	//
	/** The ocr analyze result. */
	private OCRAnalyzeResult ocrAnalyzeResult;
	//
	/** The barcode result. */
	private BarcodeResult barcodeResult;

	// used for debugging purposes
	/** The timestamp. */
	public int timestamp = 0;

	/** The iqa settings. */
	public IQASettingsIntent iqaSettings;

	/** The is binarize back same as front. */
	public boolean isBinarizeBackSameAsFront;

	/** The is barcode session. */
	public boolean isBarcodeSession;

	/** The is credit-card session. */
	private boolean isCreditCardSession;

	/** The check boundaries rect disp. */
	private CheckBoundaries barcodeBoundariesRectDisp;

	private CardScanner cardScanner;

	private float[] croppingCoordinates;
	private float[] videoQuads;

	/**
	 * Instantiates a new OCR analyze session.
	 *
	 * @param context
	 *            the context
	 * @param micrType
	 *            the micr type
	 * @param maxNumberOfRetries
	 *            the max number of retries
	 * @param minimumRatioHeightWidth
	 *            the minimum ratio height width
	 * @param maxRatioHeightWidth
	 *            the max ratio height width
	 * @param isManualCapture
	 *            the is manual capture
	 * @param isUseCustomAlgorithmOnBack
	 *            the is use custom algorithm on back
	 * @param isBinarizeBakSameAsFront
	 *            the is binarize bak same as front
	 * @param outputHeightInInch
	 *            the output height in inch
	 * @param outputWidthInInch
	 *            the output width in inch
	 * @param minRatioHWBack
	 *            the min ratio hw back
	 * @param maxRatioHWBack
	 *            the max ratio hw back
	 * @param txtValidFrom
	 *            the txt valid from
	 * @param txtValidTo
	 *            the txt valid to
	 * @param isIQAEnabled
	 *            the is iqa enabled
	 * @param iqaSettings
	 *            the iqa settings
	 * @param camera
	 *            the camera
	 * @param isBluredEnabled
	 *            the is blured enabled
	 */
	public OCRAnalyzeSession(Context context, OCRType micrType, int maxNumberOfRetries, float minimumRatioHeightWidth, float maxRatioHeightWidth, boolean isManualCapture,
			boolean isUseCustomAlgorithmOnBack, boolean isBinarizeBakSameAsFront, float outputHeightInInch, float outputWidthInInch, float minRatioHWBack, float maxRatioHWBack,
			int txtValidFrom, int txtValidTo, boolean isIQAEnabled, IQASettingsIntent iqaSettings, Camera camera, boolean isBluredEnabled) {
		final Point pointDisp = UserInterfaceUtils.getDisplayDimensions(context);
		Point point = null;
		try {
			Camera _camera;
			//_camera = CameraController.getInstance().getCamera();
			boolean isCameraOpen = false;
			if (camera == null) {
				_camera = Camera.open();
				isCameraOpen = true;
			} else {
				_camera = camera;
			}

			point = UserInterfaceUtils.getCameraPreviewResolution(context, _camera.getParameters());
			//point = CameraConfigurationManager.getCameraPreviewResolution();
			if (isCameraOpen == true)
				_camera.release();
			this.currentMICRType = micrType;
			this.maxNumberOfRetries = maxNumberOfRetries;
			this.minRatioHW = minimumRatioHeightWidth;
			this.maxRatioHW = maxRatioHeightWidth;
			this.captureMode = CaptureMode.FRONT;
			this.checkBoundariesRect = new CheckBoundaries(context, captureMode, isUseCustomAlgorithmOnBack, point.x, point.y, minRatioHW, maxRatioHW, minRatioHWBack,
					maxRatioHWBack);
			//checkBoundariesRectBckup = checkBoundariesRect;
			this.checkBoundariesRectDisp = new CheckBoundaries(context, captureMode, isUseCustomAlgorithmOnBack, pointDisp.x, pointDisp.y, minRatioHW, maxRatioHW, minRatioHWBack,
					maxRatioHWBack);

			// qr code ratio 0.9 1.1
			// other barcodes 0.225f, 0.275f

			if (CameraManagerController.useQRFrameForBarcode)
				this.barcodeBoundariesRectDisp = new CheckBoundaries(context, captureMode, isUseCustomAlgorithmOnBack, pointDisp.x, pointDisp.y, 0.9f, 1.1f, minRatioHWBack,
						maxRatioHWBack, 0.7f);
			else
				this.barcodeBoundariesRectDisp = new CheckBoundaries(context, captureMode, isUseCustomAlgorithmOnBack, pointDisp.x, pointDisp.y, 0.225f, 0.275f, minRatioHWBack,
						maxRatioHWBack, 0.7f);

			this.ocrErrorCounter = 0;
			this.ocrErrorCode = null;
			this.ocrErrorMessage = null;
			this.analyzeErrorCode = OCRAnalyzeErrorCode.NONE;

			this.ocrAnalyzeResult = new OCRAnalyzeResult();
			this.barcodeResult = new BarcodeResult();
			this.isManualCapture = isManualCapture;

			// license id
			this.isUseCustomAlgorithmOnBack = isUseCustomAlgorithmOnBack;
			this.isBinarizeBakSameAsFront = isBinarizeBakSameAsFront;
			this.minRatioHWBack = minRatioHWBack;
			this.maxRatioHWBack = maxRatioHWBack;
			this.outputHeightInInch = outputHeightInInch;
			this.outputWidthInInch = outputWidthInInch;
			this.txtValidFrom = txtValidFrom;
			this.txtValidTo = txtValidTo;
			this.isIQAEnabled = isIQAEnabled;
			this.isBluredEnabled = isBluredEnabled;
			this.iqaSettings = iqaSettings;

			this.isAutoCaptureFront = !isManualCapture;
			this.isAutoCaptureBack = !isManualCapture;
			this.isIQAPassedFront = true;
			this.isIQAPassedBack = true;

		} catch (Exception e) { // couldn't open camera.

			Logger.e("OCRAnalyzeSession", "failed to open Camera, camera could be open");

		}

	}
	public Mat getBackCroppedStillsMat() {
		return croppedBackStillsMat;
	}

	public void setBackCroppedStillsMat(Mat croppedBackStillsMat) {
		this.croppedBackStillsMat = croppedBackStillsMat;
	}

	public Mat getFrontCroppedStillsMat() {
		return croppedFrontStillsMat;
	}

	public void setFrontCroppedStillsMat(Mat croppedStillsMat) {
		this.croppedFrontStillsMat = croppedStillsMat;
	}

	/**
	 * Sets the video mat.
	 *
	 * @param video
	 *            the new video mat
	 */
	public void setVideoMat(Mat video) {
		videoMat = video;
	}

	/**
	 * Gets the video mat.
	 *
	 * @return the video mat
	 */
	public Mat getVideoMat() {
		return videoMat;
	}

	/**
	 * Gets the video bounding box.
	 *
	 * @return the video bounding box
	 */
	public Rect getVideoBoundingBox() {
		return videoBoundingBoxRect;
	}

	/**
	 * Sets the video bounding box.
	 *
	 * @param videoBoundingRect
	 *            the new video bounding box
	 */
	public void setVideoBoundingBox(Rect videoBoundingRect) {
		videoBoundingBoxRect = new Rect(videoBoundingRect.x, videoBoundingRect.y, videoBoundingRect.width, videoBoundingRect.height);
	}

	/**
	 * Gets the stills bounding box.
	 *
	 * @return the stills bounding box
	 */
	public Rect getStillsBoundingBox() {
		return stillsBoundingBoxRect;
	}

	/**
	 * Sets the stills bounding box.
	 *
	 * @param stillsBoundingRect
	 *            the new stills bounding box
	 */
	public void setStillsBoundingBox(Rect stillsBoundingRect) {
		if (stillsBoundingRect == null)
			stillsBoundingBoxRect = null;
		else
			stillsBoundingBoxRect = new Rect(stillsBoundingRect.x, stillsBoundingRect.y, stillsBoundingRect.width, stillsBoundingRect.height);
	}




	/**
	 * Gets the check boundaries.
	 *
	 * @return the check boundaries
	 */
	public CheckBoundaries getCheckBoundaries() {
		if (checkBoundariesRect == null) {
			//	return checkBoundariesRectBckup;
			return null;
		}
		checkBoundariesRect.setCaptureMode(captureMode);
		return checkBoundariesRect;
	}

	/**
	 * Gets the check boundaries disp.
	 *
	 * @return the check boundaries disp
	 */
	public CheckBoundaries getCheckBoundariesDisp() {
		checkBoundariesRect.setCaptureMode(captureMode);
		return checkBoundariesRectDisp;
	}

	/**
	 * Gets the bar-code boundaries disp.
	 *
	 * @return the bar-code boundaries disp
	 */
	public CheckBoundaries getBarcodeBoundariesRectDisp() {
		barcodeBoundariesRectDisp.setCaptureMode(captureMode);
		return barcodeBoundariesRectDisp;
	}

	private Mat lastMatForCropping;

	public void clearCurrentMat() {
		if (lastMatForCropping != null)
			lastMatForCropping.release();
		lastMatForCropping = null;
	}

	public Mat getCurrentMat(Context context, boolean saveForLaterUse, boolean forceNewCalculation) {
		Mat currentMat;
		if (forceNewCalculation)
			clearCurrentMat();

		if (lastMatForCropping != null)
			currentMat = lastMatForCropping;
		else {
			currentMat = getCurrentMat(context);
			if (saveForLaterUse)
				lastMatForCropping = currentMat;
		}

		return currentMat;
	}

	/**
	 * Gets the current mat.
	 *
	 * @param context
	 *            the context
	 * @return the current mat
	 */
    private Mat getCurrentMat(Context context) {
		String imagePath;
		Bitmap bitmap = null;
		Mat currentMat = null;

		boolean isStillCapture = CameraManagerController.isStillMode;

		try {

			imagePath = (captureMode == CaptureMode.FRONT) ? ocrAnalyzeResult.getFrontImagePath() : ocrAnalyzeResult.getBackImagePath();
			if (CameraManagerController.sessionType == SessionType.TEST) {
				currentMat = Imgcodecs.imread(imagePath);
				return currentMat;
			}
			if (!isStillCapture) { // return saved video mat
				return CameraManagerController.getOcrAnalyzeSession(context).getVideoMat();
			}

			// get the right image byte array front or back
			bitmap = BitmapFactory.decodeByteArray(captureMode == CaptureMode.FRONT ? SessionResultParams.originalFront : SessionResultParams.originalBack, 0,
					captureMode == CaptureMode.FRONT ? SessionResultParams.originalFront.length : SessionResultParams.originalBack.length);

			int requiredRotation = CameraSessionManager.getInstance().getRotationForCameraMat();

			if (requiredRotation != 0 && !RawImagesFlowManager.isLoadMode()) {

				Matrix rotationMatrix = new Matrix();
				rotationMatrix.postRotate(requiredRotation);
				bitmap = Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), rotationMatrix, true);

				ByteArrayOutputStream stream = new ByteArrayOutputStream();
				bitmap.compress(Bitmap.CompressFormat.JPEG, 100, stream);
				if (captureMode == CaptureMode.FRONT)
					SessionResultParams.originalFront = stream.toByteArray();
				else
					SessionResultParams.originalBack = stream.toByteArray();
			}

			currentMat = new Mat();

			Mat bitmapMat = new Mat(CameraConfigurationManager.captureResolutionWidth, CameraConfigurationManager.captureResolutionHeight, CvType.CV_8UC1);
			Utils.bitmapToMat(bitmap, bitmapMat);
			Imgproc.cvtColor(bitmapMat, currentMat, Imgproc.COLOR_RGBA2RGB);

		}
		catch(Exception e){
			Logger.d(TAG, e.toString());
			currentMat = null;
		}
		finally {
			if (bitmap != null && !bitmap.isRecycled())
				recycleBitmap(bitmap);
		}

		return currentMat;
	}

	/**
	 * Clear.
	 */
	public void clear() {
		ocrAnalyzeResult = new OCRAnalyzeResult();
		ocrErrorCode = null;
		ocrErrorMessage = null;
		analyzeErrorCode = OCRAnalyzeErrorCode.NONE;
		checkRect = null;
		orientation = 0;
		ocrErrorCounter = 0;
		frontImageRect = null;
		isFrontSuccessfull = false;
		isBackSuccessfull = false;
		barcodeResult = new BarcodeResult();
	}

	/**
	 * Clear error code.
	 */
	public void clearErrorCode() {
//		ocrAnalyzeResult = new OCRAnalyzeResult();
		ocrErrorCode = null;
		ocrErrorMessage = null;
		analyzeErrorCode = OCRAnalyzeErrorCode.NONE;

	}

	/**
	 * Gets the OCR analyze result.
	 *
	 * @return the OCR analyze result
	 */
	public OCRAnalyzeResult getOCRAnalyzeResult() {
		return ocrAnalyzeResult;
	}

	/**
	 * Checks if is error count exeeds maximum.
	 *
	 * @return true, if is error count exeeds maximum
	 */
	public boolean isErrorCountExeedsMaximum() {
		return ocrErrorCounter > maxNumberOfRetries;
	}

	/**
	 * Gets the faliure attemps number front.
	 *
	 * @return the faliure attemps number front
	 */
	public int getFaliureAttempsNumberFront() {
		return frontRetries;

	}

	/**
	 * Gets the faliure attemps number back.
	 *
	 * @return the faliure attemps number back
	 */
	public int getFaliureAttempsNumberBack() {
		return backRetries;

	}

	/**
	 * Inits the ocr error counter.
	 */
	public void initOcrErrorCounter() {
		ocrErrorCounter = 0;
	}

	/**
	 * Gets the ocr error counter.
	 *
	 * @return the ocr error counter
	 */
	public int getOcrErrorCounter() {
		return ocrErrorCounter;
	}

	/**
	 * Adds the to error counter.
	 */
	public void AddToErrorCounter() {
		this.ocrErrorCounter = this.ocrErrorCounter + 1;
	}

	/**
	 * Sets the image path.
	 *
	 * @param imagePath
	 *            the new image path
	 */
	public void setImagePath(String imagePath) {
		if (captureMode == CaptureMode.FRONT) {
			ocrAnalyzeResult.setFrontImagePath(imagePath);
		} else if (captureMode == CaptureMode.BACK) {
			ocrAnalyzeResult.setBackImagePath(imagePath);
		}
	}

	/**
	 * Gets the ocr result.
	 *
	 * @return the ocr result
	 */
	public OCRResult getOcrResult() {
		return ocrAnalyzeResult.getOcrResult();
	}

	/**
	 * Gets the barcode result.
	 *
	 * @return the barcode result
	 */
	public BarcodeResult getBarcodeResult() {
		return barcodeResult;
	}

	/**
	 * Sets the ocr result.
	 *
	 * @param ocrResult
	 *            the new ocr result
	 */
	public void setOcrResult(OCRResult ocrResult) {
		ocrAnalyzeResult.setOcrResult(ocrResult);
	}

	/**
	 * Gets the front image path.
	 *
	 * @return the front image path
	 */
	public String getFrontImagePath() {
		return ocrAnalyzeResult.getFrontImagePath();
	}

	/**
	 * Sets the front image path.
	 *
	 * @param frontImagePath
	 *            the new front image path
	 */
	public void setFrontImagePath(String frontImagePath) {
		ocrAnalyzeResult.setFrontImagePath(frontImagePath);
	}

	/**
	 * Gets the back image path.
	 *
	 * @return the back image path
	 */
	public String getBackImagePath() {
		return ocrAnalyzeResult.getBackImagePath();
	}

	/**
	 * Sets the back image path.
	 *
	 * @param backImagePath
	 *            the new back image path
	 */
	public void setBackImagePath(String backImagePath) {
		ocrAnalyzeResult.setBackImagePath(backImagePath);
	}





	boolean isCreditCardSession() {
		return isCreditCardSession;
	}

	void setCreditCardSession(boolean creditCardSession, int orientation) {
		isCreditCardSession = creditCardSession;
		if (cardScanner == null && isCreditCardSession) {
			cardScanner = new CardScanner(orientation);
		}
	}

	CardScanner getCardScanner() {
		return cardScanner;
	}

	CheckBoundaries getCreditCardRectDisp(int screenWidth, int screenHeight, int cardWidth, int cardHeight, int orientation) {

		if (orientation == ActivityInfo.SCREEN_ORIENTATION_PORTRAIT) {
			int tmp = cardWidth;
			cardWidth = cardHeight;
			cardHeight = tmp;

			tmp = screenWidth;
			screenWidth = screenHeight;
			screenHeight = tmp;
		}

		android.graphics.Rect guideFrame = cardScanner.getGuideFrame(orientation, cardWidth, cardHeight);

		int leftDelta = (screenWidth - cardWidth) / 2;
		int topDelta = (screenHeight - cardHeight) / 2;

		checkBoundariesRectDisp.setValidationRect(new Rect(guideFrame.left + leftDelta, guideFrame.top + topDelta, guideFrame.width(), guideFrame.height()));
		checkBoundariesRectDisp.setCaptureMode(captureMode);
		return checkBoundariesRectDisp;
	}


	/**
	 * Gets the cropping coordinates
	 *
	 * @return float[8] which contains 4 points in the following order: top-left, top-right, bottom-right, bottom-left
     */

	public float[] getCroppingCoordinates() {
		return croppingCoordinates;
	}


	/**
	 * Sets the cropping coordinates
	 *
	 * @param croppingCoordinates float[8] which contains 4 points in the following order: top-left, top-right, bottom-right, bottom-left
     */
	public void setCroppingCoordinates(float[] croppingCoordinates) {
		this.croppingCoordinates = croppingCoordinates;
	}


	/**
	 * Gets the cropping coordinates
	 *
	 * @return float[8] which contains 4 points in the following order: top-left, top-right, bottom-right, bottom-left
	 */

	public float[] getVideoQuads() {
		return videoQuads;
	}


	/**
	 * Sets the cropping coordinates
	 *
	 * @param quads float[8] which contains 4 points in the following order: top-left, top-right, bottom-right, bottom-left
	 */
	public void setVideoQuads(float[] quads) {
		this.videoQuads = quads;
	}

	/**
	 * Sets the cropping coordinates
	 *
	 * @param checkRectBundle
	 */
	public void setVideoQuads(Bundle checkRectBundle) {
		videoQuads = new float[8];
		videoQuads[0] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_TOP_LEFT)[0];
		videoQuads[1] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_TOP_LEFT)[1];
		videoQuads[2] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_TOP_RIGHT)[0];
		videoQuads[3] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_TOP_RIGHT)[1];
		videoQuads[4] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_BOTTOM_RIGHT)[0];
		videoQuads[5] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_BOTTOM_RIGHT)[1];
		videoQuads[6] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_BOTTOM_LEFT)[0];
		videoQuads[7] = checkRectBundle.getFloatArray(Constants.INTENT_CHECK_QUADS_BOTTOM_LEFT)[1];
	}


	/**
	 * Gets the front image tiff path.
	 *
	 * @return the front image tiff path
	 */
	public String getFrontImageTIFFPath() {
		return ocrAnalyzeResult.getFrontImagePathAsTiff();
	}

	public void resetBackupCheckBoudries() {
		//checkBoundariesRectBckup = null;
	}

	/**
	 * Sets the front image tiff path.
	 *
	 * @param frontImagePathAsTiff
	 *            the new front image tiff path
	 */
	public void setFrontImageTIFFPath(String frontImagePathAsTiff) {
		ocrAnalyzeResult.setFrontImagePathAsTiff(frontImagePathAsTiff);
	}

	/**
	 * Gets the back image tiff path.
	 *
	 * @return the back image tiff path
	 */
	public String getBackImageTIFFPath() {
		return ocrAnalyzeResult.getBackImagePathAsTiff();
	}

	/**
	 * Sets the back image tiff path.
	 *
	 * @param backImagePathAsTiff
	 *            the new back image tiff path
	 */
	public void setBackImageTIFFPath(String backImagePathAsTiff) {
		ocrAnalyzeResult.setBackImagePathAsTiff(backImagePathAsTiff);
	}

	/**
	 * Sets the front barcode type
	 *
	 * @param barcodeType
	 *            the bar code type
	 */
	public void setFrontBarcodeType(String barcodeType) {
		barcodeResult.setBarcodeTypeFront(barcodeType);
	}

	/**
	 * Sets the back barcode type
	 *
	 * @param barcodeType
	 *            the bar code type
	 */
	public void setBackBarcodeType(String barcodeType) {
		barcodeResult.setBarcodeTypeBack(barcodeType);
	}

	/**
	 * Sets the front barcode data
	 *
	 * @param barcodeData
	 *            the bar code data
	 */
	public void setFrontBarcodeData(String barcodeData) {
		barcodeResult.setBarcodeDataFront(barcodeData);
	}

	/**
	 * Sets the front barcode data
	 *
	 * @param barcodeData
	 *            the bar code data
	 */
	public void setBackBarcodeData(String barcodeData) {
		barcodeResult.setBarcodeDataBack(barcodeData);
	}

	/**
	 * Recycle bitmap.
	 *
	 * @param bitmap
	 *            the bitmap
	 */
	private void recycleBitmap(Bitmap bitmap) {
		if (bitmap != null) {
			bitmap.recycle();
			bitmap = null;
		}
	}

	/**
	 * Instantiates a new OCR analyze session.
	 *
	 * @param bundle
	 *            the bundle
	 */
	public OCRAnalyzeSession(Bundle bundle) {
		String analyzeErrorCodeName = bundle.getString("analyzeErrorCode");
		if (!StringUtils.isEmptyOrNull(analyzeErrorCodeName)) {
			analyzeErrorCode = OCRAnalyzeErrorCode.valueOf(analyzeErrorCodeName);
		}

		String errorCodeName = bundle.getString("ocrErrorCode");

		if (!StringUtils.isEmptyOrNull(errorCodeName)) {
			ocrErrorCode = ErrorCode.valueOf(errorCodeName);
		}

		ocrErrorMessage = bundle.getString("ocrErrorMessage");
		String captureModeName = bundle.getString("captureMode");

		if (!StringUtils.isEmptyOrNull(captureModeName)) {
			captureMode = CaptureMode.valueOf(captureModeName);
		}

		checkRect = UserInterfaceUtils.bundleToRect(bundle.getBundle("checkRect"));
		orientation = bundle.getDouble("orientation");
		frontImageRect = UserInterfaceUtils.bundleToRect(bundle.getBundle("frontImageRect"));
		isFrontSuccessfull = bundle.getBoolean("isFrontSuccessfull");
		isBackSuccessfull = bundle.getBoolean("isBackSuccessfull");

		ocrErrorCounter = bundle.getInt("ocrErrorCounter");
		maxNumberOfRetries = bundle.getInt("maxNumberOfRetries");
		String micrTypeName = bundle.getString("currentMICRType");
		if (!StringUtils.isEmptyOrNull(micrTypeName)) {
			currentMICRType = OCRType.valueOf(micrTypeName);
		}

		minRatioHW = bundle.getFloat("OCRValidationMinimumRatioHeightWidth");
		maxRatioHW = bundle.getFloat("OCRValidationMaximumRatioHeightWidth");

		txtValidFrom = bundle.getInt("digitalRowScopeFrom");
		txtValidTo = bundle.getInt("digitalRowScopeTo");
		isIQAEnabled = bundle.getBoolean("IQAEnabled");

		isSendImageAsIs = bundle.getBoolean("isSendImageAsIs");
		isManualCapture = bundle.getBoolean("isManualCapture");
		isTorchOn = bundle.getBoolean("isTorchOn");
		Bundle checkBoundariesBundle = bundle.getBundle("checkBoundariesRect");
		if (checkBoundariesBundle != null) {
			checkBoundariesRect = new CheckBoundaries(checkBoundariesBundle);
		}

		Bundle ocrAnalyzeResultBundle = bundle.getBundle("ocrAnalyzeResult");
		if (ocrAnalyzeResultBundle != null) {
			ocrAnalyzeResult = new OCRAnalyzeResult(ocrAnalyzeResultBundle);
		}

		Bundle barcodeResultBundle = bundle.getBundle("barcodeResult");
		if (barcodeResultBundle != null) {
			barcodeResult = new BarcodeResult(ocrAnalyzeResultBundle);
		}
	}

	/**
	 * To bundle.
	 *
	 * @return the bundle
	 */
	public Bundle toBundle() {
		Bundle bundle = new Bundle();
		if (analyzeErrorCode != null)
			bundle.putString("analyzeErrorCode", analyzeErrorCode.name());

		if (ocrErrorCode != null)
			bundle.putString("ocrErrorCode", ocrErrorCode.name());

		bundle.putString("ocrErrorMessage", ocrErrorMessage);

		if (captureMode != null)
			bundle.putString("captureMode", captureMode.name());

		bundle.putBundle("checkRect", UserInterfaceUtils.rectToBundle(checkRect));
		bundle.putDouble("orientation", orientation);
		bundle.putBundle("frontImageRect", UserInterfaceUtils.rectToBundle(frontImageRect));

		bundle.putBoolean("isFrontSuccessfull", isFrontSuccessfull);
		bundle.putBoolean("isBackSuccessfull", isBackSuccessfull);

		bundle.putInt("ocrErrorCounter", ocrErrorCounter);
		bundle.putInt("maxNumberOfRetries", maxNumberOfRetries);
		if (currentMICRType != null)
			bundle.putString("currentMICRType", currentMICRType.name());

		bundle.putFloat("OCRValidationMinimumRatioHeightWidth", minRatioHW);
		bundle.putFloat("OCRValidationMaximumRatioHeightWidth", maxRatioHW);
		bundle.putInt("digitalRowScopeFrom", txtValidFrom);
		bundle.putInt("digitalRowScopeTo", txtValidTo);
		bundle.putBoolean("IQAEnabled", isIQAEnabled);

		bundle.putBoolean("isSendImageAsIs", isSendImageAsIs);
		bundle.putBoolean("isManualCapture", isManualCapture);
		bundle.putBoolean("isTorchOn", isTorchOn);

		if (checkBoundariesRect != null)
			bundle.putBundle("checkBoundariesRect", checkBoundariesRect.toBundle());

		if (ocrAnalyzeResult != null)
			bundle.putBundle("ocrAnalyzeResult", ocrAnalyzeResult.toBundle());

		if (barcodeResult != null)
			bundle.putBundle("barcodeResult", barcodeResult.toBundle());
		return bundle;
	}
}