package com.topimagesystems.controllers.imageanalyze;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.opencv.android.Utils;
import org.opencv.core.Mat;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.MultiFormatReader;
import com.google.zxing.PlanarYUVLuminanceSource;
import com.google.zxing.RGBLuminanceSource;
import com.google.zxing.ReaderException;
import com.google.zxing.Result;
import com.google.zxing.common.HybridBinarizer;
import com.topimagesystems.controllers.imageanalyze.CameraTypes.TISBarcodeType;
import com.topimagesystems.util.Logger;
import com.topimagesystems.util.OcrValidationUtils;

import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Rect;
import android.os.Bundle;
import android.util.Log;

public class BarcodeReader {

	private Rect mFramingRectInPreview;
	private Rect mBoundariesRect;
	private MultiFormatReader mMultiFormatReader;
	public static final List<BarcodeFormat> ALL_FORMATS = new ArrayList<BarcodeFormat>();

	private static List<BarcodeFormat> mFormats;
	private static List<TISBarcodeType> mTISFormats;

	BarcodeReader(ArrayList<TISBarcodeType> barcodeTypes, int x, int y, int width, int height) {
		if (barcodeTypes != null) {
			mTISFormats = new ArrayList<TISBarcodeType>();
			mTISFormats.addAll(barcodeTypes);
			mFormats = translateTISBarcodeToZxingArray(mTISFormats);

			initMultiFormatReader();

			mBoundariesRect = new Rect(x, y, x + width, y + height);
		}
	}

	void setTisBarcodeFormats(List<TISBarcodeType> formats) {
		if (formats != null) {
			mTISFormats = new ArrayList<TISBarcodeType>();
			mTISFormats.addAll(formats);
			mFormats = translateTISBarcodeToZxingArray(mTISFormats);

			initMultiFormatReader();
		}
	}

	void setBarcodeBoundaries(int x, int y, int width, int height){
		mBoundariesRect = new Rect(x, y, x + width, y + height);

	}

	private void initMultiFormatReader() {
		Map<DecodeHintType, Object> hints = new EnumMap<DecodeHintType, Object>(DecodeHintType.class);
		hints.put(DecodeHintType.POSSIBLE_FORMATS, getFormats());
		mMultiFormatReader = new MultiFormatReader();
		mMultiFormatReader.setHints(hints);
	}

	public Collection<BarcodeFormat> getFormats() {
		return mFormats == null ? ALL_FORMATS : mFormats;
	}

	static {
		if (CameraManagerController.useQRFrameForBarcode) {
			ALL_FORMATS.add(BarcodeFormat.UPC_A);
			ALL_FORMATS.add(BarcodeFormat.UPC_E);
			ALL_FORMATS.add(BarcodeFormat.EAN_13);
			ALL_FORMATS.add(BarcodeFormat.EAN_8);
			ALL_FORMATS.add(BarcodeFormat.RSS_14);
			ALL_FORMATS.add(BarcodeFormat.CODE_39);
			ALL_FORMATS.add(BarcodeFormat.CODE_93);
			ALL_FORMATS.add(BarcodeFormat.CODE_128);
			ALL_FORMATS.add(BarcodeFormat.ITF);
			ALL_FORMATS.add(BarcodeFormat.CODABAR);
			ALL_FORMATS.add(BarcodeFormat.QR_CODE);

		}

		ALL_FORMATS.add(BarcodeFormat.DATA_MATRIX);
		ALL_FORMATS.add(BarcodeFormat.PDF_417);
	}

	protected static ArrayList<TISBarcodeType> getAllTISBarcodeTypes() {
		ArrayList<TISBarcodeType> barcodeTypes = new ArrayList<TISBarcodeType>();
		barcodeTypes.add(TISBarcodeType.QR_CODE);
		barcodeTypes.add(TISBarcodeType.AZTEC_CODE);
		barcodeTypes.add(TISBarcodeType.DATA_MATRIX_CODE);
		barcodeTypes.add(TISBarcodeType.UPCE_CODE);
		barcodeTypes.add(TISBarcodeType.CODE_39_CODE);
		barcodeTypes.add(TISBarcodeType.CODE_39_MOD_43_CODE);
		barcodeTypes.add(TISBarcodeType.EAN_13_CODE);
		barcodeTypes.add(TISBarcodeType.EAN_8_CODE);
		barcodeTypes.add(TISBarcodeType.CODE_93_CODE);
		barcodeTypes.add(TISBarcodeType.CODE_128_CODE);
		barcodeTypes.add(TISBarcodeType.PDF_417_CODE);
		barcodeTypes.add(TISBarcodeType.INTERLEAVED_2_OF_5_CODE);
		barcodeTypes.add(TISBarcodeType.ITF_14_CODE);
		return barcodeTypes;
	}

	/*public synchronized Rect getFramingRect(int orientation) {
		Point viewResolution = new Point(viewWidth, viewHeight);
		int width;
		int height;
		if (orientation != 1) {
			//width = findDesiredDimensionInRange(0.625F, viewResolution.x, 240, 1200);
			//height = findDesiredDimensionInRange(0.625F, viewResolution.y, 240, 675);
			width = viewWidth;//findDesiredDimensionInRange(0.625F, viewResolution.x, 240, 1200);
			height = viewHeight;//findDesiredDimensionInRange(0.625F, viewResolution.y, 240, 675);
		} else {
			width = findDesiredDimensionInRange(0.875F, viewResolution.x, 240, 945);
			height = findDesiredDimensionInRange(0.375F, viewResolution.y, 240, 720);
		}

		int leftOffset = (viewResolution.x - width) / 2;
		int topOffset = (viewResolution.y - height) / 2;
		return new Rect(leftOffset, topOffset, leftOffset + width, topOffset + height);
	}*/

	private static int findDesiredDimensionInRange(float ratio, int resolution, int hardMin, int hardMax) {
		int dim = (int) (ratio * (float) resolution);
		return dim < hardMin ? hardMin : (dim > hardMax ? hardMax : dim);
	}

	public synchronized Rect getFramingRectInPreview(int previewWidth, int previewHeight, int orientation) {

		if (mFramingRectInPreview == null) {
			org.opencv.core.Rect openCVFrame  = (CameraManagerController.getOcrAnalyzeSession(CameraController.getInstance()).getCheckBoundaries()).getValidationRect();
			Rect framingRect = new Rect(openCVFrame.x,openCVFrame.y,openCVFrame.width,openCVFrame.height);

			//new Rect(checkBoundariesRectView.getScaledValidationRect().x,checkBoundariesRectView.getScaledValidationRect().y,checkBoundariesRectView.getScaledValidationRect().width,checkBoundariesRectView.getScaledValidationRect().height);
			int viewFinderViewWidth = CameraConfigurationManager.screenResolution.x;
			int viewFinderViewHeight = CameraConfigurationManager.screenResolution.y;
			if (framingRect == null || viewFinderViewWidth == 0 || viewFinderViewHeight == 0) {
				return null;
			}

			Rect rect = new Rect(framingRect);
//			rect.left = rect.left * previewWidth / viewFinderViewWidth;
//			rect.right = rect.right * previewWidth / viewFinderViewWidth;
//			rect.top = rect.top * previewHeight / viewFinderViewHeight;
//			rect.bottom = rect.bottom * previewHeight / viewFinderViewHeight;

			mFramingRectInPreview = rect;
		}
		return mFramingRectInPreview;

	}

	Result scanImage(byte[] data, int width, int height, int orientation,Mat image) {
		if (orientation == Configuration.ORIENTATION_PORTRAIT && data != null) {
			byte[] rotatedData = new byte[data.length];
			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++)
					rotatedData[x * height + height - y - 1] = data[x + y * width];
			}
			int tmp = width;
			width = height;
			height = tmp;
			data = rotatedData;
		}

		Result rawResult = null;
		Result rawResultOnRgb = null;
		PlanarYUVLuminanceSource source = null;
		RGBLuminanceSource RgbSource = null;
		boolean useRgbImage = true;

		if (!CameraManagerController.useCameraAPI2 // here we will use cropped RGB color as input to the Zxing
				&& !CameraManagerController.deviceName.equals("LGE Nexus 5X")
				&& !CameraManagerController.isDynamicCapture
				&& CameraManagerController.BarcodeDetectionTries < CameraManagerController.maxBarcodeTries/2
				&& !RawImagesFlowManager.isLoadMode()){
			 useRgbImage = false;
		}

		if (!useRgbImage) { // here we send direct frames from the camera to Zxing.
			 source = buildLuminanceSource(data, width, height, orientation, image);
		}

		else{
			if (image == null || image.empty()){
				return null;
			}
			if (!CameraManagerController.isDynamicCapture){ // on dynamice capture we already get the cropped image
				org.opencv.core.Rect openCVFrame = (CameraManagerController.getOcrAnalyzeSession(CameraController.getInstance()).getCheckBoundaries()).getValidationRect();
				image = new Mat(image, openCVFrame);
			}

			Bitmap bmp = Bitmap.createBitmap(image.cols(),
					image.rows(), Bitmap.Config.ARGB_8888);
			Utils.matToBitmap(image, bmp);
			int[] pixels = new int[bmp.getHeight() * bmp.getWidth()];
			bmp.getPixels(pixels, 0, bmp.getWidth(), 0, 0, bmp.getWidth(),
					bmp.getHeight());
			 RgbSource = new RGBLuminanceSource(bmp.getWidth(),bmp.getHeight(),pixels);

		}
		BinaryBitmap bitmap;
		if (source != null || RgbSource!= null) {
			if (!useRgbImage)
			 bitmap = new BinaryBitmap(new HybridBinarizer(source));
			else
				bitmap = new BinaryBitmap(new HybridBinarizer(RgbSource));

			try {
				rawResult = mMultiFormatReader.decodeWithState(bitmap);
				Logger.e("BarCode Read", rawResult.getText());
			} catch (ReaderException re) {

				Logger.e("BarCode Read", Log.getStackTraceString(re));
				// continue
			} catch (NullPointerException npe) {
				// This is terrible

				Logger.e("BarCode Read", Log.getStackTraceString(npe));
			} catch (ArrayIndexOutOfBoundsException aoe) {

				Logger.e("BarCode Read", Log.getStackTraceString(aoe));
			} finally {
				mMultiFormatReader.reset();
			}

		}

		return rawResult;
	}

	private static TISBarcodeType convertZxingToTISBarcodeType(String type) {
		TISBarcodeType tisBarcodeType;
		BarcodeFormat barcodeType = BarcodeFormat.valueOf(type);
		switch (barcodeType) {
			case UPC_E:
				tisBarcodeType = TISBarcodeType.UPCE_CODE;
				break;
			case CODE_39:
				if (mTISFormats.contains(TISBarcodeType.CODE_39_MOD_43_CODE))
					tisBarcodeType = TISBarcodeType.CODE_39_MOD_43_CODE;
				else
					tisBarcodeType = TISBarcodeType.CODE_39_CODE;
				break;
			case EAN_13:
				tisBarcodeType = TISBarcodeType.EAN_13_CODE;
				break;
			case EAN_8:
				tisBarcodeType = TISBarcodeType.EAN_8_CODE;
				break;
			case CODE_93:
				tisBarcodeType = TISBarcodeType.CODE_93_CODE;
				break;
			case CODE_128:
				tisBarcodeType = TISBarcodeType.CODE_128_CODE;
				break;
			case PDF_417:
				tisBarcodeType = TISBarcodeType.PDF_417_CODE;
				break;
			case QR_CODE:
				tisBarcodeType = TISBarcodeType.QR_CODE;
				break;
			case AZTEC:
				tisBarcodeType = TISBarcodeType.AZTEC_CODE;
				break;
			case ITF:
				if (mTISFormats.contains(TISBarcodeType.INTERLEAVED_2_OF_5_CODE))
					tisBarcodeType = TISBarcodeType.INTERLEAVED_2_OF_5_CODE;
				else
					tisBarcodeType = TISBarcodeType.ITF_14_CODE;
				break;
			case DATA_MATRIX:
				tisBarcodeType = TISBarcodeType.DATA_MATRIX_CODE;
				break;
			default:
				return null;
		}
		return tisBarcodeType;
	}

	private static List<BarcodeFormat> translateTISBarcodeToZxingArray(List<TISBarcodeType> tisBarcodeList) {
		ArrayList<BarcodeFormat> barcodeTypes = new ArrayList<BarcodeFormat>();
		for (TISBarcodeType tisBarcodeType : tisBarcodeList) {
			switch (tisBarcodeType) {
				case UPCE_CODE:
					barcodeTypes.add(BarcodeFormat.UPC_E);
					break;
				case CODE_39_MOD_43_CODE:
				case CODE_39_CODE:
					barcodeTypes.add(BarcodeFormat.CODE_39);
					break;
				case EAN_13_CODE:
					barcodeTypes.add(BarcodeFormat.EAN_13);
					barcodeTypes.add(BarcodeFormat.UPC_A);
					break;
				case EAN_8_CODE:
					barcodeTypes.add(BarcodeFormat.EAN_8);
					break;
				case CODE_93_CODE:
					barcodeTypes.add(BarcodeFormat.CODE_93);
					break;
				case CODE_128_CODE:
					barcodeTypes.add(BarcodeFormat.CODE_128);
					break;
				case PDF_417_CODE:
					barcodeTypes.add(BarcodeFormat.PDF_417);
					break;
				case QR_CODE:
					barcodeTypes.add(BarcodeFormat.QR_CODE);
					break;
				case AZTEC_CODE:
					barcodeTypes.add(BarcodeFormat.AZTEC);
					break;
				case INTERLEAVED_2_OF_5_CODE:
				case ITF_14_CODE:
					barcodeTypes.add(BarcodeFormat.ITF);
					break;
				case DATA_MATRIX_CODE:
					barcodeTypes.add(BarcodeFormat.DATA_MATRIX);
					break;
				default:
					// barcodeType = 0;
			}
		}

		return barcodeTypes;
	}
	public PlanarYUVLuminanceSource buildLuminanceSource(byte[] data, int width, int height, int orientation,Mat image) {
		Rect rect = getFramingRectInPreview(width, height, orientation);
		if (rect == null) {
			return null;
		}
		PlanarYUVLuminanceSource source = null;
		RGBLuminanceSource rgbSource = null;
		org.opencv.core.Rect openCVFrame  = (CameraManagerController.getOcrAnalyzeSession(CameraController.getInstance()).getCheckBoundaries()).getValidationRect();

		image = new Mat(image,openCVFrame);
		try {
			source = new PlanarYUVLuminanceSource(data, width, height, openCVFrame.x, openCVFrame.y, openCVFrame.width, openCVFrame.height, false);

		} catch (Exception e) {
		}

		return source;
	}




	public PlanarYUVLuminanceSource getFramingRectInPreview(byte[] data, int width, int height, int orientation) {
		Rect rect =  getFramingRectInPreview(width, height, orientation);//new Rect(0,0,width,height);
//		getFramingRectInPreview(width, height, orientation);
		if (rect == null) {
			return null;
		}
		// Go ahead and assume it's YUV rather than die.
		PlanarYUVLuminanceSource source = null;

		try {
			//source = new PlanarYUVLuminanceSource(data, width, height, rect.left, rect.top, rect.width(), rect.height(), false);
			source = new PlanarYUVLuminanceSource(data, width, height, rect.left, rect.top, rect.width(), rect.height(), false);
		} catch (Exception e) {
		}

		return source;
	}

	/**
	 * Holds scanning result data of bar-codes detected in the document.
	 *
	 */
	static public class BarcodeResult {

		private static String BARCODE_TYPE_FRONT = "barcodeTypeFront";
		private static String BARCODE_TYPE_BACK = "barcodeTypeBack";
		private static String BARCODE_DATA_FRONT = "barcodeDataFront";
		private static String BARCODE_DATA_BACK = "barcodeDataBack";
		private static String BARCODE_PARSED_DATA_FRONT = "barcodeParsedDataFront";
		private static String BARCODE_PARSED_DATA_BACK = "barcodeParsedDataBack";

		private TISBarcodeType barcodeTypeFront = null;
		private TISBarcodeType barcodeTypeBack = null;
		private String barcodeDataFront = null;
		private String barcodeDataBack = null;
		private HashMap<String, String> barcodeParsedDataFront= null;
		private HashMap<String, String> barcodeParsedDataBack= null;

		protected BarcodeResult() {

		}

		public BarcodeResult(String[] barcodeResultArray) {
			if (barcodeResultArray == null || barcodeResultArray.length != 4)
				return;

			barcodeTypeFront = TISBarcodeType.getEnum(Integer.parseInt(barcodeResultArray[0]));
			barcodeTypeBack = TISBarcodeType.getEnum(Integer.parseInt(barcodeResultArray[2]));
			barcodeDataFront = barcodeResultArray[1];
			barcodeDataBack = barcodeResultArray[3];
		}

		protected BarcodeResult(Bundle bundle) {
			if (bundle != null) {
				barcodeTypeFront = TISBarcodeType.getEnum(bundle.getInt(BARCODE_TYPE_FRONT));
				barcodeTypeBack = TISBarcodeType.getEnum(bundle.getInt(BARCODE_TYPE_BACK));
				barcodeDataFront = bundle.getString(BARCODE_DATA_FRONT);
				barcodeDataBack = bundle.getString(BARCODE_DATA_BACK);
				barcodeParsedDataFront = (HashMap<String, String>) bundle.getSerializable(BARCODE_PARSED_DATA_FRONT);
				barcodeParsedDataBack = (HashMap<String, String>) bundle.getSerializable(BARCODE_PARSED_DATA_BACK);
			}
		}

		/**
		 * To bundle.
		 *
		 * @return the bundle
		 */
		protected Bundle toBundle() {
			Bundle bundle = new Bundle();
			bundle.putInt(BARCODE_TYPE_FRONT, (barcodeTypeFront == null)? -1: barcodeTypeFront.getValue());
			bundle.putInt(BARCODE_TYPE_BACK, (barcodeTypeBack == null)? -1: barcodeTypeBack.getValue());
			bundle.putString(BARCODE_DATA_FRONT, barcodeDataFront);
			bundle.putString(BARCODE_DATA_BACK, barcodeDataBack);
			bundle.putSerializable(BARCODE_PARSED_DATA_FRONT, barcodeParsedDataFront);
			bundle.putSerializable(BARCODE_PARSED_DATA_BACK, barcodeParsedDataBack);
			return bundle;
		}

		/**
		 * Indicated if any barcode data was detected and parsed
		 * @return true - if no barcode data was parsed. false - otherwise
		 */
		public boolean isEmpty() {
			return barcodeDataFront == null && barcodeDataBack == null;
		}

		/**
		 * Parse the raw data of the barcode
		 */
		public void parse() {
			if (barcodeTypeFront == TISBarcodeType.PDF_417_CODE)
				barcodeParsedDataFront = OcrValidationUtils.DLBarcodeParser.parseDLBarcode(barcodeDataFront);
			else
				barcodeParsedDataFront = new HashMap<>();
			if (barcodeTypeBack == TISBarcodeType.PDF_417_CODE)
				barcodeParsedDataBack = OcrValidationUtils.DLBarcodeParser.parseDLBarcode(barcodeDataBack);
			else
				barcodeParsedDataBack = new HashMap<>();
		}

		protected void setBarcodeTypeFront(String zxingBarcodeType) {
			barcodeTypeFront = convertZxingToTISBarcodeType(zxingBarcodeType);
		}

		protected void setBarcodeTypeBack(String zxingBarcodeType) {
			barcodeTypeBack = convertZxingToTISBarcodeType(zxingBarcodeType);
		}

		/**
		 *
		 * @return the barcode type in the front side of the document
		 */
		public TISBarcodeType getBarcodeTypeFront() {
			return barcodeTypeFront;
		}

		/**
		 *
		 * @return the barcode type in the back side of the document
		 */
		public TISBarcodeType getBarcodeTypeBack() {
			return barcodeTypeBack;
		}

		/**
		 *
		 * @return the integer value the barcode type in the front side of the document
		 */
		public int getBarcodeTypeFrontForBundle() {
			return barcodeTypeFront == null? -1: barcodeTypeFront.getValue();
		}

		/**
		 *
		 * @return the integer value the barcode type in the back side of the document
		 */
		public int getBarcodeTypeBackForBundle() {
			return barcodeTypeBack == null? -1: barcodeTypeBack.getValue();
		}

		/**
		 *
		 * @return the barcode parsed String located in the front side of the document
		 */
		public String getBarcodeDataFront() {
			return barcodeDataFront;
		}

		protected void setBarcodeDataFront(String data) {
			barcodeDataFront = data;
		}

		/**
		 *
		 * @return the barcode parsed String located in the back side of the document
		 */
		public String getBarcodeDataBack() {
			return barcodeDataBack;
		}

		protected void setBarcodeDataBack(String data) {
			barcodeDataBack = data;
		}



		public String toString() {
			return "BarcodeResult barcodeTypeFront(" + barcodeTypeFront + ") barcodeTypeBack(" + barcodeTypeBack + ") barcodeDataFront(" + barcodeDataFront + ") barcodeDataBack("
					+ barcodeDataBack + ")";
		}
	}

}
