package com.topimagesystems.controllers.cropping;

import android.annotation.TargetApi;
import android.app.AlertDialog;
import android.app.Fragment;
import android.content.DialogInterface;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.PointF;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.animation.Animation;
import android.view.animation.TranslateAnimation;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ProgressBar;

import com.topimagesystems.R;
import com.topimagesystems.controllers.imageanalyze.CameraConfigurationManager;
import com.topimagesystems.controllers.imageanalyze.CameraController;
import com.topimagesystems.controllers.imageanalyze.CameraManagerController;
import com.topimagesystems.controllers.imageanalyze.CameraTypes;
import com.topimagesystems.data.SessionResultParams;
import com.topimagesystems.micr.GenericBoundingBoxResult;
import com.topimagesystems.ui.SelectionCroppingView;
import com.topimagesystems.ui.TouchImageView;
import com.topimagesystems.util.FileUtils;
import com.topimagesystems.util.Logger;
import com.topimagesystems.util.StringUtils;
import com.topimagesystems.util.UserInterfaceUtils;

import org.opencv.android.Utils;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Size;
import org.opencv.imgproc.Imgproc;
import org.opencv.utils.Converters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.topimagesystems.controllers.imageanalyze.CameraManagerController.getOcrAnalyzeSession;

/**
 * Created by ruthlosser on 13/12/2016.
 */

@TargetApi(Build.VERSION_CODES.HONEYCOMB)
public class ManualCroppingController extends Fragment {

    public static final String EXTRA_IS_FRONT = "com.topimagesystems.controllers.manualcapture.isFront";

    public static final int ORIENTATION_LANDSCAPE = 0;
    public static final int ORIENTATION_PORTRAIT = 1;
    public static final int ORIENTATION_SESSION = 2;

    private Bitmap originalBitmap;
    private ImageView originalImageView;
    private ImageView backgroundImageView;
    private Bitmap[] scaledBitmaps = new Bitmap[4];
    private SelectionCroppingView selectionPolygonView;
    private GenericBoundingBoxResult originalBoundingBoxResult;
    private GenericBoundingBoxResult currentBoundingBoxResult;
    private FrameLayout imageFrameLayout;

    private ImageButton backBtn;
    private ImageButton rotateCWBtn;
    private ImageButton rotateCCWBtn;
    private ImageButton expandBtn;
    private ImageButton magnetBtn;
    private ImageButton confirmBtn;

    private ProgressBar spinner;

    private int rotateState = 0;

    SelectionCroppingView.ISelectionListener selectionListener;

    FrameLayout confirmLayout;

    private TranslateAnimation translateAnimation;

    private boolean shouldScaleWhileRotate = true;
    private boolean isInConfirmState = false;

    private int toDegrees;
    private CameraController activity;
    private Fragment fragment;
    private View fragmentView;

    private boolean isCurrentlyStarting;

    String TAG = getClass().getSimpleName();

    public ManualCroppingController() {
        super();
        isCurrentlyStarting = true;
    }

    public void onStart() {
        super.onStart();
        isCurrentlyStarting = false;
    }

    public synchronized boolean isStarting() {
        return isCurrentlyStarting;
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        fragment= this;
        activity = (CameraController) getActivity();

        fragmentView = inflater.inflate(R.layout.manual_cropping_layout, container, false);
        originalImageView = (ImageView) fragmentView.findViewById(R.id.originalImageView);
        backgroundImageView = (ImageView) fragmentView.findViewById(R.id.backgroundImageView);
        selectionPolygonView = (SelectionCroppingView) fragmentView.findViewById(R.id.selectionPolygonView);

        synchronized (this) {
        // get bitmap and handle exceptions if bitmap is not valid
        try {
            originalBitmap = getImageFromBundle();
        } catch (Exception e) {
            AlertDialog alertDialog =
                    new AlertDialog.Builder(activity)
                            .setTitle("Error")
                            .setMessage(e.getMessage())
                            .setCancelable(false)
                            .setPositiveButton(
                                    StringUtils.dynamicString(activity.getBaseContext(), "TISFlowOK"),
                                    new DialogInterface.OnClickListener() {
                                        @Override
                                        public void onClick(DialogInterface dialog, int which) {
                                            getActivity().getFragmentManager().beginTransaction().remove(fragment).commit();
                                        }
                                    })
                            .create();
            alertDialog.show();
            return fragmentView;
        }

        // set image frame
        imageFrameLayout = (FrameLayout) fragmentView.findViewById(R.id.imageFrameLayout);
            originalImageView.post(new Runnable() {
                @Override
                public void run() {
                    if (originalBitmap != null) {
                        if (!hasOriginalBoundingBox)
                            setBoundingBoxIfRectNotFound();
                        scaledBitmaps[0] = scaledBitmap(originalBitmap, originalImageView.getWidth(), originalImageView.getHeight());
                        initViews();
                    }
                }
            });

        }
        // set buttons

        backBtn = (ImageButton) fragmentView.findViewById(R.id.cancelImageBtn);
        backBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doBack(v);
            }
        });

        rotateCWBtn = (ImageButton) fragmentView.findViewById(R.id.rotateImageBtn);
        rotateCWBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doRotate(v);
            }
        });

        rotateCCWBtn = (ImageButton) fragmentView.findViewById(R.id.rotateRevImageBtn);
        rotateCCWBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doRotate(v);
            }
        });

        expandBtn = (ImageButton) fragmentView.findViewById(R.id.expandImageBtn);
        expandBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doExpand(v);
            }
        });
        expandBtn.setEnabled(hasOriginalBoundingBox);

        magnetBtn = (ImageButton) fragmentView.findViewById(R.id.magnetImageBtn);
        magnetBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doMagnet(v);
            }
        });
        confirmBtn = (ImageButton) fragmentView.findViewById(R.id.confirmImageBtn);
        confirmBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                doConfirm(v);
            }
        });
        selectionListener = new SelectionCroppingView.ISelectionListener() {
            @Override
            public void onSelectionChanged() {
                if (isCurrentlyStarting)
                    return;

                enableAllButtons();


                Map<Integer, PointF> currentPoints = orderedValidEdgePoints(
                        selectionPolygonView.getBoundaries(),
                        selectionPolygonView.getPointsInsideBounds()
                );

                if (hasOriginalBoundingBox) {
                    Map<Integer, PointF> magnetPoints = getOriginalEdgePoints(selectionPolygonView.getBoundaries());
                    if (!isMapPointsEqual(magnetPoints, currentPoints)) {
                        magnetBtn.setVisibility(View.VISIBLE);
                        expandBtn.setVisibility(View.GONE);
                    } else {
                        magnetBtn.setVisibility(View.GONE);
                        expandBtn.setVisibility(View.VISIBLE);
                    }
                } else {
                    Map<Integer, PointF> edgePoints = getOutlinePoints(selectionPolygonView.getBoundaries());
                    expandBtn.setEnabled(!isMapPointsEqual(edgePoints, currentPoints));
                }
                confirmBtn.setEnabled(selectionPolygonView.isValid());
            }

            @Override
            public void onSelectionStarted() {
                disableAllButtons();
            }
        };

        selectionPolygonView.setSelectionListener(selectionListener);

        confirmLayout = (FrameLayout) fragmentView.findViewById(R.id.confirmLayout);

        FrameLayout.LayoutParams confirmLayoutParams = (FrameLayout.LayoutParams) confirmLayout.getLayoutParams();
        confirmLayoutParams.leftMargin = 10000;
        confirmLayout.setLayoutParams(confirmLayoutParams);

        spinner = (ProgressBar) fragmentView.findViewById(R.id.mc_progress_bar);
        spinner.setVisibility(View.GONE);

        return fragmentView;
    }

    private void disableAllButtons() {
        backBtn.setEnabled(false);
        rotateCWBtn.setEnabled(false);
        rotateCCWBtn.setEnabled(false);
        expandBtn.setEnabled(false);
        magnetBtn.setEnabled(false);
        confirmBtn.setEnabled(false);
    }


    private void enableAllButtons() {
        backBtn.setEnabled(true);
        rotateCWBtn.setEnabled(true);
        rotateCCWBtn.setEnabled(true);
        expandBtn.setEnabled(true);
        magnetBtn.setEnabled(true);
        confirmBtn.setEnabled(true);
    }

    @Override
    public void onDetach() {
        if (originalBitmap != null && !originalBitmap.isRecycled()) {
            originalBitmap.recycle();
            originalBitmap = null;
        }
        for (int i = 0; i < scaledBitmaps.length; i++) {
            if (scaledBitmaps[i] != null && !scaledBitmaps[i].isRecycled())
                scaledBitmaps[i].recycle();
            scaledBitmaps[i] = null;
        }
        scaledBitmaps = null;
        selectionListener = null;
        if (translateAnimation != null) {
            translateAnimation.setAnimationListener(null);
            translateAnimation= null;
        }


        super.onDetach();
    }

    public void onBackPressed()
    {
        if (isInConfirmState){
            translateAnimation = new TranslateAnimation(Animation.RELATIVE_TO_SELF, 0, Animation.RELATIVE_TO_SELF, -1, Animation.ABSOLUTE, 0, Animation.ABSOLUTE, 0);
            translateAnimation.setDuration(500);
            translateAnimation.setAnimationListener(new Animation.AnimationListener() {
                @Override
                public void onAnimationStart(Animation animation) {

                }

                @Override
                public void onAnimationEnd(Animation animation) {
                    translateAnimation.setAnimationListener(null);
                    FrameLayout.LayoutParams confirmLayoutParams = (FrameLayout.LayoutParams) confirmLayout.getLayoutParams();
                    confirmLayoutParams.leftMargin = 10000;
                    confirmLayout.setLayoutParams(confirmLayoutParams);

                    enableAllButtons();

                }

                @Override
                public void onAnimationRepeat(Animation animation) {

                }
            });

            confirmLayout.startAnimation(translateAnimation);
            isInConfirmState = false;
        }
        else {
            if (activity != null && activity.getHandler() != null){
                Message message = activity.getHandler().obtainMessage(CameraTypes.MESSAGE_FINISH_CROPPING_CONTROLLER);
                message.obj = false; // did not crop
                message.sendToTarget();
            }
            else {
                try {
                    getActivity().getFragmentManager().beginTransaction().remove(this).commit();
                } catch (Exception e) {
                    Logger.e(this.getClass().getSimpleName(), e.toString());
                }
            }
        }
    }


    public void doBack(View v) {
        onBackPressed();
    }

    public void doExpand(View v) {
        if (selectionPolygonView == null || selectionPolygonView.getBoundaries() == null)
            return;

        Map<Integer, PointF> orderedPoints = getOutlinePoints(selectionPolygonView.getBoundaries());
        selectionPolygonView.setPoints(orderedPoints);

        if (hasOriginalBoundingBox) {
            expandBtn.setVisibility(View.GONE);
            magnetBtn.setVisibility(View.VISIBLE);
        } else {
            expandBtn.setEnabled(false);
        }
    }

    public void doMagnet(View v) {
        if (selectionPolygonView == null || selectionPolygonView.getBoundaries() == null || originalBoundingBoxResult == null)
            return;

        Map<Integer, PointF> orderedPoints = getOriginalEdgePoints(selectionPolygonView.getBoundaries());
        selectionPolygonView.setPoints(orderedPoints);

        magnetBtn.setVisibility(View.GONE);
        expandBtn.setVisibility(View.VISIBLE);
    }


    public void doConfirm(View v) {
//        spinner.setVisibility(View.VISIBLE);

        disableAllButtons();

        // updating getting the last user selection

        if (isInConfirmState) {

            spinner.setVisibility(View.VISIBLE);
            float[] coordinates = new float[8];
            UserInterfaceUtils.fillPointsArrayFromBoundingBox(coordinates, currentBoundingBoxResult);

            CameraManagerController.getOcrAnalyzeSession(activity).setCroppingCoordinates(coordinates);
            CameraManagerController.getOcrAnalyzeSession(activity).setStillsBoundingBox(currentBoundingBoxResult.getRect());

            if (activity != null && activity.getHandler() != null) {

                Handler handler = activity.getHandler();

                Message message = handler.obtainMessage(CameraTypes.MESSAGE_PROCESS_CAPTURED_IMAGE);
                String filePath;
                if (CameraManagerController.getOcrAnalyzeSession(activity).captureMode == CameraTypes.CaptureMode.FRONT) {
                    filePath =  getOcrAnalyzeSession(activity).getFrontImagePath();
                } else {
                    filePath =  getOcrAnalyzeSession(activity).getBackImagePath();
                }
                message.obj = filePath;
                message.arg1 = CameraConfigurationManager.videoResolutionWidth;
                message.arg2 = CameraConfigurationManager.videoResolutionHeight;
                message.sendToTarget();
            }

            return;
        }

        new CropImageAndDisplay().execute();

    }

    private class CropImageAndDisplay extends AsyncTask<Void, Integer, Bitmap>
    {

        protected void onPreExecute (){
        }

        protected Bitmap doInBackground(Void...arg0) {
            GenericBoundingBoxResult bb = calculateCurrentBoundingBox(false);
            // getting undistorted cropped image
            double topWidth = Math.sqrt(Math.pow(bb.topRightX - bb.topLeftX, 2) + Math.pow(bb.topRightY - bb.topLeftY, 2));
            double bottomWidth = Math.sqrt(Math.pow(bb.bottomRightX - bb.bottomLeftX, 2) + Math.pow(bb.bottomRightY - bb.bottomLeftY, 2));
            double leftHeight = Math.sqrt(Math.pow(bb.bottomLeftX - bb.topLeftX, 2) + Math.pow(bb.bottomLeftY - bb.topLeftY, 2));
            double rightHeight = Math.sqrt(Math.pow(bb.bottomRightX - bb.topRightX, 2) + Math.pow(bb.bottomRightY - bb.topRightY, 2));
            double destW = Math.min(bottomWidth, topWidth);
            double destH = Math.min(leftHeight, rightHeight);

            List<Point> srcPoints = new ArrayList<Point>();
            srcPoints.add(new Point(bb.topLeftX, bb.topLeftY));
            srcPoints.add(new Point(bb.topRightX, bb.topRightY));
            srcPoints.add(new Point(bb.bottomRightX, bb.bottomRightY));
            srcPoints.add(new Point(bb.bottomLeftX, bb.bottomLeftY));

            List<Point> destPoints= new ArrayList<Point>();
            destPoints.add(new Point(0, 0));
            destPoints.add(new Point(destW - 1, 0));
            destPoints.add(new Point(destW - 1, destH - 1));
            destPoints.add(new Point(0, destH));

            Mat srcMat = Converters.vector_Point2f_to_Mat(srcPoints);
            Mat dstMat = Converters.vector_Point2f_to_Mat(destPoints);

            Mat perspectiveTransformation = Imgproc.getPerspectiveTransform(srcMat,dstMat);

            Mat inputMat = new Mat();
            Utils.bitmapToMat(originalBitmap, inputMat);

            Mat outputMat = new Mat((int) destH,(int) destW, CvType.CV_8UC4);//CvType.CV_8UC1
            Imgproc.warpPerspective(inputMat, outputMat, perspectiveTransformation, new Size(destW, destH));

            Bitmap outputBmp = Bitmap.createBitmap((int)destW,(int)destH, Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(outputMat, outputBmp);
            Imgproc.cvtColor(outputMat,outputMat,Imgproc.COLOR_BGR2RGBA);
            if (CameraManagerController.getOcrAnalyzeSession(activity).captureMode == CameraTypes.CaptureMode.FRONT) {
                CameraManagerController.getOcrAnalyzeSession(activity).setFrontCroppedStillsMat(outputMat);
            } else {
                CameraManagerController.getOcrAnalyzeSession(activity).setBackCroppedStillsMat(outputMat);
            }

            return outputBmp;
        }

        protected void onProgressUpdate(Integer...a){
        }

        protected void onPostExecute(Bitmap outputBmp) {
            TouchImageView previewView = (TouchImageView) fragmentView.findViewById(R.id.previewCroppedImage);
            previewView.setImageBitmap(outputBmp);
            previewView.setScaleType(ImageView.ScaleType.MATRIX);
            previewView.setMaxZoom(4f);


            translateAnimation = new TranslateAnimation(Animation.RELATIVE_TO_SELF, -1, Animation.RELATIVE_TO_SELF, 0, Animation.ABSOLUTE, 0, Animation.ABSOLUTE, 0);
            translateAnimation.setDuration(500);

            FrameLayout.LayoutParams confirmLayoutParams = (FrameLayout.LayoutParams) confirmLayout.getLayoutParams();
            confirmLayoutParams.leftMargin = 0;
            confirmLayout.setLayoutParams(confirmLayoutParams);
            confirmLayout.startAnimation(translateAnimation);

            spinner.setVisibility(View.GONE);
            translateAnimation.setAnimationListener(new Animation.AnimationListener() {
                @Override
                public void onAnimationStart(Animation animation) {

                }

                @Override
                public void onAnimationEnd(Animation animation) {

                    isInConfirmState = true;

                    confirmBtn.setEnabled(true);
                    backBtn.setEnabled(true);
                }

                @Override
                public void onAnimationRepeat(Animation animation) {

                }
            });


        }
    }


    public void doRotate(View v) {

        fragmentView.findViewById(R.id.rotateImageBtn).setEnabled(false);
        fragmentView.findViewById(R.id.rotateRevImageBtn).setEnabled(false);

        calculateCurrentBoundingBox(false);

        boolean clockwise = v.getTag().toString().equals("clock");

        if (clockwise)
            rotateState++;
        else {
            if (rotateState == 0)
                rotateState = 4;
            rotateState--;
        }
        toDegrees = rotateState * 90;

        originalImageView.setVisibility(View.INVISIBLE);
        rotateBitmap();

        rotateState = rotateState % 4;

        shouldScaleWhileRotate = !shouldScaleWhileRotate;

    }

    private void rotateBitmap() {

        if (scaledBitmaps[(toDegrees / 90) % 4] == null) {
            Mat mat = new Mat();
            Utils.bitmapToMat(scaledBitmaps[0], mat);
            mat = FileUtils.rotateMatInAngle(mat, toDegrees);

            scaledBitmaps[(toDegrees / 90) % 4] = Bitmap.createBitmap(mat.cols(), mat.rows(), Bitmap.Config.ARGB_8888);

            Utils.matToBitmap(mat, scaledBitmaps[(toDegrees / 90) % 4]);
        }


        originalImageView.setImageBitmap(scaledBitmaps[(toDegrees / 90) % 4]);
        originalImageView.post(new Runnable() {
            @Override
            public void run() {

                resizeImageViewAndSelection();
                originalImageView.setVisibility(View.VISIBLE);

                updateSelectionBitmap();
                updateBackgroundView();
                fragmentView.findViewById(R.id.rotateRevImageBtn).setEnabled(true);
                fragmentView.findViewById(R.id.rotateImageBtn).setEnabled(true);
            }
        });
    }

    private void updateSelectionBitmap() {
        imageFrameLayout.buildDrawingCache();
        Bitmap magnifierBitmap = imageFrameLayout.getDrawingCache();
        selectionPolygonView.setBitmap(Bitmap.createBitmap(magnifierBitmap));
    }

    private void updateBackgroundView() {
        backgroundImageView.setDrawingCacheEnabled(false);
        backgroundImageView.setImageMatrix(originalImageView.getImageMatrix());
        backgroundImageView.invalidate();
        backgroundImageView.setImageDrawable(null);
        backgroundImageView.setImageBitmap(scaledBitmaps[rotateState % 4]);
    }

    private void initViews() {
        originalImageView.setImageBitmap(scaledBitmaps[0]);
        selectionPolygonView.setVisibility(View.VISIBLE);

        originalImageView.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                if (Build.VERSION.SDK_INT >= 17) {
                    originalImageView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                } else {
                    originalImageView.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                }

                resizeImageViewAndSelection();
                updateSelectionBitmap();
                updateBackgroundView();
            }
        });
    }

    private void resizeImageViewAndSelection() {
        int measuredWidth = originalImageView.getMeasuredWidth();
        int measuredHeight = originalImageView.getMeasuredHeight();

        Drawable drawable = originalImageView.getDrawable();
        RectF rectDrawable = new RectF(drawable.getBounds());

        float leftOffset = (measuredWidth - rectDrawable.width()) / 2f;
        float topOffset = (measuredHeight - rectDrawable.height()) / 2f;

        Matrix matrix = new Matrix();
        float scale = 1.0f;
        if (leftOffset < 0)
            scale = 1.0f * measuredWidth / rectDrawable.width();
        else if (topOffset < 0)
            scale = 1.0f * measuredHeight / rectDrawable.height();

        if (scale != 1.0) {
            rectDrawable = new RectF(0, 0, rectDrawable.width() * scale, rectDrawable.height() * scale);
            leftOffset = (measuredWidth - rectDrawable.width()) / 2f;
            topOffset = (measuredHeight - rectDrawable.height()) / 2f;
        }
        matrix.postTranslate(leftOffset, topOffset);
        matrix.postScale(scale, scale, leftOffset, topOffset);
        originalImageView.setImageMatrix(matrix);
        originalImageView.invalidate();
        originalImageView.setImageDrawable(null);
        originalImageView.setImageBitmap(scaledBitmaps[rotateState % 4]);
        Matrix selectionMat = new Matrix();
        RectF bounds = new RectF(rectDrawable);
        FrameLayout.LayoutParams params = (FrameLayout.LayoutParams)imageFrameLayout.getLayoutParams();

        float frameLeftOffset = imageFrameLayout.getPaddingLeft() + params.leftMargin;
        float frameTopOffset = imageFrameLayout.getPaddingTop() + params.topMargin;
        selectionMat.setRectToRect(
                bounds,
                new RectF(
                        frameLeftOffset,
                        frameTopOffset,
                        measuredWidth + frameLeftOffset,
                        measuredHeight + frameTopOffset
                ),
                Matrix.ScaleToFit.CENTER
        );
        selectionMat.mapRect(bounds);
        selectionPolygonView.setBoundaries(bounds);

        if (originalBoundingBoxResult == null) {
            selectionPolygonView.setPoints(null);
            disableAllButtons();
        }
        else
        {
            Map<Integer, PointF> pointFs = getLastEdgePoints(bounds);
            selectionPolygonView.setPoints(pointFs);
        }
    }

    private synchronized Bitmap getImageFromBundle() throws Exception {
        Bundle args = getArguments();

        if (args == null)
            throw new Exception("Has no image to display");

        boolean isFront = args.getBoolean(EXTRA_IS_FRONT, true);
        byte[] data = isFront ? SessionResultParams.originalFront : SessionResultParams.originalBack;
        if (data == null)
            throw new Exception("Has no image to display");

        Bitmap bitmap = BitmapFactory.decodeByteArray(data, 0, data.length);
        if (bitmap == null)
            throw new Exception("Not a valid bitmap");

        return bitmap;
    }

    // have to be true as default (set to false only if rect is not found)
    private boolean hasOriginalBoundingBox = true;

    private void setBoundingBoxIfRectNotFound() {
        // should be called only if originalBitmap exist
        if (originalBitmap == null || hasOriginalBoundingBox)
            return;

        originalBoundingBoxResult = new GenericBoundingBoxResult();

        originalBoundingBoxResult.x = originalBoundingBoxResult.y = 0;

        originalBoundingBoxResult.width = originalBitmap.getWidth();
        originalBoundingBoxResult.height = originalBitmap.getHeight();

        originalBoundingBoxResult.topLeftX = originalBoundingBoxResult.topLeftY = originalBoundingBoxResult.bottomLeftX = originalBoundingBoxResult.topRightY = 0.0f;
        originalBoundingBoxResult.bottomRightX = originalBoundingBoxResult.topRightX = (float) originalBitmap.getWidth();
        originalBoundingBoxResult.bottomLeftY = originalBoundingBoxResult.bottomRightY = (float) originalBitmap.getHeight();
    }

    public synchronized void setFirstCoordinates(GenericBoundingBoxResult gbbr) {

        originalBoundingBoxResult = gbbr;

        if (originalBoundingBoxResult == null) {
            hasOriginalBoundingBox = false;
            if (originalBitmap != null)
                setBoundingBoxIfRectNotFound();
        }

        if (backBtn != null)
            enableAllButtons();

        if (selectionPolygonView != null) {
            RectF bounds = selectionPolygonView.getBoundaries();
            Map<Integer, PointF> pointFs = getLastEdgePoints(bounds);
            selectionPolygonView.setPoints(pointFs);
        }

        if (expandBtn != null)
            expandBtn.setEnabled(hasOriginalBoundingBox);
    }

    private boolean isMapPointsEqual(Map<Integer, PointF> firstMap, Map<Integer, PointF> secondMap){
        final int delta = 2;

        if (
                firstMap.get(0) != null && secondMap.get(0) != null &&
                Math.abs(firstMap.get(0).x - secondMap.get(0).x) < delta && Math.abs(firstMap.get(0).y - secondMap.get(0).y) < delta &&
                firstMap.get(1) != null && secondMap.get(1) != null &&
                Math.abs(firstMap.get(1).x - secondMap.get(1).x) < delta && Math.abs(firstMap.get(1).y - secondMap.get(1).y) < delta &&
                firstMap.get(2) != null && secondMap.get(2) != null &&
                Math.abs(firstMap.get(2).x - secondMap.get(2).x) < delta && Math.abs(firstMap.get(2).y - secondMap.get(2).y) < delta &&
                firstMap.get(3) != null && secondMap.get(3) != null &&
                Math.abs(firstMap.get(3).x - secondMap.get(3).x) < delta && Math.abs(firstMap.get(3).y - secondMap.get(3).y) < delta
                )

            return true;

        return false;
    }

    private Map<Integer,PointF> getLastEdgePoints(RectF bounds) {
        List<PointF> pointFs = getContourEdgePoints(bounds, false);
        return orderedValidEdgePoints(bounds, pointFs);
    }

    private Map<Integer,PointF> getOriginalEdgePoints(RectF bounds) {
        List<PointF> pointFs = getContourEdgePoints(bounds, true);
        return orderedValidEdgePoints(bounds, pointFs);
    }

    private synchronized Bitmap scaledBitmap(Bitmap bitmap, int width, int height) {
        int toWidth, toHeight, longerAxis, shorterAxis;

        longerAxis = Math.max(width, height);
        shorterAxis = Math.min(width, height);

        if (bitmap.getWidth() > bitmap.getHeight()) {
            // bitmap is landscape
            toWidth = longerAxis;
            toHeight = shorterAxis;
        }
        else {
            // bitmap is portrait
            toWidth = shorterAxis;
            toHeight = longerAxis;
        }

        Matrix m = new Matrix();
        m.setRectToRect(new RectF(0, 0, bitmap.getWidth(), bitmap.getHeight()), new RectF(0, 0, toWidth, toHeight), Matrix.ScaleToFit.CENTER);
        return  Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), m, true);

    }

    private Map<Integer, PointF> orderedValidEdgePoints(RectF rect, List<PointF> pointFs) {
        Map<Integer, PointF> orderedPoints = selectionPolygonView.getOrderedPoints(pointFs);
        if (!selectionPolygonView.isValid()) {
            orderedPoints = getOutlinePoints(rect);
        }
        return orderedPoints;
    }

    private Map<Integer, PointF> getOutlinePoints(RectF rect) {
        Map<Integer, PointF> outlinePoints = new HashMap<>();
        outlinePoints.put(0, new PointF(0, 0));
        outlinePoints.put(1, new PointF(rect.width(), 0));
        outlinePoints.put(2, new PointF(0, rect.height()));
        outlinePoints.put(3, new PointF(rect.width(), rect.height()));
        return outlinePoints;
    }


    private float[] getBoundingBoxWithRotation(GenericBoundingBoxResult boundingBoxResult, int rotation) {
        float[] resultPoints = new float[8];

        float[] genericPoints = new float[8];
        UserInterfaceUtils.fillPointsArrayFromBoundingBox(genericPoints, boundingBoxResult);

        float pivotX = originalBitmap.getWidth() / 2;
        float pivotY = originalBitmap.getHeight() / 2;

        if (rotation == 1)
            pivotX = pivotY;
        else if (rotation == 3)
            pivotY = pivotX;

        Matrix rotateMat = new Matrix();
        rotateMat.setRotate(rotation * 90, pivotX, pivotY);
        rotateMat.mapPoints(resultPoints, genericPoints);

        return resultPoints;
    }

    private List<PointF> getContourEdgePoints(RectF rect, boolean forceOriginalBox) {
        if (originalBoundingBoxResult == null)
            return null;

        float xRatio = rect.width() * 1.0f / ((rotateState % 2 == 0)? originalBitmap.getWidth(): originalBitmap.getHeight());
        float yRatio = rect.height() * 1.0f / ((rotateState % 2 == 0)? originalBitmap.getHeight(): originalBitmap.getWidth());

        float[] points = getBoundingBoxWithRotation(
                (forceOriginalBox || currentBoundingBoxResult == null)? originalBoundingBoxResult: currentBoundingBoxResult
                , rotateState);

        float x1 = points[0] * xRatio;
        float y1 = points[1] * yRatio;
        float x2 = points[2] * xRatio;
        float y2 = points[3] * yRatio;
        float x3 = points[4] * xRatio;
        float y3 = points[5] * yRatio;
        float x4 = points[6] * xRatio;
        float y4 = points[7] * yRatio;

        List<PointF> pointFs = new ArrayList<>();
        pointFs.add(new PointF(x1, y1));
        pointFs.add(new PointF(x2, y2));
        pointFs.add(new PointF(x3, y3));
        pointFs.add(new PointF(x4, y4));
        return pointFs;
    }

    private final int[][] pointIndexForRotation  = {{0, 1, 2, 3},{1, 3, 0, 2},{3, 2, 1, 0},{2, 0, 3, 1}};
    private int getPointIndexForRotation(int index, int rotation) {

        return pointIndexForRotation[rotation % 4][index];

    }

    private GenericBoundingBoxResult calculateCurrentBoundingBox(boolean returnRotated) {

        final int[] currentIndexesOrder=
                // will re-arrange the coordinates so when rotating back to 0 degrees the bounding box will be valid for processing
                new int[]{
                        getPointIndexForRotation(0, rotateState),
                        getPointIndexForRotation(1, rotateState),
                        getPointIndexForRotation(2, rotateState),
                        getPointIndexForRotation(3, rotateState)
                };

        if (currentBoundingBoxResult == null)
            currentBoundingBoxResult = new GenericBoundingBoxResult();

        // get latest user selection points

        Map<Integer, PointF> currentPoints = orderedValidEdgePoints(
                selectionPolygonView.getBoundaries(),
                selectionPolygonView.getPointsInsideBounds()
        );

        // translate the coordinates to original bitmap size

        RectF currentBounds = selectionPolygonView.getBoundaries();
        float xRatio =  currentBounds.width() * 1.0f / ((rotateState % 2 == 0)? originalBitmap.getWidth(): originalBitmap.getHeight());
        float yRatio = currentBounds.height() * 1.0f / ((rotateState % 2 == 0)? originalBitmap.getHeight(): originalBitmap.getWidth());
        float[] selectionPoints = new float[8];
        fillPointsArrWithPointsMap(selectionPoints, currentPoints, new PointF(xRatio, yRatio), currentIndexesOrder);

        // rotate coordinates to original rotation

        float pivotX = originalBitmap.getWidth() / 2;
        float pivotY = originalBitmap.getHeight() / 2;

        if (rotateState == 1)
            pivotX = pivotY;
        else if (rotateState == 3)
            pivotY = pivotX;

        Matrix rotateMat = new Matrix();
        rotateMat.setRotate(-rotateState * 90, pivotX, pivotY);
        float[] points = new float[8];
        rotateMat.mapPoints(points, selectionPoints);

        // update current bounding box
        UserInterfaceUtils.fillGenericBBUsingPointsArr(currentBoundingBoxResult, points);

        if (!returnRotated || (rotateState % 4) == 0)
            return currentBoundingBoxResult;

        // will return bounding box for displaying the image rotated after using OpenCV's alignment Algorithm
        fillPointsArrWithPointsMap(selectionPoints, currentPoints, new PointF(xRatio, yRatio), new int[]{0, 1, 2, 3});
        points = new float[8];
        rotateMat.mapPoints(points, selectionPoints);

        GenericBoundingBoxResult bbForRotation = new GenericBoundingBoxResult();
        UserInterfaceUtils.fillGenericBBUsingPointsArr(bbForRotation, points);
        return bbForRotation;
    }

    private void fillPointsArrWithPointsMap(float[] points, Map<Integer, PointF> map, PointF ratioConversion, int[] indexOrder) {
        points[0] = map.get(indexOrder[0]).x / ratioConversion.x;
        points[1] = map.get(indexOrder[0]).y / ratioConversion.y;
        points[2] = map.get(indexOrder[1]).x / ratioConversion.x;
        points[3] = map.get(indexOrder[1]).y / ratioConversion.y;
        points[4] = map.get(indexOrder[2]).x / ratioConversion.x;
        points[5] = map.get(indexOrder[2]).y / ratioConversion.y;
        points[6] = map.get(indexOrder[3]).x / ratioConversion.x;
        points[7] = map.get(indexOrder[3]).y / ratioConversion.y;
    }

}
