/*
 * Copyright 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.topimagesystems.camera2;

import android.Manifest;
import android.annotation.TargetApi;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.DialogFragment;
import android.app.Fragment;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.ImageFormat;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.SurfaceTexture;
import android.hardware.camera2.CameraAccessException;
import android.hardware.camera2.CameraCaptureSession;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.hardware.camera2.CameraMetadata;
import android.hardware.camera2.CaptureRequest;
import android.hardware.camera2.CaptureResult;
import android.hardware.camera2.TotalCaptureResult;
import android.hardware.camera2.params.MeteringRectangle;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.Image;
import android.media.ImageReader;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import android.util.Range;
import android.util.Size;
import android.util.SparseIntArray;
import android.view.Surface;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.topimagesystems.Constants;
import com.topimagesystems.R;
import com.topimagesystems.controllers.imageanalyze.CameraConfigurationManager;
import com.topimagesystems.controllers.imageanalyze.CameraController;
import com.topimagesystems.controllers.imageanalyze.CameraManagerController;
import com.topimagesystems.controllers.imageanalyze.CameraOverlayLayout;
import com.topimagesystems.controllers.imageanalyze.CameraSessionManager;
import com.topimagesystems.controllers.imageanalyze.PreviewCallback;
import com.topimagesystems.intent.CaptureIntent;
import com.topimagesystems.util.Logger;

import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.File;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;


@TargetApi(Build.VERSION_CODES.LOLLIPOP)
public class CameraAPI2Manager //extends Fragment
        implements View.OnClickListener {

    /**
     * Conversion from screen rotation to JPEG orientation.
     */
    private static final SparseIntArray ORIENTATIONS = new SparseIntArray();
    private static final int REQUEST_CAMERA_PERMISSION = 1;
    private static final String FRAGMENT_DIALOG = "dialog";

    static {
        ORIENTATIONS.append(Surface.ROTATION_0, 90);
        ORIENTATIONS.append(Surface.ROTATION_90, 0);
        ORIENTATIONS.append(Surface.ROTATION_180, 270);
        ORIENTATIONS.append(Surface.ROTATION_270, 180);
    }

    /**
     * Tag for the {@link Log}.
     */
    public static final String TAG = "CameraAPI2Manager";

    /**
     * Camera state: Showing camera preview.
     */
    private static final int STATE_PREVIEW = 0;

    /**
     * Camera state: Waiting for the focus to be locked.
     */
    private static final int STATE_WAITING_LOCK = 1;

    /**
     * Camera state: Waiting for the exposure to be precapture state.
     */
    private static final int STATE_WAITING_PRECAPTURE = 2;

    /**
     * Camera state: Waiting for the exposure state to be something other than precapture.
     */
    private static final int STATE_WAITING_NON_PRECAPTURE = 3;

    /**
     * Camera state: Picture was taken.
     */
    private static final int STATE_PICTURE_TAKEN = 4;

    /**
     * Max preview width that is guaranteed by Camera2 API
     */
    private static final int MAX_PREVIEW_WIDTH = 1920;

    /**
     * Max preview height that is guaranteed by Camera2 API
     */
    private static final int MAX_PREVIEW_HEIGHT = 1080;

    public CameraOverlayLayout cameraLayout;
    private ViewGroup cameraView;
    private PreviewCallback processCallback;
    public boolean isSessionClosed = false;
    public static boolean stillCaptureStarted = false;
    private static boolean convertImageToMatInProcess = false;
    private Image image;



    /**
     * {@link TextureView.SurfaceTextmTextureView
     * ureListener} handles several lifecycle events on a
     * {@link TextureView}.
     */
    private final TextureView.SurfaceTextureListener mSurfaceTextureListener
            = new TextureView.SurfaceTextureListener() {

        @Override
        public void onSurfaceTextureAvailable(SurfaceTexture texture, int width, int height) {
            openCamera(width, height);
        }

        @Override
        public void onSurfaceTextureSizeChanged(SurfaceTexture texture, int width, int height) {
            configureTransform(width, height);
        }

        @Override
        public boolean onSurfaceTextureDestroyed(SurfaceTexture texture) {
            return true;
        }

        @Override
        public void onSurfaceTextureUpdated(SurfaceTexture texture) {
        }

    };

    public void setCameraOverlayView(ViewGroup v){
        cameraView = v;
    }

    /**
     * ID of the current {@link CameraDevice}.
     */
    private String mCameraId;

    /**
     * An {@link AutoFitTextureView} for camera preview.
     */
    private AutoFitTextureView mTextureView;

    /**
     * A {@link CameraCaptureSession } for camera preview.
     */
    private CameraCaptureSession mCaptureSession;

    /**
     * A reference to the opened {@link CameraDevice}.
     */
    private CameraDevice mCameraDevice;

    /**
     * The {@link Size} of camera preview.
     */
    private static Size mPreviewSize;

    /**
     * {@link CameraDevice.StateCallback} is called when {@link CameraDevice} changes its state.
     */
    private final CameraDevice.StateCallback mStateCallback = new CameraDevice.StateCallback() {

        @Override
        public void onOpened(CameraDevice cameraDevice) {
            // This method is called when the camera is opened.  We start camera preview here.
            mCameraOpenCloseLock.release();
            mCameraDevice = cameraDevice;
            createCameraPreviewSession();


//            mImageReader.getSurface();
//         //   mImageReader = ImageReader.newInstance(width, height, ImageFormat.JPEG, 5);
//            mImageReader.setOnImageAvailableListener(new ImageReader.OnImageAvailableListener() {
//                @Override
//                public void onImageAvailable(ImageReader reader) {
//
//                    Logger.i("Image Read",reader.toString());
//
//                }
//            }, mBackgroundHandler);


        }

        public CameraOverlayLayout getcameraOverlayView(){
            return cameraLayout;
        }
        @Override
        public void onDisconnected(CameraDevice cameraDevice) {
            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
        }

        @Override
        public void onError(CameraDevice cameraDevice, int error) {
            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
            Activity activity = CameraController.getInstance();
            if (null != activity) {
                activity.finish();
            }
        }

    };

    /**
     * An additional thread for running tasks that shouldn't block the UI.
     */
    private HandlerThread mBackgroundThread;

    /**
     * A {@link Handler} for running tasks in the background.
     */
    private Handler mBackgroundHandler;

    /**
     * An {@link ImageReader} that handles still image capture.
     */
    private ImageReader mImageReader;

    /**
     * This is the output file for our picture.
     */


    private ImageReader mStillsImageReader;

    private File mFile;

    /**
     * This a callback object for the {@link ImageReader}. "onImageAvailable" will be called when a
     * still image is ready to be saved.
     */
    private final ImageReader.OnImageAvailableListener mOnImageAvailableListener
            = new ImageReader.OnImageAvailableListener() {

        @Override
        public void onImageAvailable(ImageReader reader) {

            try {

                //Logger.i("mOnImageAvailableListener", reader.toString());

//                if (PreviewCallback.processingVideo || CameraController.processStart){ // we need to convert the image to Mat before aquire another image.
//                    Logger.e(TAG, "image not processed");
//                    image = reader.acquireLatestImage();
//                    if (image != null)
//                        image.close();
//                    return;
//                }else{
//                  //  image.close();
//                    image = reader.acquireLatestImage();
//                }

                image = reader.acquireLatestImage();
                if (image != null && !isSessionClosed) {
                    if (image.getWidth() > 1920 && !CameraController.processStart){
                        Logger.e(TAG, "got stills Image");
                        new convertFrameToMat().execute(image);

                    }
                    else if (!CameraController.processStart && !PreviewCallback.processingVideo && !isSessionClosed) {
                        Logger.e(TAG, "image processed");
                        new convertFrameToMat().execute(image);

                    }
                    else {
                        image.close();
                        Logger.e(TAG, "image not processed");
                    }

                }
            }catch(Exception e){
                Logger.e(TAG,Log.getStackTraceString(e));
            }
            finally {
               // image.close();
            }

    }};


    /**
     * {@link CaptureRequest.Builder} for the camera preview
     */
    private CaptureRequest.Builder mPreviewRequestBuilder;

    /**
     * {@link CaptureRequest} generated by {@link #mPreviewRequestBuilder}
     */
    private CaptureRequest mPreviewRequest;

    /**
     * The current state of camera state for taking pictures.
     *
     * @see #mCaptureCallback
     */
    private int mState = STATE_PREVIEW;

    /**
     * A {@link Semaphore} to prevent the app from exiting before closing the camera.
     */
    private Semaphore mCameraOpenCloseLock = new Semaphore(1);

    /**
     * Whether the current camera device supports Flash or not.
     */
    private boolean mFlashSupported;

    /**
     * A {@link CameraCaptureSession.CaptureCallback} that handles events related to JPEG capture.
     */
    private CameraCaptureSession.CaptureCallback mCaptureCallback
            = new CameraCaptureSession.CaptureCallback() {

        private void process(CaptureResult result) {
            switch (mState) {
                case STATE_PREVIEW: {
                    // We have nothing to do when the camera preview is working normally.
                      processVideo();

//
//                    int afState = result.get(CaptureResult.CONTROL_AF_STATE);
//                    if (CaptureResult.CONTROL_AF_TRIGGER_START == afState) {
//                        if (§§) {
//                            processVideo();
//                        }
//                    }
//                    if (CaptureResult.CONTROL_AF_STATE_PASSIVE_FOCUSED == afState) {
//                        areWeFocused = true;
//                    } else {
//                        areWeFocused = false;
//                    }


                    break;
                }
                case STATE_WAITING_LOCK: {
                    Integer afState = result.get(CaptureResult.CONTROL_AF_STATE);
                    if (afState == null) {
                        captureStillPicture();
                    } else if (CaptureResult.CONTROL_AF_STATE_FOCUSED_LOCKED == afState ||
                            CaptureResult.CONTROL_AF_STATE_NOT_FOCUSED_LOCKED == afState) {
                        // CONTROL_AE_STATE can be null on some devices
                        Integer aeState = result.get(CaptureResult.CONTROL_AE_STATE);
                        if (aeState == null ||
                                aeState == CaptureResult.CONTROL_AE_STATE_CONVERGED) {
                            mState = STATE_PICTURE_TAKEN;
                            captureStillPicture();
                        } else {
                            runPrecaptureSequence();
                        }
                    }
                    break;
                }
                case STATE_WAITING_PRECAPTURE: {
                    // CONTROL_AE_STATE can be null on some devices
                    Integer aeState = result.get(CaptureResult.CONTROL_AE_STATE);
                    if (aeState == null ||
                            aeState == CaptureResult.CONTROL_AE_STATE_PRECAPTURE ||
                            aeState == CaptureRequest.CONTROL_AE_STATE_FLASH_REQUIRED) {
                        mState = STATE_WAITING_NON_PRECAPTURE;
                    }
                    break;
                }
                case STATE_WAITING_NON_PRECAPTURE: {
                    // CONTROL_AE_STATE can be null on some devices
                    Integer aeState = result.get(CaptureResult.CONTROL_AE_STATE);
                    if (aeState == null || aeState != CaptureResult.CONTROL_AE_STATE_PRECAPTURE) {
                        mState = STATE_PICTURE_TAKEN;
                        captureStillPicture();
                    }
                    break;
                }
            }
        }

        @Override
        public void onCaptureProgressed(CameraCaptureSession session,
                                        CaptureRequest request,
                                        CaptureResult partialResult) {
            process(partialResult);
        }

        @Override
        public void onCaptureCompleted(CameraCaptureSession session,
                                       CaptureRequest request,
                                       TotalCaptureResult result) {
            process(result);
        }

    };
    public void closeSession(){
            //stopRepeating();
        isSessionClosed = true;
        processCallback.processingVideo = false;
        stillCaptureStarted = false;
        closeCamera();
        stopBackgroundThread();

    }
    public void stopRepeating(){
        try {
            mCaptureSession.stopRepeating();
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    private void processVideo() {
        try {
            if (mPreviewRequestBuilder != null && mImageReader != null && mImageReader.getSurface() != null) {
                try {
                    List<CaptureRequest> captureList = new ArrayList<CaptureRequest>();
                    mPreviewRequestBuilder.addTarget(mImageReader.getSurface());
                    for (int i = 0; i < 10; i++) {
                        captureList.add(mPreviewRequestBuilder.build());
                        //Log.i("mPreviewRequestBuilder",String.valueOf(i));
                    }

//                mCaptureSession.stopRepeating();
                    mCaptureSession.capture(mPreviewRequestBuilder.build(), cameraCaptureCallback, mBackgroundHandler);
//                mPreviewRequestBuilder.removeTarget(mImageReader.getSurface());
                } catch (CameraAccessException e) {
                    e.printStackTrace();
                }
            }
        }catch(Exception e){

        }
    }
    int mPictureCounter = 0;
    CameraCaptureSession.CaptureCallback cameraCaptureCallback = new CameraCaptureSession.CaptureCallback() {
        @Override
        public void onCaptureCompleted(CameraCaptureSession session, CaptureRequest request,
                                       TotalCaptureResult result) {
            //Log.d("camera","saved " +  mPictureCounter);
            mPictureCounter++;
          //  if (mPictureCounter >= 10)
                //unlockFocus();
            //List<CaptureResult> cameraResult = result.getPartialResults();

        }
    };

    /**
     * Shows a {@link Toast} on the UI thread.
     *
     * @param text The message to show
     */
    private void showToast(final String text) {
        final Activity activity = CameraController.getInstance();//getActivity();
        if (activity != null) {
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    Toast.makeText(activity, text, Toast.LENGTH_SHORT).show();
                }
            });
        }
    }

    /**
     * Given {@code choices} of {@code Size}s supported by a camera, choose the smallest one that
     * is at least as large as the respective texture view size, and that is at most as large as the
     * respective max size, and whose aspect ratio matches with the specified value. If such size
     * doesn't exist, choose the largest one that is at most as large as the respective max size,
     * and whose aspect ratio matches with the specified value.
     *
     * @param choices           The list of sizes that the camera supports for the intended output
     *                          class
     * @param textureViewWidth  The width of the texture view relative to sensor coordinate
     * @param textureViewHeight The height of the texture view relative to sensor coordinate
     * @param maxWidth          The maximum width that can be chosen
     * @param maxHeight         The maximum height that can be chosen
     * @param aspectRatio       The aspect ratio
     * @return The optimal {@code Size}, or an arbitrary one if none were big enough
     */
    private static Size chooseOptimalSize(Size[] choices, int textureViewWidth,
                                          int textureViewHeight, int maxWidth, int maxHeight, Size aspectRatio) {

        // Collect the supported resolutions that are at least as big as the preview Surface
        List<Size> bigEnough = new ArrayList<>();
        // Collect the supported resolutions that are smaller than the preview Surface
        List<Size> notBigEnough = new ArrayList<>();
        int w = aspectRatio.getWidth();
        int h = aspectRatio.getHeight();
        for (Size option : choices) {
            if (option.getWidth() <= maxWidth && option.getHeight() <= maxHeight &&
                    option.getHeight() == option.getWidth() * h / w) {
                if (option.getWidth() >= textureViewWidth &&
                        option.getHeight() >= textureViewHeight) {
                    bigEnough.add(option);
                } else {
                    notBigEnough.add(option);
                }
            }
        }

        // Pick the smallest of those big enough. If there is no one big enough, pick the
        // largest of those not big enough.
        if (bigEnough.size() > 0) {
            return Collections.min(bigEnough, new CompareSizesByArea());
        } else if (notBigEnough.size() > 0) {
            return Collections.max(notBigEnough, new CompareSizesByArea());
        } else {
            Log.e(TAG, "Couldn't find any suitable preview size");
            return choices[0];
        }
    }

    public static CameraAPI2Manager newInstance() {
        return new CameraAPI2Manager();
    }
    public void startCamera(){
        mTextureView = (AutoFitTextureView) CameraController.getInstance().findViewById(R.id.texture);
        startBackgroundThread();
//
//        // When the screen is turned off and turned back on, the SurfaceTexture is already
//        // available, and "onSurfaceTextureAvailable" will not be called. In that case, we can open
//        // a camera and start preview from here (otherwise, we wait until the surface is ready in
//        // the SurfaceTextureListener).
        if (mTextureView.isAvailable()) {
            openCamera(mTextureView.getWidth(), mTextureView.getHeight());
        } else {
            mTextureView.setSurfaceTextureListener(mSurfaceTextureListener);
        }
    }

//    @Override
//    public View onCreateView(LayoutInflater inflater, ViewGroup container,
//                             Bundle savedInstanceState) {
//      //  View inflater.inflate(R.layout.mbck_camera_layout, container, false);
//
//        cameraLayout = (CameraOverlayLayout) inflater.inflate(
//                CameraManagerController.isCustomView ? R.layout.custom_mbck_camera_layout : R.layout.mbck_camera_layout, null);
//        cameraLayout.setIsTorchOn(CameraManagerController.getOcrAnalyzeSession(getActivity().getApplicationContext()).isTorchOn);
//        cameraLayout.setIsManualCapture(false);
//
//        if (CameraManagerController.getOcrAnalyzeSession(getActivity()).isBarcodeSession)
//            //cameraOverlayView.setCheckBoundariesRect(CameraManagerController.getOcrAnalyzeSession(getApplicationContext()).getBarcodeBoundariesRectDisp());
//            cameraLayout.setCheckBoundariesRect(CameraManagerController.getOcrAnalyzeSession(getActivity()).getCheckBoundariesDisp());
//        else
//            cameraLayout.setCheckBoundariesRect(CameraManagerController.getOcrAnalyzeSession(getActivity()).getCheckBoundariesDisp());
//
//       // cameraLayout.setActionClickListener(new CameraController.ActionClickListenerImpl());
//
//       //cameraLayout.setActionClickListener(new CameraController.ActionClickListenerImpl());
//        //cameraLayout.setCameraController(CameraController.getInstance());
//        CameraController.getInstance().cameraOverlayView = cameraLayout;
//        return cameraLayout;
//    }
//
//    @Override
//    public void onViewCreated(final View view, Bundle savedInstanceState) {
//     //   view.findViewById(R.id.pictureImg).setOnClickListener(this);
//       // view.findViewById(R.id.infoImg).setOnClickListener(this);
//        mTextureView = (AutoFitTextureView) view.findViewById(R.id.texture);
//    }
//
//    @Override
//    public void onActivityCreated(Bundle savedInstanceState) {
//        super.onActivityCreated(savedInstanceState);
//        mFile = new File(CameraController.getInstance().getExternalFilesDir(null), "pic.jpg");
//    }
//
//
//
//    @Override
//    public void onResume() {
//        super.onResume();
//        startBackgroundThread();
//
//        // When the screen is turned off and turned back on, the SurfaceTexture is already
//        // available, and "onSurfaceTextureAvailable" will not be called. In that case, we can open
//        // a camera and start preview from here (otherwise, we wait until the surface is ready in
//        // the SurfaceTextureListener).
//        if (mTextureView.isAvailable()) {
//            openCamera(mTextureView.getWidth(), mTextureView.getHeight());
//        } else {
//            mTextureView.setSurfaceTextureListener(mSurfaceTextureListener);
//        }
//    }

//    @Override
//    public void onPause() {
//        closeCamera();
//        stopBackgroundThread();
//        super.onPause();
//    }
    public void closeSessionAndResources(){

        closeCamera();
        stopBackgroundThread();
    }

    private void requestCameraPermission() {
//        if (FragmentCompat.shouldShowRequestPermissionRationale(this, Manifest.permission.CAMERA)) {
//            new ConfirmationDialog().show(getChildFragmentManager(), FRAGMENT_DIALOG);
//        } else {
//            FragmentCompat.requestPermissions(this, new String[]{Manifest.permission.CAMERA},
//                    REQUEST_CAMERA_PERMISSION);
//        }
    }

    //@Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions,
                                           int[] grantResults) {
//        if (requestCode == REQUEST_CAMERA_PERMISSION) {
//            if (grantResults.length != 1 || grantResults[0] != PackageManager.PERMISSION_GRANTED) {
//                ErrorDialog.newInstance(getString(R.string.request_permission))
//                        .show(getChildFragmentManager(), FRAGMENT_DIALOG);
//            }
//        } else {
//            super.onRequestPermissionsResult(requestCode, permissions, grantResults);
//        }
    }

    /**
     * Sets up member variables related to camera.
     *
     * @param width  The width of available size for camera preview
     * @param height The height of available size for camera preview
     */
    private void setUpCameraOutputs(int width, int height) {
        Activity activity = CameraController.getInstance();//getActivity();
        CameraManager manager = (CameraManager) activity.getSystemService(Context.CAMERA_SERVICE);
        try {
            for (String cameraId : manager.getCameraIdList()) {
                CameraCharacteristics characteristics
                        = manager.getCameraCharacteristics(cameraId);

                // We don't use a front facing camera in this sample.
                Integer facing = characteristics.get(CameraCharacteristics.LENS_FACING);
                if (facing != null && facing == CameraCharacteristics.LENS_FACING_FRONT) {
                    continue;
                }

                StreamConfigurationMap map = characteristics.get(
                        CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
                if (map == null) {
                    continue;
                }

                // For still image captures, we use the largest available size.
                Size stillsCaptureSize;
                Size largest = null;

                StreamConfigurationMap streamConfigurationMap = characteristics.get(
                        CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
                Size[] mSizes = streamConfigurationMap.getOutputSizes(SurfaceTexture.class);
                ArrayList<Size> validPreviewSizes = new ArrayList<Size>();
                for (Size s : mSizes) {
                    if (s.getWidth() <= 1920 && s.getHeight() <= 1080) { // get mim value above video resolution for stills.
                        validPreviewSizes.add(s);
                    }
                }
                mPreviewSize = Collections.max(
                        validPreviewSizes,
                        new CompareSizesByArea());

                if(CameraManagerController.imageType == CaptureIntent.TISDocumentType.FULL_PAGE || CameraManagerController.useMaxResolutionStills) {
                    map.getOutputSizes(ImageFormat.JPEG);
                    stillsCaptureSize = Collections.max(
                            Arrays.asList(map.getOutputSizes(ImageFormat.JPEG)),
                            new CompareSizesByArea());

                }
                else {
                    Size[] cameraSizes = map.getOutputSizes(ImageFormat.JPEG);

                    Point choosedCaptureSize  = CameraConfigurationManager.chooseCaptureResolution(cameraSizes,CameraController.getInstance());
                    stillsCaptureSize = new Size(choosedCaptureSize.x,choosedCaptureSize.y);
//                    ArrayList<Size> validSizes = new ArrayList<Size>();
//                    for (Size s : cameraSizes) {
//                        if (s.getWidth() > mPreviewSize.getWidth() && s.getHeight() > mPreviewSize.getHeight()) { // get mim value above video resolution for stills.
//                            validSizes.add(s);
//                        }
//                    }
//                    stillsCaptureSize = Collections.min(
//                            validSizes,
//                            new CompareSizesByArea());
//
//                     largest = Collections.max(
//                            Arrays.asList(map.getOutputSizes(ImageFormat.JPEG)),
//                            new CompareSizesByArea());
                }


                //CameraConfigurationManager.stillHeightRes = mPreviewSize.getHeight();
                //CameraConfigurationManager.stillWidthRes = mPreviewSize.getWidth();

                mImageReader = ImageReader.newInstance(mPreviewSize.getWidth(), mPreviewSize.getHeight(),
                        ImageFormat.YUV_420_888,3);
                mStillsImageReader = ImageReader.newInstance(stillsCaptureSize.getWidth(), stillsCaptureSize.getHeight(),
                        ImageFormat.YUV_420_888,3);
                CameraConfigurationManager.stillHeightRes =  stillsCaptureSize.getHeight();
                CameraConfigurationManager.stillWidthRes =  stillsCaptureSize.getWidth();
                CameraConfigurationManager.captureResolutionWidth = CameraConfigurationManager.stillWidthRes;
                CameraConfigurationManager.captureResolutionHeight = CameraConfigurationManager.stillHeightRes;

                CameraConfigurationManager.videoResolutionHeight =  mPreviewSize.getHeight();
                CameraConfigurationManager.videoResolutionWidth =  mPreviewSize.getWidth();

                mImageReader.setOnImageAvailableListener(
                        mOnImageAvailableListener, mBackgroundHandler);

                mStillsImageReader.setOnImageAvailableListener(
                        mOnImageAvailableListener, mBackgroundHandler);
                // Find out if we need to swap dimension to get the preview size relative to sensor
                // coordinate.
                int displayRotation = activity.getWindowManager().getDefaultDisplay().getRotation();
                int sensorOrientation =
                        characteristics.get(CameraCharacteristics.SENSOR_ORIENTATION);
                boolean swappedDimensions = false;
                switch (displayRotation) {
                    case Surface.ROTATION_0:
                    case Surface.ROTATION_180:
                        if (sensorOrientation == 90 || sensorOrientation == 270) {
                            swappedDimensions = true;
                        }
                        break;
                    case Surface.ROTATION_90:
                    case Surface.ROTATION_270:
                        if (sensorOrientation == 0 || sensorOrientation == 180) {
                            swappedDimensions = true;
                        }
                        break;
                    default:
                        Log.e(TAG, "Display rotation is invalid: " + displayRotation);
                }

                Point displaySize = new Point();
                activity.getWindowManager().getDefaultDisplay().getSize(displaySize);
                int rotatedPreviewWidth = width;
                int rotatedPreviewHeight = height;
                int maxPreviewWidth = displaySize.x;
                int maxPreviewHeight = displaySize.y;
                //CameraSessionManager.getInstance().configManager.setScreenResolution(displaySize);
                CameraConfigurationManager.screenResolution = new Point();
                CameraConfigurationManager.screenResolution.x = displaySize.x;
                CameraConfigurationManager.screenResolution.y = displaySize.y;
                if (swappedDimensions) {
                    rotatedPreviewWidth = height;
                    rotatedPreviewHeight = width;
                    maxPreviewWidth = displaySize.y;
                    maxPreviewHeight = displaySize.x;
                }

                if (maxPreviewWidth > MAX_PREVIEW_WIDTH) {
                    maxPreviewWidth = MAX_PREVIEW_WIDTH;
                }

                if (maxPreviewHeight > MAX_PREVIEW_HEIGHT) {
                    maxPreviewHeight = MAX_PREVIEW_HEIGHT;
                }

                // Danger, W.R.! Attempting to use too large a preview size could  exceed the camera
                // bus' bandwidth limitation, resulting in gorgeous previews but the storage of
                // garbage capture data.
//                mPreviewSize = chooseOptimalSize(map.getOutputSizes(SurfaceTexture.class),
//                        rotatedPreviewWidth, rotatedPreviewHeight, maxPreviewWidth,
//                        maxPreviewHeight, largest);


                // We fit the aspect ratio of TextureView to the size of preview we picked.
                int orientation = CameraController.getInstance().getResources().getConfiguration().orientation;//getResources().getConfiguration().orientation;
                if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
                    mTextureView.setAspectRatio(
                            mPreviewSize.getWidth(), mPreviewSize.getHeight());
                } else {
                    mTextureView.setAspectRatio(
                            mPreviewSize.getHeight(), mPreviewSize.getWidth());
                }

                // Check if the flash is supported.
                Boolean available = characteristics.get(CameraCharacteristics.FLASH_INFO_AVAILABLE);
                mFlashSupported = available == null ? false : available;

                mCameraId = cameraId;
                return;
            }
        } catch (CameraAccessException e) {
            e.printStackTrace();
        } catch (NullPointerException e) {
            // Currently an NPE is thrown when the Camera2API is used but not supported on the
            // device this code runs.
//            ErrorDialog.newInstance(getString(R.string.camera_error))
//                    .show(getChildFragmentManager(), FRAGMENT_DIALOG);
        }
    }

    /**
     * Opens the camera specified by {@link CameraAPI2Manager#mCameraId}.
     */
    @TargetApi(21)
    private void openCamera(int width, int height) {

        setUpCameraOutputs(width, height);
        configureTransform(width, height);
        isSessionClosed = false;
        CameraManager manager = (CameraManager) CameraController.getInstance().getSystemService(Context.CAMERA_SERVICE);
        CameraController.getInstance().getSystemService(Context.CAMERA_SERVICE);
        try {
            if (!mCameraOpenCloseLock.tryAcquire(2500, TimeUnit.MILLISECONDS)) {
                throw new RuntimeException("Time out waiting to lock camera opening.");
            }
            manager.openCamera(mCameraId, mStateCallback, mBackgroundHandler);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {

                if (CameraController.getInstance().checkSelfPermission(Manifest.permission.CAMERA) != PackageManager.PERMISSION_GRANTED) {

                    // TODO: Consider calling
                    //    Activity#requestPermissions
                    // here to request the missing permissions, and then overriding
                    //   public void onRequestPermissionsResult(int requestCode, String[] permissions,
                    //                                          int[] grantResults)
                    // to handle the case where the user grants the permission. See the documentation
                    // for Activity#requestPermissions for more details.
                }
            }

        } catch (CameraAccessException e) {
            e.printStackTrace();
        } catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while trying to lock camera opening.", e);
        }
    }

    /**
     * Closes the current {@link CameraDevice}.
     */
    private void closeCamera() {
        try {
            isSessionClosed = true;
            mCameraOpenCloseLock.acquire();
            if (null != mCaptureSession) {
                mCaptureSession.close();
                mCaptureSession = null;
            }
            if (null != mCameraDevice) {
                mCameraDevice.close();
                mCameraDevice = null;
            }
            if (null != mImageReader) {
                mImageReader.close();
                mImageReader = null;
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while trying to lock camera closing.", e);
        } finally {
            mCameraOpenCloseLock.release();
        }
    }

    /**
     * Starts a background thread and its {@link Handler}.
     */
    private void startBackgroundThread() {
        mBackgroundThread = new HandlerThread("CameraBackground");
        mBackgroundThread.start();
        mBackgroundHandler = new Handler(mBackgroundThread.getLooper());
    }

    /**
     * Stops the background thread and its {@link Handler}.
     */
    private void stopBackgroundThread() {

        if (mBackgroundThread != null) {
            mBackgroundThread.quitSafely();
            try {
                mBackgroundThread.join();
                mBackgroundThread = null;
                mBackgroundHandler = null;
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Creates a new {@link CameraCaptureSession} for camera preview.
     */
    private void createCameraPreviewSession() {
        try {
            if (mBackgroundHandler == null){
                return;
            }
            SurfaceTexture texture = mTextureView.getSurfaceTexture();
            assert texture != null;

            // We configure the size of default buffer to be the size of camera preview we want.
            texture.setDefaultBufferSize(mPreviewSize.getWidth(), mPreviewSize.getHeight());

            // This is the output Surface we need to start preview.
            Surface surface = new Surface(texture);

            // We set up a CaptureRequest.Builder with the output Surface.
            mPreviewRequestBuilder
                    = mCameraDevice.createCaptureRequest(CameraDevice.TEMPLATE_PREVIEW);
            CameraManager cameraManager = (CameraManager) CameraController.getInstance().getSystemService(Context.CAMERA_SERVICE);
            CameraCharacteristics cameraCharacteristics = cameraManager.getCameraCharacteristics(mCameraId);
            Range<Integer> fps[] = cameraCharacteristics.get(CameraCharacteristics.CONTROL_AE_AVAILABLE_TARGET_FPS_RANGES);
            //mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AE_TARGET_FPS_RANGE, fps[1]); // take second fastest frame rate
            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_VIDEO_STABILIZATION_MODE,1);

            mPreviewRequestBuilder.addTarget(surface);


            // Here, we create a CameraCaptureSession for camera preview.
            ArrayList<Surface> surfacesList =  new ArrayList<Surface>();
            surfacesList.add(mImageReader.getSurface()); // target for the video feed
            surfacesList.add(mStillsImageReader.getSurface()); // target for stills capture.
            surfacesList.add(surface);
            mCameraDevice.createCaptureSession(surfacesList,
                    new CameraCaptureSession.StateCallback() {

                        @Override
                        public void onConfigured( CameraCaptureSession cameraCaptureSession) {
                            // The camera is already closed
                            if (null == mCameraDevice) {
                                return;
                            }

                            // When the session is ready, we start displaying the preview.
                            mCaptureSession = cameraCaptureSession;

                            try {
                                // Auto focus should be continuous for camera preview.
                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE,
                                        CaptureRequest.CONTROL_AF_MODE_CONTINUOUS_PICTURE);

                           //     mPreviewRequestBuilder.set(CaptureRequest.CONTROL_MODE, CameraMetadata.CONTROL_MODE_AUTO);

//                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE,
//                                        CaptureRequest.CONTROL_AF_MODE_CONTINUOUS_PICTURE);
//
//
//
//                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_MODE, CaptureRequest.CONTROL_MODE_AUTO);
//                                // before capture lock focus
//                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER,
//                                        CaptureRequest.CONTROL_AF_TRIGGER_START);
//                                // set area for focusing
                                MeteringRectangle[] focusArea = new MeteringRectangle[1];
                                focusArea[0] = new MeteringRectangle(new Rect(0,0,CameraConfigurationManager.videoResolutionWidth,CameraConfigurationManager.videoResolutionWidth),1);
                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_REGIONS, focusArea);
//
//                                mPreviewRequestBuilder.set(
//                                        CaptureRequest.LENS_OPTICAL_STABILIZATION_MODE,
//                                        CaptureRequest.LENS_OPTICAL_STABILIZATION_MODE_ON);
//                                mPreviewRequestBuilder.set(
//                                        CaptureRequest.COLOR_CORRECTION_ABERRATION_MODE,
//                                        CaptureRequest.COLOR_CORRECTION_ABERRATION_MODE_HIGH_QUALITY);
//                                mPreviewRequestBuilder.set(CaptureRequest.NOISE_REDUCTION_MODE,
//                                        CaptureRequest.NOISE_REDUCTION_MODE_HIGH_QUALITY);


//                                mPreviewRequestBuilder.set(CaptureRequest.EDGE_MODE,
//                                        CaptureRequest.EDGE_MODE_OFF);
//                                mPreviewRequestBuilder.set(
//                                        CaptureRequest.LENS_OPTICAL_STABILIZATION_MODE,
//                                        CaptureRequest.LENS_OPTICAL_STABILIZATION_MODE_ON);
//                                mPreviewRequestBuilder.set(
//                                        CaptureRequest.COLOR_CORRECTION_ABERRATION_MODE,
//                                        CaptureRequest.COLOR_CORRECTION_ABERRATION_MODE_HIGH_QUALITY);
//                                mPreviewRequestBuilder.set(CaptureRequest.NOISE_REDUCTION_MODE,
//                                        CaptureRequest.NOISE_REDUCTION_MODE_HIGH_QUALITY);
////                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER,
////                                        CaptureRequest.CONTROL_AE_MODE);
//
//                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AE_LOCK, true);
//                                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AWB_LOCK, true);


                                // Flash is automatically enabled when necessary.
                                setAutoFlash(mPreviewRequestBuilder);

                                // Finally, we start displaying the camera preview.
                                mPreviewRequest = mPreviewRequestBuilder.build();
                                mCaptureSession.setRepeatingRequest(mPreviewRequest,
                                        mCaptureCallback, mBackgroundHandler);



                            } catch (CameraAccessException e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onConfigureFailed(
                                CameraCaptureSession cameraCaptureSession) {
                          //  showToast("Failed");
                        }
                    }, null
            );
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    /**
     * Configures the necessary {@link Matrix} transformation to `mTextureView`.
     * This method should be called after the camera preview size is determined in
     * setUpCameraOutputs and also the size of `mTextureView` is fixed.
     *
     * @param viewWidth  The width of `mTextureView`
     * @param viewHeight The height of `mTextureView`
     */
    private void configureTransform(int viewWidth, int viewHeight) {
        Activity activity = CameraController.getInstance();
        if (null == mTextureView || null == mPreviewSize || null == activity) {
            return;
        }
        int rotation = activity.getWindowManager().getDefaultDisplay().getRotation();
        Matrix matrix = new Matrix();
        RectF viewRect = new RectF(0, 0, viewWidth, viewHeight);
        RectF bufferRect = new RectF(0, 0, mPreviewSize.getHeight(), mPreviewSize.getWidth());
        float centerX = viewRect.centerX();
        float centerY = viewRect.centerY();
        if (Surface.ROTATION_90 == rotation || Surface.ROTATION_270 == rotation) {
            bufferRect.offset(centerX - bufferRect.centerX(), centerY - bufferRect.centerY());
            matrix.setRectToRect(viewRect, bufferRect, Matrix.ScaleToFit.FILL);
            float scale = Math.max(
                    (float) viewHeight / mPreviewSize.getHeight(),
                    (float) viewWidth / mPreviewSize.getWidth());
            matrix.postScale(scale, scale, centerX, centerY);
            matrix.postRotate(90 * (rotation - 2), centerX, centerY);
        } else if (Surface.ROTATION_180 == rotation) {
            matrix.postRotate(180, centerX, centerY);
        }
        mTextureView.setTransform(matrix);
    }

    /**
     * Initiate a still image capture.
     */
    public void takePicture() {
        if(!stillCaptureStarted)
            lockFocus();
    }

    /**
     * Lock the focus as the first step for a still image capture.
     */
    private void lockFocus() {
        try {

            if (mBackgroundHandler == null){
                return;
            }
            if (mCaptureSession!= null && mPreviewRequestBuilder != null) {
                // This is how to tell the camera to lock focus.
                mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER,
                        CameraMetadata.CONTROL_AF_TRIGGER_START);
                // Tell #mCaptureCallback to wait for the lock.
                mState = STATE_WAITING_LOCK;
                mCaptureSession.capture(mPreviewRequestBuilder.build(), mCaptureCallback,
                        mBackgroundHandler);
            }
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    /**
     * Run the precapture sequence for capturing a still image. This method should be called when
     * we get a response in {@link #mCaptureCallback} from {@link #lockFocus()}.
     */
    private void runPrecaptureSequence() {
        try {
            if (mBackgroundHandler == null ||mCaptureSession == null || mCaptureCallback == null || isSessionClosed ){
                return;
            }
            // This is how to tell the camera to trigger.
            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AE_PRECAPTURE_TRIGGER,
                    CaptureRequest.CONTROL_AE_PRECAPTURE_TRIGGER_START);
            // Tell #mCaptureCallback to wait for the precapture sequence to be set.
            mState = STATE_WAITING_PRECAPTURE;

            mCaptureSession.capture(mPreviewRequestBuilder.build(), mCaptureCallback,
                    mBackgroundHandler);
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    /**
     * Capture a still picture. This method should be called when we get a response in
     * {@link #mCaptureCallback} from both {@link #lockFocus()}.
     */


    private void captureStillPicture() {
        try {
            Log.e(TAG,"captureStillPicture started" );
            final Activity activity = CameraController.getInstance();
            if (null == activity || null == mCameraDevice || isSessionClosed) {
                return;
            }
            // This is the CaptureRequest.Builder that we use to take a picture.
            final CaptureRequest.Builder captureBuilder =
                    mCameraDevice.createCaptureRequest(CameraDevice.TEMPLATE_STILL_CAPTURE);
            captureBuilder.addTarget(mStillsImageReader.getSurface());

            // Use the same AE and AF modes as the preview.
            captureBuilder.set(CaptureRequest.CONTROL_AF_MODE,
                    CaptureRequest.CONTROL_AF_MODE_CONTINUOUS_PICTURE);
            setAutoFlash(captureBuilder);

            // Orientation
            int rotation = activity.getWindowManager().getDefaultDisplay().getRotation();
            captureBuilder.set(CaptureRequest.JPEG_ORIENTATION, ORIENTATIONS.get(rotation));

            CameraCaptureSession.CaptureCallback CaptureCallback
                    = new CameraCaptureSession.CaptureCallback() {

                @Override
                public void onCaptureCompleted( CameraCaptureSession session,
                                               CaptureRequest request,
                                               TotalCaptureResult result) {
//                    showToast("Saved: " + mFile);
//                    Log.d(TAG, mFile.toString());
                    unlockFocus();
                }
            };

            if (mCaptureSession!= null && !isSessionClosed) {
                mCaptureSession.stopRepeating();
                mCaptureSession.capture(captureBuilder.build(), CaptureCallback, null);
            }
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    /**
     * Unlock the focus. This method should be called when still image capture sequence is
     * finished...
     */
    public void unlockFocus() {
        try {
            if (mBackgroundHandler == null || mCaptureSession== null || isSessionClosed  ){
                return;
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                if (!mCaptureSession.isReprocessable()){ // session already closed
                    return;
                }
            }
            // Reset the auto-focus trigger
            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER,
                    CameraMetadata.CONTROL_AF_TRIGGER_CANCEL);
            setAutoFlash(mPreviewRequestBuilder);
            mCaptureSession.capture(mPreviewRequestBuilder.build(), mCaptureCallback,
                    mBackgroundHandler);
            // After this, the camera will go back to the normal state of preview.
            mState = STATE_PREVIEW;
            mCaptureSession.setRepeatingRequest(mPreviewRequest, mCaptureCallback,
                    mBackgroundHandler);
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onClick(View view) {

//            if (view.getId() == R.id.pictureImg) {
//                takePicture();
//
//            }
//            else {
//                if (view.getId() == R.id.infoImg) {
//                    Activity activity = CameraController.getInstance();
//                    if (null != activity) {
//                        new AlertDialog.Builder(activity)
//                                .setMessage(R.string.intro_message)
//                                .setPositiveButton(android.R.string.ok, null)
//                                .show();
//                    }
//                }
//            }

    }

    private void setAutoFlash(CaptureRequest.Builder requestBuilder) {
        if (mFlashSupported) {
            requestBuilder.set(CaptureRequest.CONTROL_AE_MODE,
                    CaptureRequest.CONTROL_AE_MODE_ON_AUTO_FLASH);
        }
    }

    private class convertFrameToMat extends AsyncTask<Image, Void, Mat> {

        @Override
        protected Mat doInBackground(Image... params) {
            Image img = params[0];
            try {

                convertImageToMatInProcess = true;
                PreviewCallback.processingVideo = true;
                Mat mYuvMat = imageToMat(img);
                Mat bgrMat = new Mat(img.getHeight(), img.getWidth(), CvType.CV_8UC4);
                Imgproc.cvtColor(mYuvMat, bgrMat, Imgproc.COLOR_YUV2BGR_I420);
                Mat rgbaMatOut = new Mat();
                Imgproc.cvtColor(bgrMat, rgbaMatOut, Imgproc.COLOR_BGR2RGBA, 0);
                return rgbaMatOut;
            }finally {
                if (img!= null)
                    img.close();

            }
        }

        @Override
        protected void onPostExecute(Mat mat) {
            if (mBackgroundHandler != null) {
                super.onPostExecute(mat);
             //   convertImageToMatInProcess = false;
                mBackgroundHandler.post(new ImageSaver(mat, mFile));
                convertImageToMatInProcess = false;
            }
        }


      /* (non-Javadoc)
                         * @see android.os.AsyncTask#onPreExecute()
                         */

        /* (non-Javadoc)
         * @see android.os.AsyncTask#onProgressUpdate(java.lang.Object[])
         */
        @Override
        protected void onProgressUpdate(Void... values) {
        }
    }

    /**
     * Saves a JPEG {@link Image} into the specified {@link File}.
     */
    private static class ImageSaver implements Runnable {

        /**
         * The JPEG image
         */
        private final Mat mImage;
        /**
         * The file we save the image into.
         */
        private final File mFile;

        public ImageSaver(Mat image, File file) {
            mImage = image;
            mFile = file;
        }

        @Override
        public void run() {
            ByteBuffer buffer;
            byte[] bytes;
            try{

                if (mImage != null) {
                    CameraConfigurationManager.cameraCaptureResolution = new Point(mImage.cols(),mImage.rows());
                }

                if (mImage.cols() > CameraConfigurationManager.videoResolutionWidth  || mImage.rows() > CameraConfigurationManager.videoResolutionWidth){
                   if( !CameraController.processStart && CameraController.captureStillStarted) {
                       stillCaptureStarted = true;
                       CameraSessionManager.getInstance().startStillsFlow(mImage);
                   }

                }
                else {
                    CameraSessionManager.getInstance().previewCallback.processMat(null, null, mPreviewSize.getWidth(), mPreviewSize.getHeight(), mImage);
                }



        }finally {
             //   mImage.close();
                buffer = null;
                bytes  = null;
            }
            }

    }

    /**
     * Compares two {@code Size}s based on their areas.
     */
    static class CompareSizesByArea implements Comparator<Size> {

        @Override
        public int compare(Size lhs, Size rhs) {
            // We cast here to ensure the multiplications won't overflow
            return Long.signum((long) lhs.getWidth() * lhs.getHeight() -
                    (long) rhs.getWidth() * rhs.getHeight());
        }

    }

    public static Mat imageToMat(Image image) {
        ByteBuffer buffer;
        int rowStride;
        int pixelStride;
        int width = image.getWidth();
        int height = image.getHeight();
        int offset = 0;

        Image.Plane[] planes = image.getPlanes();
        byte[] data = new byte[image.getWidth() * image.getHeight() * ImageFormat.getBitsPerPixel(ImageFormat.YUV_420_888) / 8];
        byte[] rowData = new byte[planes[0].getRowStride()];

        for (int i = 0; i < planes.length; i++) {
            buffer = planes[i].getBuffer();
            rowStride = planes[i].getRowStride();
            pixelStride = planes[i].getPixelStride();
            int w = (i == 0) ? width : width / 2;
            int h = (i == 0) ? height : height / 2;
            for (int row = 0; row < h; row++) {
                int bytesPerPixel = ImageFormat.getBitsPerPixel(ImageFormat.YUV_420_888) / 8;
                if (pixelStride == bytesPerPixel) {
                    int length = w * bytesPerPixel;
                    buffer.get(data, offset, length);

                    if (h - row != 1) {
                        buffer.position(buffer.position() + rowStride - length);
                    }
                    offset += length;
                } else {


                    if (h - row == 1) {
                        buffer.get(rowData, 0, width - pixelStride + 1);
                    } else {
                        buffer.get(rowData, 0, rowStride);
                    }

                    for (int col = 0; col < w; col++) {
                        data[offset++] = rowData[col * pixelStride];
                    }
                }
            }
        }

        Mat mat = new Mat(height + height / 2, width, CvType.CV_8UC1);
        mat.put(0, 0, data);

        return mat;
    }

    /**
     * Shows an error message dialog.
     */
    public static class ErrorDialog extends DialogFragment {

        private static final String ARG_MESSAGE = "message";

        public static ErrorDialog newInstance(String message) {
            ErrorDialog dialog = new ErrorDialog();
            Bundle args = new Bundle();
            args.putString(ARG_MESSAGE, message);
            dialog.setArguments(args);
            return dialog;
        }

        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            final Activity activity = getActivity();
            return new AlertDialog.Builder(activity)
                    .setMessage(getArguments().getString(ARG_MESSAGE))
                    .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialogInterface, int i) {
                            activity.finish();
                        }
                    })
                    .create();
        }

    }

    @TargetApi(23)
    private boolean askPermission(){

        PackageManager pm = CameraController.getInstance().getPackageManager();
        String permission = "android.permission.CAMERA";
        int res = pm.checkPermission(permission,CameraController.getInstance().getPackageName());
        if (res == PackageManager.PERMISSION_GRANTED){
            return true;
        }
        else{
            CameraController.getInstance().requestPermissions(
                    new String[]{Manifest.permission.CAMERA},
                    Constants.CAMERA_PERMISSIONS_REQUEST);
        }
        return false;
    }

    /**
     * Shows OK/Cancel confirmation dialog about camera permission.
     */
    public static class ConfirmationDialog extends DialogFragment {

        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            final Fragment parent = getParentFragment();
            return new AlertDialog.Builder(getActivity())
                    .setMessage("permission")
                    .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
//                            FragmentCompat.requestPermissions(parent,
//                                    new String[]{Manifest.permission.CAMERA},
//                                    REQUEST_CAMERA_PERMISSION);
                        }
                    })
                    .setNegativeButton(android.R.string.cancel,
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int which) {
                                    Activity activity = parent.getActivity();
                                    if (activity != null) {
                                        activity.finish();
                                    }
                                }
                            })
                    .create();
        }
    }

}
