package com.tinkerpatch.sdk.okhttp;

import com.squareup.okhttp.Call;
import com.squareup.okhttp.Callback;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;
import com.squareup.okhttp.ResponseBody;
import com.tencent.tinker.lib.util.TinkerLog;
import com.tencent.tinker.loader.shareutil.SharePatchFileUtil;
import com.tinkerpatch.sdk.server.model.DataFetcher;
import com.tinkerpatch.sdk.server.model.TinkerClientUrl;
import com.tinkerpatch.sdk.util.ContentLengthInputStream;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;



/**
 * Created by simsun{github.com/simpleton}.
 */

public class OkHttpStreamFetcher implements DataFetcher<InputStream> {

    private static final String    TAG           = "Tinker.OkHttpFetcher";
    private static       MediaType mediaTypeForm = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");

    private final TinkerClientUrl tkUrl;
    private OkHttpClient    client;

    private ResponseBody responseBody;
    private InputStream  stream;

    /**
     * @param client okhttp3 instance
     * @param tkUrl tinkerpatch request url
     */
    public OkHttpStreamFetcher(OkHttpClient client, TinkerClientUrl tkUrl) {
        this.client = client;
        this.tkUrl = tkUrl;
    }

    @Override
    public void loadData(final DataCallback<? super InputStream> callback) {
        Request.Builder requestBuilder = new Request.Builder().url(tkUrl.getStringUrl());
        switch (tkUrl.getMethod()) {
            case "GET":
                requestBuilder = requestBuilder.get();
                break;
            case "POST":
                requestBuilder = requestBuilder.post(RequestBody.create(mediaTypeForm, tkUrl.getBody()));
                break;
            default:
                throw new RuntimeException("Unsupported request Method" + tkUrl.getMethod());
        }
        for (Map.Entry<String, String> headerEntry : tkUrl.getHeaders().entrySet()) {
            String key = headerEntry.getKey();
            requestBuilder.addHeader(key, headerEntry.getValue());
        }
        final Request request = requestBuilder.build();

        Call call = client.newCall(request);
        call.enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                TinkerLog.w(TAG, "failed to obtain result", e);
                callback.onLoadFailed(e);
            }

            @Override
            public void onResponse(Response response) throws IOException {
                responseBody = response.body();
                if (responseBody != null && response.isSuccessful()) {
                    long contentLength = responseBody.contentLength();
                    TinkerLog.d(TAG, "got success response: " + response.code() + ", " + response.message());
                    stream = ContentLengthInputStream.obtain(responseBody.byteStream(), contentLength);
                    callback.onDataReady(stream);
                } else {
                    TinkerLog.w(TAG, "got error response: " + response.code() + ", " + response.message());
                    callback.onLoadFailed(new RuntimeException(String.format("Okhttp3 error:%s", response.message())));
                }
            }
        });
    }

    @Override
    public void cleanup() {
        this.client = null;
        SharePatchFileUtil.closeQuietly(stream);
        SharePatchFileUtil.closeQuietly(responseBody);
    }
}
