/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 Shengjie Sim Sun
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.tinkerpatch.sdk;

import android.content.Context;
import android.os.Looper;
import android.os.MessageQueue;

import com.tencent.tinker.lib.listener.PatchListener;
import com.tencent.tinker.lib.patch.AbstractPatch;
import com.tencent.tinker.lib.patch.UpgradePatch;
import com.tencent.tinker.lib.reporter.LoadReporter;
import com.tencent.tinker.lib.reporter.PatchReporter;
import com.tencent.tinker.lib.service.AbstractResultService;
import com.tencent.tinker.lib.tinker.Tinker;
import com.tencent.tinker.lib.tinker.TinkerInstaller;
import com.tencent.tinker.lib.util.TinkerLog;
import com.tencent.tinker.loader.TinkerRuntimeException;
import com.tencent.tinker.loader.app.ApplicationLike;
import com.tencent.tinker.loader.shareutil.ShareTinkerInternals;
import com.tinkerpatch.sdk.server.TinkerServerClient;
import com.tinkerpatch.sdk.server.callback.ConfigRequestCallback;
import com.tinkerpatch.sdk.server.callback.PatchRequestCallback;
import com.tinkerpatch.sdk.server.callback.RollbackCallBack;
import com.tinkerpatch.sdk.server.callback.TinkerPatchRequestCallback;
import com.tinkerpatch.sdk.server.utils.ServerUtils;
import com.tinkerpatch.sdk.tinker.callback.ResultCallBack;
import com.tinkerpatch.sdk.tinker.crash.TinkerServerExceptionHandler;
import com.tinkerpatch.sdk.tinker.library.TinkerPatchReflectLibrary;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerLoadReporter;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerPatchListener;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerPatchReporter;
import com.tinkerpatch.sdk.tinker.service.TinkerServerResultService;
import com.tinkerpatch.sdk.util.PermissionUtil;
import com.tinkerpatch.sdk.util.TinkerPatchContext;
import com.tinkerpatch.sdk.util.TinkerPatchUtils;
import com.tinkerpatch.sdk.util.VersionInfo;


public class TinkerPatch {
    private static final String TAG = "Tinker.TinkerPatch";
    private static volatile TinkerPatch sInstance;
    private final Tinker             tinkerClient;
    private final TinkerServerClient tinkerServerClient;
    private final ApplicationLike    applicationLike;
    /**
     * TinkerPatch 构造函数
     * @param context {@link Context}
     * @param applicationLike {@link ApplicationLike} tinker中的applicationLike
     * @param loadReporter {@link LoadReporter}
     * @param patchReporter {@link PatchReporter}
     * @param patchListener {@link PatchListener}
     * @param upgradePatch {@link AbstractPatch}
     * @param serviceClass 执行patch的service class
     * @param patchRequestCallback {@link PatchRequestCallback} 请求TinkerPatch后台的相关callback
     */
    public TinkerPatch(Context context,
                       ApplicationLike applicationLike,
                       LoadReporter loadReporter,
                       PatchReporter patchReporter,
                       PatchListener patchListener,
                       AbstractPatch upgradePatch,
                       Class<? extends AbstractResultService> serviceClass,
                       PatchRequestCallback patchRequestCallback) {
        TinkerPatchContext.setContext(context);
        this.applicationLike = applicationLike;
        //ensure can get app key and version from manifest
        if (ServerUtils.getManifestAppKey(context) == null || ServerUtils.getManifestAppVersion(context) == null) {
            TinkerLog.e(TAG, "Can't get appVersion or appKey from manifest, just disable TinkerPatch SDK");
            this.tinkerServerClient = null;
            this.tinkerClient = null;
            return;
        }
        //read version File first
        VersionInfo.init(context);
        //init exception handler
        Thread.setDefaultUncaughtExceptionHandler(new TinkerServerExceptionHandler(this.applicationLike));

        this.tinkerServerClient = TinkerServerClient.init(context, patchRequestCallback);
        this.tinkerClient = TinkerInstaller.install(applicationLike,
            loadReporter, patchReporter, patchListener,
            serviceClass, upgradePatch);
        TinkerLog.i(TAG, "Init TinkerPatch sdk success, version:%s", com.tinkerpatch.sdk.BuildConfig.VERSION_NAME);
    }

    /**
     * 设置Tinker相关Log的真正实现,用于自定义日志输出
     * @param imp {@link com.tencent.tinker.lib.util.TinkerLog.TinkerLogImp} log implementation
     */
    public static void setLogIml(TinkerLog.TinkerLogImp imp) {
        TinkerLog.setTinkerLogImp(imp);
    }

    /**
     * 用默认的构造参数初始化TinkerPatch的SDK
     *
     * @param applicationLike {@link ApplicationLike} instance
     * @return {@link TinkerPatch} instance
     */
    public static TinkerPatch init(ApplicationLike applicationLike) {
        synchronized (TinkerPatch.class) {
            if (sInstance == null) {
                synchronized (TinkerPatch.class) {
                    sInstance = new TinkerPatch.Builder(applicationLike).build();
                }
            } else {
                TinkerLog.e(TAG, "TinkerPatch instance is already set. this invoking will be ignored");
            }
        }
        return sInstance;
    }

    /**
     * 自定义参数初始化TinkerPatch的SDK
     *
     * @param tinkerPatch {@link TinkerPatch} instance
     * @return {@link TinkerPatch} instance
     */
    public static TinkerPatch init(TinkerPatch tinkerPatch) {
        if (tinkerPatch == null) {
            throw new TinkerRuntimeException("TinkerPatch init, tinkerPatch should not be null.");
        }
        synchronized (TinkerPatch.class) {
            if (sInstance == null) {
                sInstance = tinkerPatch;
            } else {
                TinkerLog.e(TAG, "TinkerPatch instance is already set. this invoking will be ignored");
            }
        }
        return sInstance;
    }

    /**
     * 获得TinkerPatch的实例
     *
     * @return {@link TinkerPatch} instance
     */
    public static TinkerPatch with() {
        if (sInstance == null) {
            throw new TinkerRuntimeException("you must init TinkerPatch sdk first");
        }
        return sInstance;
    }

    /**
     * 获得ApplicationLike的实例
     *
     * @return {@link ApplicationLike} instance
     */
    public ApplicationLike getApplcationLike() {
        if (applicationLike == null) {
            throw new TinkerRuntimeException("you must init TinkerPatch sdk first.");
        }
        return applicationLike;
    }

    /**
     * 反射补丁的Library path, 自动加载library
     * 是否自动反射Library路径,无须手动加载补丁中的So文件
     * 注意,调用在反射接口之后才能生效,你也可以使用Tinker的方式加载Library
     *
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch reflectPatchLibrary() {
        //reflect library
        final Context context = TinkerPatchContext.getContext();
        TinkerPatchReflectLibrary.attachPatchNative(context);
        return sInstance;
    }

    /**
     * 获得当前的补丁版本
     *
     * @return 当前补丁版本号。（此版本号由后台管理，且单调递增）
     */
    public Integer getPatchVersion() {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "getPatchVersion, tinkerServerClient, just return");
            return null;
        }
        return VersionInfo.with().patchVersion();
    }

    /**
     * 向后台获得动态配置,默认的访问间隔为3个小时
     * 若参数为true,即每次调用都会真正的访问后台配置
     *
     * @param configRequestCallback {@link ConfigRequestCallback} config request callback
     * @param immediately           是否立刻请求,忽略时间间隔限制
     *
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch fetchDynamicConfig(final ConfigRequestCallback configRequestCallback,
                                          final boolean immediately) {
        if (tinkerServerClient == null || tinkerClient == null) {
            TinkerLog.e(TAG, "fetchDynamicConfig, tinkerServerClient or tinkerClient is null, just return");
            return sInstance;
        }
        final Context context = TinkerPatchContext.getContext();
        if (!PermissionUtil.checkPermission(context)) {
            TinkerLog.e(TAG, "fetchDynamicConfig, permission refuse, "
                + "you must access INTERNET and ACCESS_NETWORK_STATE permission first");
            return sInstance;
        }
        //only check at the main process
        if (tinkerClient.isMainProcess()) {
            Looper.getMainLooper().myQueue().addIdleHandler(new MessageQueue.IdleHandler() {
                @Override
                public boolean queueIdle() {
                    tinkerServerClient.fetchDynamicConfig(configRequestCallback, immediately);
                    return false;
                }
            });
        }
        return sInstance;
    }

    /**
     * 向后台获取是否有补丁包更新,默认的访问间隔为3个小时
     * 若参数为true,即每次调用都会真正的访问后台配置
     *
     * @param immediately 是否立刻检查,忽略时间间隔限制
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch fetchPatchUpdate(final boolean immediately) {
        if (tinkerServerClient == null || tinkerClient == null) {
            TinkerLog.e(TAG, "fetchPatchUpdate, tinkerServerClient or tinkerClient is null, just return");
            return sInstance;
        }
        final Context context = TinkerPatchContext.getContext();
        if (!PermissionUtil.checkPermission(context)) {
            TinkerLog.e(TAG, "fetchPatchUpdate, permission refuse, "
                + "you must access INTERNET and ACCESS_NETWORK_STATE permission first");
            return sInstance;
        }
        if (!tinkerClient.isTinkerEnabled() || !ShareTinkerInternals.isTinkerEnableWithSharedPreferences(context)) {
            TinkerLog.e(TAG, "fetchPatchUpdate, tinker is disable, just return");
            return sInstance;
        }
        //only check at the main process
        if (tinkerClient.isMainProcess()) {
            Looper.getMainLooper().myQueue().addIdleHandler(new MessageQueue.IdleHandler() {
                @Override
                public boolean queueIdle() {
                    tinkerServerClient.fetchPatchUpdate(immediately);
                    return false;
                }
            });
        }
        return sInstance;
    }

    /**
     * 启动tinkerPatch，开始同步patch
     */
    public TinkerPatch fetchPatchUpdateAndPollWithInterval() {
        if (tinkerServerClient == null || tinkerClient == null) {
            TinkerLog.e(TAG, "fetchPatchUpdateWithIntervalPoll, "
                + "tinkerServerClient or tinkerClient is null, just return");
            return sInstance;
        }
        tinkerServerClient.fetchPatchWithInterval();
        return sInstance;
    }

    /**
     * 设置当前渠道号,对于某些渠道我们可能会想屏蔽补丁功能
     * 设置渠道后,我们就可以使用后台的条件控制渠道更新
     *
     * @param channel channel name
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setAppChannel(String channel) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setAppChannel, tinkerServerClient == null, just return");
            return sInstance;
        }
        TinkerPatchUtils.setAppChannel(channel);
        tinkerServerClient.updatePatchCondition(TinkerServerClient.CONDITION_CHANNEL, channel);
        return sInstance;
    }

    /**
     * 屏蔽部分渠道的补丁功能
     *
     * @param channel channel name
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch addIgnoreAppChannel(String channel) {
        TinkerPatchUtils.addAppIgnoreChannel(channel);
        return sInstance;
    }

    /**
     * 设置tinkerpatch平台的条件下发参数
     * 默认内置的条件有[wifi, sdk, brand, model, cpu, cpu]
     * 若调用了setAppChannel, 能增加[channel]条件
     *
     * @param key key
     * @param value value
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setPatchCondition(String key, String value) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setPatchCondition, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.updatePatchCondition(key, value);
        return sInstance;
    }

    /**
     * 设置访问后台动态配置的时间间隔,默认为3个小时
     *
     * @param hours interval hours
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setFetchDynamicConfigIntervalByHours(int hours) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setFetchDynamicConfigIntervalByHours, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.setFetchDynamicConfigIntervalByHours(hours);
        return sInstance;
    }

    /**
     * 设置访问后台补丁包更新配置的时间间隔,默认为3个小时
     *
     * @param hours interval hours
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setFetchPatchIntervalByHours(int hours) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setFetchPatchIntervalByHours, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.setFetchPatchIntervalByHours(hours);
        return sInstance;
    }

    /**
     * 设置补丁合成成功后,是否通过锁屏重启程序,这样可以加快补丁的生效时间
     * 默认为false, 即等待应用自身重新启动时加载
     *
     * @param restartOnScreenOff enable screen off restart flag
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setPatchRestartOnSrceenOff(boolean restartOnScreenOff) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerServerResultService.setRestartOnScreenOff(restartOnScreenOff);
        return sInstance;
    }

    /**
     * 我们可以通过ResultCallBack设置对合成后的回调
     * 例如我们也可以不锁屏,而是在这里通过弹框咨询用户等方式
     *
     * @param resultCallBack {@link ResultCallBack} result callback
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setPatchResultCallback(ResultCallBack resultCallBack) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerServerResultService.setResultCallBack(resultCallBack);
        return sInstance;
    }

    /**
     * 设置收到后台回退要求时,是否在锁屏时清除补丁
     * 默认为false,即等待应用下一次重新启动时才会去清除补丁
     *
     * @param rollbackOnScreenOff rollback on screen off flag
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setPatchRollbackOnScreenOff(boolean rollbackOnScreenOff) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRollbackOnScreenOff, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerPatchRequestCallback.setRollbackOnScreenOff(rollbackOnScreenOff);
        return sInstance;
    }

    /**
     * 我们可以通过RollbackCallBack设置对回退时的回调
     *
     * @param rollbackCallBack {@link RollbackCallBack} rollback callback
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch setPatchRollBackCallback(RollbackCallBack rollbackCallBack) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerPatchRequestCallback.setPatchRollbackCallBack(rollbackCallBack);
        return sInstance;
    }

    /**
     * 清除所有信息, 包括补丁、缓存的版本信息和动态配置
     *
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch cleanAll() {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "cleanPatch, tinkerClient is null, just return");
            return sInstance;
        }
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "cleanPatch, tinkerServerClient is null, just return");
            return sInstance;
        }

        tinkerServerClient.cleanLocalStore();
        tinkerClient.cleanPatch();
        return sInstance;
    }

    /**
     * 仅清除Tinker的补丁文件
     *
     * @return {@link TinkerPatch} instance
     */
    public TinkerPatch cleanPatch() {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "cleanPatch, tinkerClient is null, just return");
            return sInstance;
        }

        tinkerClient.cleanPatch();
        return sInstance;
    }

    public static class Builder {
        private final Context         context;
        private final ApplicationLike applicationLike;

        private LoadReporter                           loadReporter;
        private PatchReporter                          patchReporter;
        private PatchListener                          listener;
        private AbstractPatch                          upgradePatch;
        private Class<? extends AbstractResultService> serviceClass;

        private PatchRequestCallback patchRequestCallback;

        /**
         * Start building a new {@link TinkerPatch} instance.
         */
        public Builder(ApplicationLike applicationLike) {
            if (applicationLike == null) {
                throw new TinkerRuntimeException("applicationLike must not be null.");
            }

            this.context = applicationLike.getApplication();
            this.applicationLike = applicationLike;
        }

        public TinkerPatch.Builder loadReporter(LoadReporter loadReporter) {
            if (loadReporter == null) {
                throw new TinkerRuntimeException("loadReporter must not be null.");
            }
            if (this.loadReporter != null) {
                throw new TinkerRuntimeException("loadReporter is already set.");
            }
            this.loadReporter = loadReporter;
            return this;
        }

        public TinkerPatch.Builder patchReporter(PatchReporter patchReporter) {
            if (patchReporter == null) {
                throw new TinkerRuntimeException("patchReporter must not be null.");
            }
            if (this.patchReporter != null) {
                throw new TinkerRuntimeException("patchReporter is already set.");
            }
            this.patchReporter = patchReporter;
            return this;
        }

        public TinkerPatch.Builder listener(PatchListener listener) {
            if (listener == null) {
                throw new TinkerRuntimeException("listener must not be null.");
            }
            if (this.listener != null) {
                throw new TinkerRuntimeException("listener is already set.");
            }
            this.listener = listener;
            return this;
        }

        public TinkerPatch.Builder upgradePatch(AbstractPatch upgradePatch) {
            if (upgradePatch == null) {
                throw new TinkerRuntimeException("upgradePatch must not be null.");
            }
            if (this.upgradePatch != null) {
                throw new TinkerRuntimeException("upgradePatch is already set.");
            }
            this.upgradePatch = upgradePatch;
            return this;
        }

        public TinkerPatch.Builder resultServiceClass(Class<? extends AbstractResultService> resultServiceClass) {
            if (resultServiceClass == null) {
                throw new TinkerRuntimeException("resultServiceClass must not be null.");
            }
            if (this.serviceClass != null) {
                throw new TinkerRuntimeException("resultServiceClass is already set.");
            }
            this.serviceClass = resultServiceClass;
            return this;
        }

        public TinkerPatch.Builder patchRequestCallback(PatchRequestCallback patchRequestCallback) {
            if (patchRequestCallback == null) {
                throw new TinkerRuntimeException("patchRequestCallback must not be null.");
            }
            if (this.patchRequestCallback != null) {
                throw new TinkerRuntimeException("patchRequestCallback is already set.");
            }
            this.patchRequestCallback = patchRequestCallback;
            return this;
        }

        public TinkerPatch build() {
            if (loadReporter == null) {
                loadReporter = new TinkerServerLoadReporter(context);
            }

            if (patchReporter == null) {
                patchReporter = new TinkerServerPatchReporter(context);
            }

            if (listener == null) {
                listener = new TinkerServerPatchListener(context);
            }

            if (upgradePatch == null) {
                upgradePatch = new UpgradePatch();
            }

            if (serviceClass == null) {
                serviceClass = TinkerServerResultService.class;
            }

            if (patchRequestCallback == null) {
                patchRequestCallback = new TinkerPatchRequestCallback();
            }

            return new TinkerPatch(context, applicationLike, loadReporter, patchReporter, listener, upgradePatch,
                serviceClass, patchRequestCallback);
        }
    }
}
