/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2013-2016 Shengjie Sim Sun
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.tinkerpatch.sdk;

import android.content.Context;
import android.os.Looper;
import android.os.MessageQueue;

import com.tencent.tinker.lib.listener.PatchListener;
import com.tencent.tinker.lib.patch.AbstractPatch;
import com.tencent.tinker.lib.patch.UpgradePatch;
import com.tencent.tinker.lib.reporter.LoadReporter;
import com.tencent.tinker.lib.reporter.PatchReporter;
import com.tencent.tinker.lib.service.AbstractResultService;
import com.tencent.tinker.lib.tinker.Tinker;
import com.tencent.tinker.lib.tinker.TinkerInstaller;
import com.tencent.tinker.lib.util.TinkerLog;
import com.tencent.tinker.loader.TinkerRuntimeException;
import com.tencent.tinker.loader.app.ApplicationLike;
import com.tencent.tinker.loader.shareutil.ShareTinkerInternals;
import com.tinkerpatch.sdk.server.TinkerServerClient;
import com.tinkerpatch.sdk.server.callback.ConfigRequestCallback;
import com.tinkerpatch.sdk.server.callback.PatchRequestCallback;
import com.tinkerpatch.sdk.server.callback.RollbackCallBack;
import com.tinkerpatch.sdk.server.callback.TinkerPatchRequestCallback;
import com.tinkerpatch.sdk.server.utils.ServerUtils;
import com.tinkerpatch.sdk.tinker.callback.ResultCallBack;
import com.tinkerpatch.sdk.tinker.crash.TinkerServerExceptionHandler;
import com.tinkerpatch.sdk.tinker.library.TinkerPatchReflectLibrary;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerLoadReporter;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerPatchListener;
import com.tinkerpatch.sdk.tinker.reporter.TinkerServerPatchReporter;
import com.tinkerpatch.sdk.tinker.service.TinkerServerResultService;
import com.tinkerpatch.sdk.util.TinkerPatchUtils;
import com.tinkerpatch.sdk.util.VersionInfo;

/**
 * Created by shwenzhang on 16/12/7.
 */

public class TinkerPatch {
    private static final String TAG = "Tinker.TinkerPatch";
    private static TinkerPatch sInstance;

    private final Tinker             tinkerClient;
    private final TinkerServerClient tinkerServerClient;
    private final ApplicationLike    applicationLike;

    private final Context context;

    public TinkerPatch(Context context, ApplicationLike applicationLike,
                       LoadReporter loadReporter, PatchReporter patchReporter,
                       PatchListener patchListener, AbstractPatch upgradePatch,
                       Class<? extends AbstractResultService> serviceClass,
                       PatchRequestCallback patchRequestCallback) {
        this.context = context;
        this.applicationLike = applicationLike;
        //ensure can get app key and version from manifest
        if (ServerUtils.getManifestAppKey(context) == null
            || ServerUtils.getManifestAppVersion(context) == null) {
            TinkerLog.e(TAG, "can get appVersion or appKey from manifest, just disable TinkerPatch SDK");
            this.tinkerServerClient = null;
            this.tinkerClient = null;
            return;
        }
        //read version File first
        VersionInfo.init(context);
        //init exception handler
        Thread.setDefaultUncaughtExceptionHandler(new TinkerServerExceptionHandler(this.applicationLike));

        this.tinkerServerClient = TinkerServerClient.init(this.context, patchRequestCallback);
        this.tinkerClient = TinkerInstaller.install(applicationLike,
            loadReporter, patchReporter, patchListener,
            serviceClass, upgradePatch);
    }

    public static void setLogIml(TinkerLog.TinkerLogImp imp) {
        TinkerLog.setTinkerLogImp(imp);
    }

    public static TinkerPatch init(ApplicationLike applicationLike) {
        if (sInstance != null) {
            throw new TinkerRuntimeException("TinkerPatch instance is already set.");
        }
        sInstance = new TinkerPatch.Builder(applicationLike).build();
        return sInstance;
    }

    public static TinkerPatch init(TinkerPatch tinkerPatch) {
        if (tinkerPatch == null) {
            throw new TinkerRuntimeException("TinkerPatch init, tinkerPatch should not be null.");
        }
        if (sInstance != null) {
            throw new TinkerRuntimeException("TinkerPatch instance is already set.");
        }
        sInstance = tinkerPatch;
        return sInstance;
    }

    public static TinkerPatch with() {
        if (sInstance == null) {
            throw new TinkerRuntimeException("you must init TinkerPatch sdk first");
        }
        return sInstance;
    }

    public ApplicationLike getApplcationLike() {
        if (applicationLike == null) {
            throw new TinkerRuntimeException("you must init TinkerPatch sdk first.");
        }
        return applicationLike;
    }

    /**
     * 反射补丁的Library path, 自动加载library
     * @return
     */
    public TinkerPatch reflectPatchLibrary() {
        //reflect library
        TinkerPatchReflectLibrary.attachPatchNative(context);
        return sInstance;
    }

    /**
     * 向服务器请求在线参数信息
     *
     * @param configRequestCallback
     * @param immediately           是否立刻请求,忽略时间间隔限制
     */
    public TinkerPatch fetchDynamicConfig(final ConfigRequestCallback configRequestCallback,
                                          final boolean immediately) {
        if (tinkerServerClient == null || tinkerClient == null) {
            TinkerLog.e(TAG, "fetchDynamicConfig, tinkerServerClient or tinkerClient is null, just return");
            return sInstance;
        }
        //only check at the main process
        if (tinkerClient.isMainProcess()) {
            Looper.getMainLooper().myQueue().addIdleHandler(new MessageQueue.IdleHandler() {
                @Override
                public boolean queueIdle() {
                    tinkerServerClient.fetchDynamicConfig(configRequestCallback, immediately);
                    return false;
                }
            });
        }
        return sInstance;
    }

    /**
     * 检查服务器是否有补丁更新
     *
     * @param immediately 是否立刻检查,忽略时间间隔限制
     */
    public TinkerPatch fetchPatchUpdate(final boolean immediately) {
        if (tinkerServerClient == null || tinkerClient == null) {
            TinkerLog.e(TAG, "fetchPatchUpdate, tinkerServerClient or tinkerClient is null, just return");
            return sInstance;
        }
        if (!tinkerClient.isTinkerEnabled() || !ShareTinkerInternals.isTinkerEnableWithSharedPreferences(context)) {
            TinkerLog.e(TAG, "fetchPatchUpdate, tinker is disable, just return");
            return sInstance;
        }
        //only check at the main process
        if (tinkerClient.isMainProcess()) {
            Looper.getMainLooper().myQueue().addIdleHandler(new MessageQueue.IdleHandler() {
                @Override
                public boolean queueIdle() {
                    tinkerServerClient.fetchPatchUpdate(immediately);
                    return false;
                }
            });
        }
        return sInstance;
    }

    public TinkerPatch setAppChannel(String channel) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setAppChannel, tinkerServerClient == null, just return");
            return sInstance;
        }
        TinkerPatchUtils.setAppChannel(channel);
        tinkerServerClient.updatePatchCondition(TinkerServerClient.CONDITION_CHANNEL, channel);
        return sInstance;
    }

    public TinkerPatch addIgnoreAppChannel(String channel) {
        TinkerPatchUtils.addAppIgnoreChannel(channel);
        return sInstance;
    }

    /**
     * 设置条件下发的属性
     *
     * @param key
     * @param value
     */
    public TinkerPatch setPatchCondition(String key, String value) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setPatchCondition, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.updatePatchCondition(key, value);
        return sInstance;
    }

    public TinkerPatch setFetchDynamicConfigIntervalByHours(int hours) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setFetchDynamicConfigIntervalByHours, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.setFetchDynamicConfigIntervalByHours(hours);
        return sInstance;
    }

    public TinkerPatch setFetchPatchIntervalByHours(int hours) {
        if (tinkerServerClient == null) {
            TinkerLog.e(TAG, "setFetchPatchIntervalByHours, tinkerServerClient == null, just return");
            return sInstance;
        }
        tinkerServerClient.setFetchPatchIntervalByHours(hours);
        return sInstance;
    }

    public TinkerPatch setPatchRestartOnSrceenOff(boolean restartOnScreenOff) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerServerResultService.setRestartOnScreenOff(restartOnScreenOff);
        return sInstance;
    }

    public TinkerPatch setPatchResultCallback(ResultCallBack resultCallBack) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerServerResultService.setResultCallBack(resultCallBack);
        return sInstance;
    }

    public TinkerPatch setPatchRollbackOnScreenOff(boolean rollbackOnScreenOff) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRollbackOnScreenOff, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerPatchRequestCallback.setRollbackOnScreenOff(rollbackOnScreenOff);
        return sInstance;
    }

    public TinkerPatch setPatchRollBackCallback(RollbackCallBack rollbackCallBack) {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "setPatchRestartConfig, tinkerClient is null, just return");
            return sInstance;
        }
        TinkerPatchRequestCallback.setPatchRollbackCallBack(rollbackCallBack);
        return sInstance;
    }

    public TinkerPatch cleanPatch() {
        if (tinkerClient == null) {
            TinkerLog.e(TAG, "cleanPatch, tinkerClient is null, just return");
            return sInstance;
        }
        tinkerClient.cleanPatch();
        return sInstance;
    }

    public static class Builder {
        private final Context         context;
        private final ApplicationLike applicationLike;

        private LoadReporter                           loadReporter;
        private PatchReporter                          patchReporter;
        private PatchListener                          listener;
        private AbstractPatch                          upgradePatch;
        private Class<? extends AbstractResultService> serviceClass;

        private PatchRequestCallback patchRequestCallback;

        /**
         * Start building a new {@link TinkerPatch} instance.
         */
        public Builder(ApplicationLike applicationLike) {
            if (applicationLike == null) {
                throw new TinkerRuntimeException("applicationLike must not be null.");
            }

            this.context = applicationLike.getApplication();
            this.applicationLike = applicationLike;
        }

        public TinkerPatch.Builder loadReporter(LoadReporter loadReporter) {
            if (loadReporter == null) {
                throw new TinkerRuntimeException("loadReporter must not be null.");
            }
            if (this.loadReporter != null) {
                throw new TinkerRuntimeException("loadReporter is already set.");
            }
            this.loadReporter = loadReporter;
            return this;
        }

        public TinkerPatch.Builder patchReporter(PatchReporter patchReporter) {
            if (patchReporter == null) {
                throw new TinkerRuntimeException("patchReporter must not be null.");
            }
            if (this.patchReporter != null) {
                throw new TinkerRuntimeException("patchReporter is already set.");
            }
            this.patchReporter = patchReporter;
            return this;
        }

        public TinkerPatch.Builder listener(PatchListener listener) {
            if (listener == null) {
                throw new TinkerRuntimeException("listener must not be null.");
            }
            if (this.listener != null) {
                throw new TinkerRuntimeException("listener is already set.");
            }
            this.listener = listener;
            return this;
        }

        public TinkerPatch.Builder upgradePatch(AbstractPatch upgradePatch) {
            if (upgradePatch == null) {
                throw new TinkerRuntimeException("upgradePatch must not be null.");
            }
            if (this.upgradePatch != null) {
                throw new TinkerRuntimeException("upgradePatch is already set.");
            }
            this.upgradePatch = upgradePatch;
            return this;
        }

        public TinkerPatch.Builder resultServiceClass(Class<? extends AbstractResultService> resultServiceClass) {
            if (resultServiceClass == null) {
                throw new TinkerRuntimeException("resultServiceClass must not be null.");
            }
            if (this.serviceClass != null) {
                throw new TinkerRuntimeException("resultServiceClass is already set.");
            }
            this.serviceClass = resultServiceClass;
            return this;
        }

        public TinkerPatch.Builder patchRequestCallback(PatchRequestCallback patchRequestCallback) {
            if (patchRequestCallback == null) {
                throw new TinkerRuntimeException("patchRequestCallback must not be null.");
            }
            if (this.patchRequestCallback != null) {
                throw new TinkerRuntimeException("patchRequestCallback is already set.");
            }
            this.patchRequestCallback = patchRequestCallback;
            return this;
        }

        public TinkerPatch build() {
            if (loadReporter == null) {
                loadReporter = new TinkerServerLoadReporter(context);
            }

            if (patchReporter == null) {
                patchReporter = new TinkerServerPatchReporter(context);
            }

            if (listener == null) {
                listener = new TinkerServerPatchListener(context);
            }

            if (upgradePatch == null) {
                upgradePatch = new UpgradePatch();
            }

            if (serviceClass == null) {
                serviceClass = TinkerServerResultService.class;
            }

            if (patchRequestCallback == null) {
                patchRequestCallback = new TinkerPatchRequestCallback();
            }

            return new TinkerPatch(context, applicationLike, loadReporter, patchReporter, listener, upgradePatch,
                serviceClass, patchRequestCallback);
        }
    }

}
