package com.terracotta.management.security;

import java.io.Console;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;

/**
 * Utility methods useful for doing secret conversion from byte[] to char[] and back while
 * wiping out from the memory the previous password instance.
 *
 * @author Ludovic Orban
 */
public class SecretUtils {

  private static final Charset UTF8_CHARSET = Charset.forName("UTF-8");

  /**
   * Transform an array of chars containing a secret into an array of bytes, then
   * replace all characters of the char[] with 0 (character zero).
   *
   * @param secret the char[] secret.
   * @return the byte[] secret
   */
  public static byte[] toBytesAndWipe(char[] secret) {
    if (secret == null) {
      return null;
    }
    ByteBuffer source = UTF8_CHARSET.encode(CharBuffer.wrap(secret));
    byte[] array = new byte[source.remaining()];
    System.arraycopy(source.array(), 0, array, 0, array.length);
    Arrays.fill(secret, (char)0);
    return array;
  }

  /**
   * Transform an array of bytes containing a secret into an array of chars, then
   * replace all bytes of the byte[] with 0 (byte zero).
   *
   * @param secret the byte[] secret.
   * @return the char[] secret
   */
  public static char[] toCharsAndWipe(byte[] secret) {
    if (secret == null) {
      return null;
    }
    CharBuffer source = UTF8_CHARSET.decode(ByteBuffer.wrap(secret));
    char[] array = new char[source.remaining()];
    System.arraycopy(source.array(), 0, array, 0, array.length);
    Arrays.fill(secret, (byte)0);
    return array;
  }

  /**
   * Fetch a secret from the console.
   *
   * @param prompt The prompt to be displayed on the console.
   * @return the secret as a byte[].
   * @throws IllegalStateException thrown if a console instance cannot be found.
   */
  public static byte[] fetchSecretFromConsole(String prompt) throws IllegalStateException {
    char[] secretChars = fetchPasswordFromConsole(prompt);
    return toBytesAndWipe(secretChars);
  }

  /**
   * Fetch a password from the console.
   *
   * @param prompt The prompt to be displayed on the console.
   * @return the password as a char[].
   * @throws IllegalStateException thrown if a console instance cannot be found.
   */
  public static char[] fetchPasswordFromConsole(String prompt) throws IllegalStateException {
    Console console = System.console();
    if (console == null) {
      throw new IllegalStateException("Couldn't access a Console instance to fetch the password from!");
    }
    return console.readPassword(prompt);
  }

}
