package com.terracotta.management.security;

import java.util.concurrent.atomic.AtomicReference;

/**
 * @author Alex Snaps
 */
public class SecretProvider {

  private static final AtomicReference<SecretProviderBackEnd> backEnd = new AtomicReference<SecretProviderBackEnd>();

  public static final String CUSTOM_IMPL_PROP = "com.tc.SecretProvider.impl";

  public static byte[] getSecret() {
    return getBackEnd().getSecret();
  }

  @SuppressWarnings("unchecked")
  public static void fetchSecret() {
    SecretProviderBackEnd secretProviderBackEnd = backEnd.get();
    if(secretProviderBackEnd == null) {
      final String implClass = System.getProperty(CUSTOM_IMPL_PROP);
      final Class<? extends SecretProviderBackEnd> clazz;
      if (implClass != null) {
        try {
          clazz = (Class<? extends SecretProviderBackEnd>) Class.forName(implClass);
          secretProviderBackEnd = clazz.newInstance();
        } catch (ClassNotFoundException e) {
          throw new RuntimeException("Couldn't load class '" + implClass +"'", e);
        } catch (Exception e) {
          throw new RuntimeException("Couldn't instantiate SecretProvider's backing implementation '"
                                     + implClass + "'", e);
        }
      } else {
        secretProviderBackEnd = new ConsoleFetchingBackEnd();
      }
      backEnd.compareAndSet(null, secretProviderBackEnd);
    }
    getBackEnd().fetchSecret();
  }

  public static boolean fetchSecret(SecretProviderBackEnd back) {
    final boolean installed = backEnd.compareAndSet(null, back);
    if(installed) {
      getBackEnd().fetchSecret();
    }
    return installed;
  }

  static SecretProviderBackEnd getBackEnd() {
    return backEnd.get();
  }

  private static class ConsoleFetchingBackEnd implements SecretProviderBackEnd {

    private byte[] secret;

    @Override
    public byte[] getSecret() {
      return secret;
    }

    @Override
    public void fetchSecret() {
      secret = System.getProperty("SecretProvider.secret") == null ?
          null : ((String) System.getProperties().get("SecretProvider.secret")).getBytes();

      if (secret == null) {
        secret = SecretUtils.fetchSecretFromConsole("Enter secret to key store: ");
      }
    }


  }
}
