package com.terracotta.management.keychain;

import com.terracotta.management.keychain.crypto.AesEnigmaMachine;
import com.terracotta.management.keychain.crypto.EnigmaMachine;
import com.terracotta.management.keychain.crypto.SecretMismatchException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.channels.FileLock;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * @author Alex Snaps
 */
public class FileStoreKeyChain implements KeyChain {

  private final EnigmaMachine enigmaMachine;
  private final File file;

  private final Lock readLock;
  private final Lock writeLock;

  {
    ReadWriteLock lock = new ReentrantReadWriteLock();
    readLock = lock.readLock();
    writeLock = lock.writeLock();
  }

  private volatile Map<KeyName, byte[]> store;
  private byte[] masterKey;

  public static FileStoreKeyChain createNewKeyStore(final EnigmaMachine enigmaMachine, File file) throws IOException {
      return createNewKeyStore(enigmaMachine, file, null);
  }
  
  public static FileStoreKeyChain createNewKeyStore(final EnigmaMachine enigmaMachine, File file, byte[] key) throws IOException {
    if(file.getParentFile() != null && !file.getParentFile().mkdirs() && !file.getParentFile().exists()) {
      throw new IOException("Couldn't create the parent directory/ies: " + file.getParentFile().getAbsolutePath());
    }
    if (!file.createNewFile()) {
      throw new IllegalStateException("File exists already!");
    }
    final FileStoreKeyChain fileStoreKeyChain = new FileStoreKeyChain(enigmaMachine, file);
    if(key !=  null){
      fileStoreKeyChain.masterKey = key;
      fileStoreKeyChain.writeStoreToFile();
      fileStoreKeyChain.lock();
    }
    return fileStoreKeyChain;
  }

  public FileStoreKeyChain(URL url) {
    this(new AesEnigmaMachine(), new File(url.getFile()));
  }

  public FileStoreKeyChain(final EnigmaMachine enigmaMachine, final File file) {
    this.enigmaMachine = enigmaMachine;
    if (!file.exists() || !file.canRead() || !file.isFile()) {
      throw new IllegalArgumentException(file + " doesn't point to a valid file");
    }
    this.file = file;
  }

  @Override
  public byte[] getPassword(final byte[] key, final KeyName entryName) {
    readLock.lock();
    try {
      verifyUnlocked();
      final byte[] crypted = store.get(entryName);
      return crypted != null ? enigmaMachine.decrypt(key, crypted) : null;
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public boolean storePassword(final byte[] key, final KeyName entryName, final byte[] password) {
    writeLock.lock();
    try {
      verifyUnlocked();
      final byte[] previous = store.put(entryName, enigmaMachine.encrypt(key, password));
      writeStoreToFile();
      return previous == null;
    } catch (IOException e) {
      resetInMemoryData();
      throw new RuntimeException("Couldn't write to file " + file, e);
    } finally {
      writeLock.unlock();
    }
  }

  @Override
  public boolean removePassword(final KeyName entryName) {
    writeLock.lock();
    try {
      verifyUnlocked();
      final boolean removed = store.remove(entryName) != null;
      writeStoreToFile();
      return removed;
    } catch (IOException e) {
      resetInMemoryData();
      throw new RuntimeException("Couldn't write to file " + file, e);
    } finally {
      writeLock.unlock();
    }
  }

  private void resetInMemoryData() {
    byte[] masterKey = this.masterKey;
    lock();
    unlock(masterKey);
  }

  @Override
  public void lock() {
    writeLock.lock();
    try {
      masterKey = null;
      store = null;
    } finally {
      writeLock.unlock();
    }
  }

  @Override
  public void unlock(final byte[] key) {
    writeLock.lock();
    try {
      masterKey = key;
      if (file.length() > 0) {
        store = readStoreFromFile();
        if(store == null){
            //we tried to read from the file, but there is no store in it yet
            store = new HashMap<KeyName, byte[]>();
        }
      } else {
        store = new HashMap<KeyName, byte[]>();
      }
    } catch (SecretMismatchException smex) {
      throw smex;
    } catch (Exception e) {
      throw new RuntimeException("Couldn't read from file " + file, e);
    } finally {
      writeLock.unlock();
    }
  }

  public Set<KeyName> keys() {
    readLock.lock();
    try {
      verifyUnlocked();
      return store.keySet();
    } finally {
      readLock.unlock();
    }
  }

  public byte[] getPassword(KeyName entryName) {
    return getPassword(masterKey, entryName);
  }

  private Map<KeyName, byte[]> readStoreFromFile() throws ClassNotFoundException, IOException {
    byte[] fileContent;

    // file must be opened read/write or getChannel().lock() throws java.nio.channels.NonWritableChannelException
    RandomAccessFile raf = new RandomAccessFile(file, "rw");
    try {
      FileLock lock = raf.getChannel().lock();
      try {
        fileContent = new byte[(int) raf.length()];
        raf.readFully(fileContent);
      } finally {
        lock.release();
      }
    } finally {
      raf.close();
    }

    byte[] bytes = enigmaMachine.decrypt(masterKey, fileContent);
    return deserializeStore(bytes);
  }

  @SuppressWarnings("unchecked")
  private Map<KeyName, byte[]> deserializeStore(byte[] bytes) throws IOException, ClassNotFoundException {
    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bytes));
    try {
      return (Map<KeyName, byte[]>) ois.readObject();
    } finally {
      ois.close();
    }
  }

  private void writeStoreToFile() throws IOException {
    FileOutputStream fos = new FileOutputStream(file);
    try {
      FileLock lock = fos.getChannel().lock();
      try {
        fos.write(enigmaMachine.encrypt(masterKey, serializeStore()));
      } finally {
        lock.release();
      }
    } finally {
      fos.close();
    }
  }

  private byte[] serializeStore() throws IOException {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    ObjectOutputStream oos = new ObjectOutputStream(baos);
    try {
      oos.writeObject(store);
    } finally {
      oos.close();
    }
    return baos.toByteArray();
  }

  private void verifyUnlocked() {
    if (store == null || masterKey == null) {
      throw new IllegalStateException("Store is still locked! You need to unlock it first...");
    }
  }
}


