/*
 * All content copyright (c) Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.keychain;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Implements rules for standardizing URIs to be used a ke
 * @author jhouse
 *
 */
public class URIKeyName implements KeyName {

  private static final long serialVersionUID = 8278719757373245391L;
  
  private URI theURI;
  
  public URIKeyName(String uri) throws URISyntaxException {
    theURI = standardizeURI(new URI(uri));
  }
  
  public URIKeyName(URI uri) {
    theURI = standardizeURI(uri);
  }
  
  public String toString() {
    return theURI.toString();
  }
  
  private URI standardizeURI(URI uri) {
    String scheme = uri.getScheme();
    String host = uri.getHost();
    int port = uri.getPort();
    String path = uri.getPath();
    String query = uri.getQuery();
    
    
    if(scheme != null && (scheme.equals("http") || scheme.equals("https"))) {
      if(host.equals("127.0.0.1")) {
        host = "localhost";
      }
      
      if(path != null && path.endsWith("/")) {
        path = path.substring(0, path.length() -1);
      }

      if(path == null || path.length() == 0) {
        path = "/";
      }
        
      if(port < 1) {
        port = (scheme.equals("http")) ? 80 : 443;
      }

      try {
        if(query == null || query.length() == 0) {
          return new URI(scheme + "://" + host + ":" + port + path);
        }

        return new URI(scheme + "://" + host + ":" + port + path + "?" + query);
        
      } catch (URISyntaxException e) {
        throw new RuntimeException("Invalid URI unexpectedly generated!", e);
      }
    }
    
    return uri;
  }

  public URI getURI() {
    return theURI;
  }
  
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((theURI == null) ? 0 : theURI.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    URIKeyName other = (URIKeyName) obj;
    if (theURI == null) {
      if (other.theURI != null)
        return false;
    } else if (!theURI.equals(other.theURI))
      return false;
    return true;
  }
}
