package com.terracotta.management.keychain;


/**
 * A persistent store, that keeps URI to password.
 * Passwords are kept in encrypted form, and never kept decrypted in memory.
 * The store (optionally) supports lock/unlocking mechanism.
 * @author Alex Snaps
 */
public interface KeyChain {

  /**
   * Retrieves the decrypted password for the URI from the KeyChain
   * <p>To make this call the KeyChain needs to be {@link KeyChain#unlock(byte[]) unlock'ed}
   * @param key the key to decrypt the password
   * @param entryName the entry name for which we're querying the password
   * @return the decrypted password, or null if no matching record found
   * @throws IllegalStateException If the KeyChain is still locked
   */
  byte[] getPassword(byte[] key, KeyName entryName);

  /**
   * Stores the password in encrypted form in the keyChain
   * <p>To make this call the KeyChain needs to be {@link KeyChain#unlock(byte[]) unlock'ed}
   * @param key the key to encrypt the password
   * @param entryName the entry name for the password
   * @param password the decrypted password
   * @return true if this was an addition, false if an update to an existing entry
   * @throws IllegalStateException If the KeyChain is still locked
   */
  boolean storePassword(byte[] key, KeyName entryName, byte[] password);

  /**
   * Removes a password from the chain
   * <p>To make this call the KeyChain needs to be {@link KeyChain#unlock(byte[]) unlock'ed}
   * @param entryName the entry name for the password we want to delete
   * @return true if successful, false otherwise
   * @throws IllegalStateException If the KeyChain is still locked
   */
  boolean removePassword(KeyName entryName);

  /**
   * Locks the KeyChain. Removing all entries from memory...
   */
  void lock();

  /**
   * Unlocks the KeyChain, while this might load all entries in memories, passwords themselves will remain
   * in encrypted form. Decryption only happens when querying the KeyChain
   * @param key The key to unlock the store
   */
  void unlock(byte[] key);
}
