/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.dao;

import java.io.Serializable;

/**
 * A generic dao interface for crud operations on model objects in the management application.
 *
 * @author brandony
 *
 */
public interface GenericDao<T, ID extends Serializable>
{
  /**
   * Returns the obj T with the given id, or null if there is no such entity.
   *
   * @param id the id of the entity to return
   * @return T or null
   * @throws DataAccessException
   */
  T getById(ID id) throws DataAccessException;

  /**
   * Create the submitted obj.
   *
   * @param data obj to be created
   * @throws DataAccessException
   */
  void create(T data) throws DataAccessException;

  /**
   * Create or update the submitted obj.
   *
   * @param data the obj to create or update.
   * @throws DataAccessException
   */
  void createOrUpdate(T data) throws DataAccessException;

  /**
   * Deletes the submitted obj
   *
   * @param data obj to delete
   * @throws DataAccessException
   */
  void delete(T data) throws DataAccessException;

  /**
   * Synchronizes persistence layer's state with the datastore.
   */
  void flush();

  /**
   * Evicts the obj from persistent cache.
   *
   * @param data the obj to evict
   */
  void evict(T data);
}