package com.terracotta.management.security.shiro.realm;

import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.ldap.JndiLdapRealm;
import org.apache.shiro.realm.ldap.LdapContextFactory;
import org.apache.shiro.realm.ldap.LdapUtils;
import org.apache.shiro.subject.PrincipalCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapContext;
import java.util.*;
import java.util.Map.Entry;

/**
 * @author Anthony Dahanne
 */
public class LdapRealm extends JndiLdapRealm {


  private static final String GROUPDN_SUBSTITUTION_TOKEN = "{0}";

  private static final Logger log = LoggerFactory.getLogger(LdapRealm.class);

  private static final String OPERATOR = "operator";
  private static final String ADMIN = "admin";


  private String groupAttributeMatching;

  private boolean dynamicGroupConfiguration;

  private String groupDnTemplate;

  protected static final String ROLE_NAMES_DELIMETER = ",";

  /**
   * the map contains the groups, each set contains the roles for the group :
   *
   * Map<String,Set<String>> groupRolesMap = new HashMap<String, Set<String>>();
   * Set<String> dudesRoles = new HashSet<String>();
   * dudesRoles.add("admin");
   * groupRolesMap.put("dudes", dudesRoles);
   * Set<String> guysRoles = new HashSet<String>();
   * guysRoles.add("admin");
   * guysRoles.add("operator");
   * groupRolesMap.put("guys", guysRoles);
   *
   */
  protected Map<String,Set<String>> groupRolesMap;

  protected String searchBase;



  /**
   * Builds an {@link org.apache.shiro.authz.AuthorizationInfo} object by querying the active directory LDAP context for the groups that a user is a
   * member of. The groups are then translated to role names by using the configured {@link #groupRolesMap}.
   * <p/>
   * This implementation expects the <tt>principal</tt> argument to be a String username.
   * <p/>
   * Subclasses can override this method to determine authorization data (roles, permissions, etc) in a more complex way. Note that this default
   * implementation does not support permissions, only roles.
   * 
   * @param principals
   *          the principal of the Subject whose account is being retrieved.
   * @param ldapContextFactory
   *          the factory used to create LDAP connections.
   * @return the AuthorizationInfo for the given Subject principal.
   * @throws NamingException
   *           if an error occurs when searching the LDAP server.
   */
  @Override
  protected AuthorizationInfo queryForAuthorizationInfo(PrincipalCollection principals, LdapContextFactory ldapContextFactory) throws NamingException {

    String username = (String) getAvailablePrincipal(principals);

    // Perform context search
    LdapContext ldapContext = ldapContextFactory.getSystemLdapContext();

    Set<String> roleNames;

    try {
      roleNames = getRoleNamesForUser(username, ldapContext);
    } finally {
      LdapUtils.closeContext(ldapContext);
    }

    return buildAuthorizationInfo(roleNames);
  }

  protected AuthorizationInfo buildAuthorizationInfo(Set<String> roleNames) {
    return new SimpleAuthorizationInfo(roleNames);
  }

  protected Set<String> getRoleNamesForUser(String username, LdapContext ldapContext) throws NamingException {
    Set<String> roleNames;
    roleNames = new LinkedHashSet<String>();

    if (dynamicGroupConfiguration) {
      // dynamic group matching
      SearchControls searchCtls = new SearchControls();
      searchCtls.setSearchScope(SearchControls.SUBTREE_SCOPE);

      // SHIRO-115 - prevent potential code injection:
      String searchFilter = "(&(objectClass=*)("+getUserDnPrefix()+"{0}))";
      Object[] searchArguments = new Object[] { username };
      NamingEnumeration answer = ldapContext.search(searchBase, searchFilter, searchArguments, searchCtls);

      while (answer.hasMoreElements()) {
        SearchResult sr = (SearchResult) answer.next();

        if (log.isDebugEnabled()) {
          log.debug("Retrieving group names for user [" + sr.getName() + "]");
        }

        Attributes attrs = sr.getAttributes();

        if (attrs != null) {
          NamingEnumeration ae = attrs.getAll();
          while (ae.hasMore()) {
            Attribute attr = (Attribute) ae.next();

            if (attr.getID().equalsIgnoreCase(groupAttributeMatching)) {

              Collection<String> groupNames = LdapUtils.getAllAttributeValues(attr);

              if (log.isDebugEnabled()) {
                log.debug("Groups found for user [" + username + "]: " + groupNames);
              }

              Collection<String> rolesForGroups = getRoleNamesForGroups(groupNames);
              roleNames.addAll(rolesForGroups);
            }
          }
        }
      }

    } else {
      //static groups
      Map<String,Set<String>> groupAndSubGroupsRolesMap = getGroupAndSubGroupRolesMap(ldapContext);
      for (Entry<String, Set<String>> groupRoleEntry : groupAndSubGroupsRolesMap.entrySet()) {

        // we fetch the group entry
        Attributes attributes = ldapContext.getAttributes(groupRoleEntry.getKey());
        NamingEnumeration<? extends Attribute> all = attributes.getAll();
        //we iterate through all its fields
        while (all.hasMore()) {
          Attribute next = all.next();
          // the field is a field where there are members of the group configured
          if (isAMemberAttribute(next)) {
            Collection<String> allAttributeValues = LdapUtils.getAllAttributeValues(next);
            for (String attributeValues : allAttributeValues) {
              if (attributeValues.equals(getUserDn(username))) {
                // one of these members is the current user, he has the roles associated to this group
                for (String role : groupRoleEntry.getValue()) {
                  roleNames.add(role);
                }
              }
            }
          }
        }
      }
    }

    return roleNames;
  }

  protected Collection<String> getRoleNamesForGroups(Collection<String> groupNames) {
    Set<String> roleNames = new HashSet<String>(groupNames.size());

    if (groupRolesMap != null) {
      for (String groupName : groupNames) {
        Set<String> rolesForGroup = groupRolesMap.get(groupName);
        if (rolesForGroup != null) {
          for (String roleName : rolesForGroup) {

            if (log.isDebugEnabled()) {
              log.debug("User is member of group [" + groupName + "] so adding role [" + roleName + "]");
            }

            roleNames.add(roleName);

          }
        }
      }
    }
    return roleNames;
  }

  /**
   * From a groupRolesMap in a short form, return a groupRolesMap ina  full DN form
   * ex : Group will be translated to CN=Group,OU=Company,DC=MyDomain,DC=local
   * according to the groupDnTemplate
   *
   * @param groupRolesMap
   * @return complete groupRolesMap
   */
  private Map<String, Set<String>> getGroupRolesMapDn(Map<String, Set<String>> groupRolesMap) {
    Map<String, Set<String>> groupRolesMapWithDn =  new HashMap<String,Set<String>>();
    for (Entry<String, Set<String>> groupRolesEntry : groupRolesMap.entrySet()) {
      String simpleGroupName = groupRolesEntry.getKey();
      String groupNameWithDn = getGroupDn(simpleGroupName);
      groupRolesMapWithDn.put(groupNameWithDn,groupRolesEntry.getValue());
    }
    return groupRolesMapWithDn;
  }

  private String getGroupDn(String groupRolesMapKey) {
    return groupDnTemplate.replace(GROUPDN_SUBSTITUTION_TOKEN, groupRolesMapKey);
  }

  /**
   * The admin must be an operator too
   * so if we find the admin role, we add the operator role too
   *
   * @param roleNames thr current roles names
   */
  private void fixRoleNamesForTMS(Set<String> roleNames) {
    if (roleNames.contains(ADMIN)) {
      if (log.isDebugEnabled()) {
        log.debug("User has role [admin] so adding role [operator]");
      }
      roleNames.add(OPERATOR);
    }

  }

  private boolean isAMemberAttribute(Attribute attribute) {
    return attribute.getID().equals(groupAttributeMatching);
  }

  /**
   *
   * For each group, we look for a subgroup (that will inherit the roles of the parent group)
   * and we add it to the new map of groupRoles
   *
   * @param ldapContext
   * @return
   */
  private Map<String,Set<String>> getGroupAndSubGroupRolesMap(LdapContext ldapContext) {
    Map<String,Set<String>>  groupAndSubGroupsRolesMap =  new HashMap<String,Set<String>>();
    SearchControls searchCtls = new SearchControls();
    searchCtls.setSearchScope(SearchControls.SUBTREE_SCOPE);
    
    for (Entry<String, Set<String>> groupRoleEntry : getGroupRolesMapDn(groupRolesMap).entrySet()) {
      groupAndSubGroupsRolesMap.put(groupRoleEntry.getKey(), groupRoleEntry.getValue());
      try {
        NamingEnumeration<SearchResult> enumeration = ldapContext.search(groupRoleEntry.getKey(), "(objectClass=*)", searchCtls);
        while (enumeration.hasMoreElements()) {
          SearchResult sr = enumeration.next();

          Attributes attrs = sr.getAttributes();

          if (attrs != null) {
            NamingEnumeration ae = attrs.getAll();
            while (ae.hasMore()) {
              Attribute attr = (Attribute) ae.next();
              if(isAMemberAttribute(attr)) {
                groupAndSubGroupsRolesMap.put(sr.getNameInNamespace(), groupRoleEntry.getValue());
              }
            }
          }
            }
      } catch (NamingException e) {
        log.error("Impossible to search the group : "+groupRoleEntry.getKey(), e);
      }
      
      
    }
    return groupAndSubGroupsRolesMap;
  }

  public void setGroupAttributeMatching(String groupAttributeMatching) {
    this.groupAttributeMatching = groupAttributeMatching;
  }

  public void setDynamicGroupConfiguration(boolean dynamicGroupConfiguration) {
    this.dynamicGroupConfiguration = dynamicGroupConfiguration;
  }

  public void setGroupRolesMap(Map<String,Set<String>> groupRolesMap) {
    this.groupRolesMap = groupRolesMap;
  }

  public void setGroupRolesMapAsString(Map<String,String> groupRolesMap) {
    this.groupRolesMap =  new HashMap<String, Set<String>>();
    for (Entry<String, String> entry : groupRolesMap.entrySet()) {
      Set<String> rolesForGroup = getRolesFromString(entry.getValue());
      this.groupRolesMap.put(entry.getKey(), rolesForGroup);
    }
  }

  private Set<String> getRolesFromString(String rolesInString) {
    String[] splitRoles = rolesInString.split(",");
    Set<String> roles = new HashSet<String>();
    for (String role : splitRoles) {
      roles.add(role.trim());
    }
    return roles;
  }

  public void setSearchBase(String searchBase) {
    this.searchBase = searchBase;
  }

  public String getGroupDnTemplate() {
    return groupDnTemplate;
  }

  public void setGroupDnTemplate(String groupDnTemplate) {
    this.groupDnTemplate = groupDnTemplate;
  }

  /**
   * This method replaces calls to TCJndiLdapContextFactory.setSystemUsername to
   * allow the user to configure the LdapContext leveraging the userDnTemplate;
   * ie : ludovic instead of uid=ludovic,ou=users,dc=mycompany,dc=com
   *
   * @param systemUsername the "simple" system username
   */
  public void setSystemUsername(String systemUsername) {
    ((TCJndiLdapContextFactory) getContextFactory()).setSystemUsername(getUserDn(systemUsername));
    ((TCJndiLdapContextFactory) getContextFactory()).setSimpleSystemUsername(systemUsername);
  }

}
