/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.user;

import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

/**
 * An enumeration describing what are currently <em>implicitly</em> defined user roles for the application.
 *
 * @author brandony
 */
public enum UserRole {
  ADMIN("admin"), OPERATOR("operator"), TERRACOTTA("terracotta");

  private final String roleName;

  /**
   * A convenience method for getting a {@code UserRole} by its role name.
   *
   * @param roleName to use for lookup
   * @return the associated {@code UserRole}
   */
  public static UserRole byName(String roleName) {
    UserRole returnVal = null;
    for (UserRole role : EnumSet.allOf(UserRole.class)) {
      if (roleName.equals(role.toString())) returnVal = role;
    }

    return returnVal;
  }

  /**
   * A convenience method for converting a collection of role names to a Set of {@code UserRole} enums.
   *
   * @param roleNames to convert
   * @return the {@code Set} of {@code UserRole}s
   */
  public static Set<UserRole> convertRoleNames(Collection<String> roleNames) {
    Set<UserRole> roles = new HashSet<UserRole>(roleNames.size());

    for (String roleName : roleNames) {
      UserRole role = byName(roleName);
      if (role != null) roles.add(role);
    }

    return roles;
  }

  /**
   * A convenience method for converting a collection of {@code UserRole}s to a Set of role names.
   *
   * @param roles to convert
   * @return the {@code Set} of role names
   */
  public static Set<String> convertRoles(Collection<UserRole> roles) {
    Set<String> roleNames = new HashSet<String>(roles.size());

    for(UserRole role : roles) {
      roleNames.add(role.toString());
    }

    return roleNames;
  }

  private UserRole(String roleName) {
    this.roleName = roleName;
  }

  @Override
  public String toString() {
    return roleName;
  }
}
