/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.user.dao.impl;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.user.UserInfo;
import com.terracotta.management.user.UserRole;
import com.terracotta.management.user.dao.DatastoreNotFoundException;
import com.terracotta.management.user.dao.UserInfoDao;
import com.terracotta.management.user.impl.DfltUserInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.terracotta.management.security.shiro.ShiroIniFileConstants.DFLT_INI_FILE_LOCATION;
import static com.terracotta.management.security.shiro.ShiroIniFileConstants.JVM_INI_LOCATION_PROP;

/**
 * An implementation of {@link UserInfoDao} that uses a file in the Shiro Ini format as a datastore.
 * <p/>
 * By default this dao implementation will try and load and persist its data from a ini file location
 * provided via the JVM property {@code com.tc.management.security.ini}. If that JVM property is not set, it will look
 * to {@code $&#123;user.home&#125;/.tc/mgmt/security.ini}.
 *
 * @author brandony
 */
public final class IniFileUserInfoDao implements UserInfoDao {
  private static final String USR_HEADER = "[users]";

  private static final Pattern INVALID_USRNAME_CHARS = Pattern.compile("[=]");

  private static final Logger LOG = LoggerFactory.getLogger(IniFileUserInfoDao.class);

  private static final String LINE_SEP = System.getProperty("line.separator");

  private final File iniFile;

  public IniFileUserInfoDao() throws DataAccessException {
    this(new File(System.getProperty(JVM_INI_LOCATION_PROP) == null ? DFLT_INI_FILE_LOCATION : System
        .getProperty(JVM_INI_LOCATION_PROP)));
  }

  /**
   * A constructor that allows the ini file location used to be explicitly set.
   *
   * @param iniFile the file holding security data
   */
  public IniFileUserInfoDao(File iniFile) throws DataAccessException {
    this(iniFile, true);
  }

  /**
   * A constructor that allows the ini file location used to be explicitly set.
   *
   * @param iniFile    the file holding security data
   * @param createFile will create the file, if not already existent
   */
  public IniFileUserInfoDao(File iniFile,
                            boolean createFile) throws DataAccessException {
    this.iniFile = iniFile;

    if (createFile && !this.iniFile.exists()) initIniFile();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserInfo getById(final String username) throws DataAccessException {
    final AtomicReference<UserInfo> usrRef = new AtomicReference<UserInfo>(null);

    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {

      @Override
      void doWork() throws DataAccessException, IOException {
        long cSize = fileChannel.size();
        ByteBuffer mbb = ByteBuffer.allocate((int) cSize);
        fileChannel.read(mbb);
        mbb.flip();

        int p = findUserPosition(username, mbb);

        if (p > -1) {
          mbb.position(p);
          ByteArrayOutputStream baos = new ByteArrayOutputStream();

          byte b;
          while ((b = mbb.get()) != LINE_SEP.charAt(0)) {
            baos.write(b);
          }

          usrRef.set(buildUserInfo(new String(baos.toByteArray())));
        }
      }

      @Override
      String getWorkDescription() {
        return String.format("Retrieving user with id '%s'", username);
      }

    };

    task.execute();

    return usrRef.get();
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void create(final UserInfo user) throws DataAccessException {
    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {
      @Override
      void doWork() throws DataAccessException, IOException {
        long fcSize = fileChannel.size();
        ByteBuffer mbb = ByteBuffer.allocate((int) fcSize);
        fileChannel.read(mbb);
        mbb.flip();

        if (findUserPosition(user.getUsername(), mbb) <= -1) {
          String line = buildNewUserLine(user);
          fileChannel.position(fcSize);
          ByteBuffer bb = ByteBuffer.wrap(line.getBytes());

          while (bb.hasRemaining()) fileChannel.write(bb);
        }
      }

      @Override
      String getWorkDescription() {
        return String.format("Create user %s", user);
      }
    };

    task.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createOrUpdate(final UserInfo user) throws DataAccessException {
    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {
      @Override
      void doWork() throws DataAccessException, IOException {
        update(fileChannel, user, true);
      }

      @Override
      String getWorkDescription() {
        return String.format("Creating or updating user %s", user);
      }
    };

    task.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void delete(final UserInfo user) throws DataAccessException {
    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {
      @Override
      void doWork() throws DataAccessException, IOException {
        update(fileChannel, user, false);
      }

      @Override
      String getWorkDescription() {
        return String.format("Deleting user %s", user);
      }
    };

    task.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void flush() {
    throw new UnsupportedOperationException();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void evict(UserInfo user) {
    throw new UnsupportedOperationException();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasUserInfos() throws DataAccessException {
    final AtomicBoolean available = new AtomicBoolean(false);

    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {
      @Override
      void doWork() throws DataAccessException, IOException {
        long fcSize = fileChannel.size();
        ByteBuffer mbb = ByteBuffer.allocate((int) fcSize);
        fileChannel.read(mbb);
        mbb.flip();

        if (mbb.compareTo(ByteBuffer.wrap((USR_HEADER + LINE_SEP).getBytes())) > 0) {
          available.set(true);
        }
      }

      @Override
      String getWorkDescription() {
        return String.format("Determining whether datastore has any UserInfo objects available");
      }
    };

    task.execute();

    return available.get();
  }

  @Override
  public void truncate() throws DataAccessException {
    SynchronizedFileLockingTask task = new SynchronizedFileLockingTask() {
      @Override
      void doWork() throws DataAccessException, IOException {
        fileChannel.truncate((long) (USR_HEADER + LINE_SEP).getBytes().length);
      }

      @Override
      String getWorkDescription() {
        return String.format("Truncating UserInfo datastore");
      }
    };

    task.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void validate(boolean establish) throws DataAccessException {
    if (!iniFile.exists()) {
      if (establish) initIniFile();
      else throw new DatastoreNotFoundException();
    }
  }

  private static String buildNewUserLine(UserInfo userInfo) throws DataAccessException {
    Matcher m = INVALID_USRNAME_CHARS.matcher(userInfo.getUsername());
    if (m.find()) throw new DataAccessException(
        String.format("Invalid username '%s' detected! Unable to update datastore.", userInfo.getUsername()));

    StringBuilder sb = new StringBuilder(userInfo.getUsername());
    sb.append("=").append(userInfo.getPasswordHash());

    Set<UserRole> roles = userInfo.getRoles();
    if (roles != null) {
      Iterator<UserRole> rItr = roles.iterator();
      do {
        sb.append(",").append(rItr.next().toString());
      } while (rItr.hasNext());
    }
    sb.append(LINE_SEP);
    return sb.toString();
  }

  private static UserInfo buildUserInfo(String iniEntry) {
    String[] components = iniEntry.split(",");

    String[] credentials = components[0].split("=", 2);

    Set<UserRole> roles = null;

    for (int i = 1; i < components.length; i++) {
      if (roles == null) roles = new HashSet<UserRole>();
      roles.add(UserRole.byName(trimToNull(components[i])));
    }

    return new DfltUserInfo(trimToNull(credentials[0]), trimToNull(credentials[1]), roles);
  }

  private void initIniFile() throws DataAccessException {
    File parent = iniFile.getParentFile();
    if (parent != null) parent.mkdirs();

    try {
      File temp = File.createTempFile("tmp", UUID.randomUUID().toString(), parent);

      try {
        BufferedWriter writer = new BufferedWriter(new FileWriter(temp, true));
        try {
          writer.write(USR_HEADER + LINE_SEP);
        } finally {
          try {
            writer.close();
          } catch (IOException e) {
            LOG.warn("Failed to close FileWriter creating new security ini file.");
          }
        }

        if (!iniFile.exists()) {
          temp.renameTo(iniFile);
        } else {
          temp.delete();
        }
      } catch (IOException e) {
        if (temp.exists()) temp.delete();
        throw new DataAccessException(String.format("Failure writing new security ini file."), e);
      }

    } catch (IOException e) {
      throw new DataAccessException("Failed to create ini file.", e);
    }
  }

  private void update(FileChannel fc,
                      UserInfo user,
                      boolean replace) throws IOException, DataAccessException {
    long cSize = fc.size();
    ByteBuffer mbb = ByteBuffer.allocate((int) cSize);
    fc.read(mbb);
    mbb.flip();
    int p = findUserPosition(user.getUsername(), mbb);

    mbb.position(p == -1 ? 0 : p);

    UserInfo thisUser = null;
    if (p > -1) {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();

      byte b;
      while ((b = mbb.get()) != LINE_SEP.charAt(0)) {
        baos.write(b);
      }

      thisUser = buildUserInfo(new String(baos.toByteArray()));
    }

    byte[] tail = null;
    if (replace || user.equals(thisUser)) {

      if (p > -1) {
        int end = mbb.position();

        tail = new byte[(int) cSize - end];

        mbb.get(tail);

        fc.truncate(p);
        fc.position(fc.size());
      } else {
        fc.position(cSize);
      }

      if (replace) {
        ByteBuffer bbReplace = ByteBuffer.wrap(buildNewUserLine(user).getBytes());
        while (bbReplace.hasRemaining()) fc.write(bbReplace);
      }

      if (tail != null) {
        ByteBuffer bbTail = ByteBuffer.wrap(tail);
        while (bbTail.hasRemaining()) fc.write(bbTail);
      }
    }
  }

  private int findUserPosition(String username,
                               ByteBuffer buffer) throws IOException {
    int pos = -1;

    byte[] matchMe = (username + "=").getBytes();

    byte[] forMatch = null;
    int count = 0;
    while (buffer.hasRemaining()) {
      byte b = buffer.get();

      if (b == LINE_SEP.charAt(0)) {
        forMatch = new byte[matchMe.length];

        int lsl = LINE_SEP.length();
        if (lsl > 1) {
          for (; lsl > 1; lsl--) buffer.get();
        }
      } else if (count == matchMe.length) {
        if (Arrays.equals(matchMe, forMatch)) {
          pos = buffer.position() - matchMe.length - 1;
          break;
        } else {
          forMatch = null;
          count = 0;
        }
      } else if (forMatch != null) {
        forMatch[count++] = b;
      }
    }

    return pos;
  }

  /**
   * <p>A convenience method to prevent adding some massive dependency like commons lang.</p>
   *
   * @param string to trim
   * @return trimmed string or {@code null}
   */
  private static String trimToNull(String string) {
    return string == null || string.trim().length() == 0 ? null : string.trim();
  }

  /**
   * An abstract inner class to allow a task block (a closure approximation) to be executed with synchronization to
   * prevent race conditions and work collisions from multiple threads in the same JVM and file locking to attempt to
   * prevent the same issues across different processes/JVMS.
   *
   * @author brandony
   */
  private abstract class SynchronizedFileLockingTask {
    private final Logger TASK_LOG = LoggerFactory.getLogger(SynchronizedFileLockingTask.class);

    protected FileChannel fileChannel;

    /**
     * Execute the task.
     *
     * @throws DataAccessException if task execution fails
     */
    public synchronized void execute() throws DataAccessException {
      try {
        RandomAccessFile raf = new RandomAccessFile(iniFile, "rw");
        fileChannel = raf.getChannel();

        try {
          FileLock lock = fileChannel.lock();
          try {
            doWork();
          } finally {
            try {
              lock.release();
            } catch (IOException e) {
              TASK_LOG.warn(String.format("[%s]: Failure to release lock.", getWorkDescription()));
            }
          }
        } finally {
          try {
            fileChannel.close();
          } catch (IOException e) {
            TASK_LOG.warn(String.format("[%s]: Failure to close FileChannel.", getWorkDescription()));
          }

          try {
            raf.close();
          } catch (IOException e) {
            TASK_LOG.warn(String.format("[%s]: Failure to close IniFile.", getWorkDescription()));
          }
        }

      } catch (IOException e) {
        throw new DataAccessException(String.format("[%s]: Failed to handle request.", getWorkDescription()), e);
      }
    }

    /**
     * An abstract method to be implemented with the work for this task.
     *
     * @throws DataAccessException if the work block fails to execute successfully
     * @throws IOException         if operations on the underlying ini file fail
     */
    abstract void doWork() throws DataAccessException, IOException;

    /**
     * An abstract method to be implemented with a description of the work being done.
     *
     * @return the work description
     */
    abstract String getWorkDescription();
  }
}
