/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.impl;

import com.terracotta.management.keychain.KeyName;
import com.terracotta.management.keychain.URIKeyName;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.SSLContextFactory;
import com.terracotta.management.security.SecretUtils;
import org.terracotta.management.resource.services.Utils;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

/**
 * @author brandony
 */
public final class TMCStoresSSLContextFactory implements SSLContextFactory {
  private static final String JVM_ID_LOCATION_PROP = "javax.net.ssl.keyStore";

  private static final String JVM_ID_PASSWD_PROP = "javax.net.ssl.keyStorePassword";

  private static final String JVM_TRUST_LOCATION_PROP = "javax.net.ssl.trustStore";

  private static final String JVM_TRUST_PASSWD_PROP = "javax.net.ssl.trustStorePassword";

  private static final String SS_PROTOCOL = "TLS";

  private static final String FILE_URI_PREFIX = "file://";

  private static final String SECURE_RNDM_ALG = "SHA1PRNG";

  private final KeyChainAccessor keyChainAccessor;

  private final String idStoreLocation;

  private final String trustStoreLocation;

  private final boolean usingClientAuth;

  public TMCStoresSSLContextFactory(KeyChainAccessor keyChainAccessor,
                                    String dfltIdStoreLocation,
                                    String dfltTrustStoreLocation) {
    this(keyChainAccessor, dfltIdStoreLocation, dfltTrustStoreLocation, false);
  }

  public TMCStoresSSLContextFactory(KeyChainAccessor keyChainAccessor,
                                    String dfltIdStoreLocation,
                                    String dfltTrustStoreLocation,
                                    boolean usingClientAuth){
    this.keyChainAccessor = keyChainAccessor;
    this.idStoreLocation = System.getProperty(JVM_ID_LOCATION_PROP) == null ? dfltIdStoreLocation : System
        .getProperty(JVM_ID_LOCATION_PROP);
    this.trustStoreLocation = System.getProperty(JVM_TRUST_LOCATION_PROP) == null ? dfltTrustStoreLocation : System
        .getProperty(JVM_TRUST_LOCATION_PROP);
    this.usingClientAuth = usingClientAuth;
  }

  public SSLContext create()
      throws NoSuchAlgorithmException, IOException, KeyStoreException, CertificateException, UnrecoverableKeyException,
      KeyManagementException, URISyntaxException {
    KeyManagerFactory kmf = prepareKeyManagerFactory();
    TrustManagerFactory tmf = prepareTrustManagerFactory();

    TrustManager[] trustManagers;
    if (Boolean.getBoolean("tc.ssl.trustAllCerts")) {
      trustManagers = new TrustManager[] {
          new X509TrustManager() {
            @Override
            public X509Certificate[] getAcceptedIssuers() {
              return null;
            }
            @Override
            public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            }
            @Override
            public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            }
          }
      };
    } else {
      trustManagers = (tmf == null) ? null : tmf.getTrustManagers();
    }

    SSLContext sslCtxt = SSLContext.getInstance(SS_PROTOCOL);
    sslCtxt.init(kmf == null ? null : kmf.getKeyManagers(), trustManagers, SecureRandom.getInstance(SECURE_RNDM_ALG));
    return sslCtxt;
  }

  @Override
  public boolean isUsingClientAuth() {
    return usingClientAuth;
  }


  private KeyManagerFactory prepareKeyManagerFactory()
      throws NoSuchAlgorithmException, KeyStoreException, IOException, CertificateException, UnrecoverableKeyException,
      URISyntaxException {
    KeyManagerFactory kmf = null;

    if (idStoreLocation != null) {
      File idStoreFile = new File(idStoreLocation);
      if (idStoreFile.exists()) {
        char[] idStorePasswd = locateKeyStorePassword(JVM_ID_PASSWD_PROP, idStoreLocation);

        kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        KeyStore identityStore = KeyStore.getInstance(KeyStore.getDefaultType());
        identityStore.load(new FileInputStream(idStoreFile), idStorePasswd);
        kmf.init(identityStore, idStorePasswd);

      }
    }

    return kmf;
  }

  private TrustManagerFactory prepareTrustManagerFactory()
      throws NoSuchAlgorithmException, KeyStoreException, IOException, CertificateException, UnrecoverableKeyException,
      URISyntaxException {

    TrustManagerFactory tmf = null;

    if (trustStoreLocation != null) {
      File trustStoreFile = new File(trustStoreLocation);
      if (trustStoreFile.exists()) {
        char[] trustStorePasswd = locateKeyStorePassword(JVM_TRUST_PASSWD_PROP, trustStoreLocation);

        tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        trustStore.load(new FileInputStream(trustStoreFile), trustStorePasswd);
        tmf.init(trustStore);

      }
    }

    return tmf;
  }

  private char[] locateKeyStorePassword(String jvmPasswordProp,
                                        String keyStoreLocation) throws URISyntaxException {
    char[] passwd;
    String jvmStorePasswd = System.getProperty(jvmPasswordProp);

    if (Utils.trimToNull(jvmStorePasswd) == null) {
      URI idStoreURI = new URI(FILE_URI_PREFIX + keyStoreLocation);

      KeyName kname = new URIKeyName(idStoreURI);
      byte[] secret = keyChainAccessor.retrieveSecret(kname);

      if(secret == null) {
        throw new RuntimeException(String
            .format("Failure creating SSLContext because the store password could not be located for %s.",
                keyStoreLocation));
      }

      passwd = SecretUtils.toCharsAndWipe(secret);
    } else {
      passwd = jvmStorePasswd.toCharArray();
    }

    return passwd;
  }

}
